<?php
/**
 * WebPage Schema Class for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_WebPage extends BW_Schema_Base {
	
	/**
	 * Generate webpage schema
	 */
	public function generate( $post, $type = 'WebPage' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema = array_merge( $schema, $this->get_common_properties( $post ) );
		
		// Add breadcrumb
		$schema['breadcrumb'] = $this->get_breadcrumb_list( $post );
		
		// Add main entity of page for specific page types
		switch ( $type ) {
			case 'AboutPage':
				$schema['mainEntity'] = $this->get_about_page_entity( $post );
				$schema['lastReviewed'] = get_the_modified_date( 'c', $post );
				break;
				
			case 'ContactPage':
				$schema['mainEntity'] = $this->get_organization_properties();
				break;
				
			case 'CollectionPage':
				$schema['mainEntity'] = array(
					'@type' => 'ItemList',
					'numberOfItems' => $this->get_collection_count( $post ),
				);
				break;
				
			case 'ProfilePage':
				$schema['mainEntity'] = $this->get_profile_entity( $post );
				break;
				
			case 'QAPage':
				$schema['mainEntity'] = $this->get_qa_entity( $post );
				break;
		}
		
		// Add publisher
		$schema['publisher'] = $this->get_organization_properties();
		
		// Add copyright information
		$schema['copyrightYear'] = date( 'Y' );
		$schema['copyrightHolder'] = $this->get_organization_properties();
		
		// Removed speakable property - unnecessary complexity for most sites
		
		// Add isPartOf
		$schema['isPartOf'] = array(
			'@type' => 'WebSite',
			'name' => get_bloginfo( 'name' ),
			'url' => home_url( '/' ),
		);
		
		// Add AI-optimized properties
		$schema = $this->add_ai_properties( $schema, $post );
		
		return apply_filters( 'bw_schema_webpage', $schema, $post, $type );
	}
	
	/**
	 * Get about page entity
	 */
	private function get_about_page_entity( $post ) {
		$org = $this->get_organization_properties();
		
		// Add founding date if available
		$founding_date = get_option( 'bw_schema_founding_date' );
		if ( ! empty( $founding_date ) ) {
			$org['foundingDate'] = $founding_date;
		}
		
		// Add founders/leadership
		$founders = get_option( 'bw_schema_founders', array() );
		if ( ! empty( $founders ) && is_array( $founders ) ) {
			$founder_list = array();
			foreach ( $founders as $founder ) {
				$person = array(
					'@type' => 'Person',
					'name' => $founder['name'],
				);
				
				if ( ! empty( $founder['jobTitle'] ) ) {
					$person['jobTitle'] = $founder['jobTitle'];
				}
				
				if ( ! empty( $founder['url'] ) ) {
					$person['url'] = $founder['url'];
				}
				
				if ( ! empty( $founder['sameAs'] ) && is_array( $founder['sameAs'] ) ) {
					$person['sameAs'] = array_filter( $founder['sameAs'] );
				}
				
				$founder_list[] = $person;
			}
			$org['founder'] = count( $founder_list ) === 1 ? $founder_list[0] : $founder_list;
		}
		
		// Add number of employees
		$employee_count = get_option( 'bw_schema_employee_count' );
		if ( ! empty( $employee_count ) ) {
			$org['numberOfEmployees'] = array(
				'@type' => 'QuantitativeValue',
				'value' => intval( $employee_count ),
			);
		}
		
		// Add awards
		$awards = get_option( 'bw_schema_awards', array() );
		if ( ! empty( $awards ) && is_array( $awards ) ) {
			$org['award'] = $awards;
		}
		
		// Add certifications
		$certifications = get_option( 'bw_schema_certifications', array() );
		if ( ! empty( $certifications ) && is_array( $certifications ) ) {
			$cert_list = array();
			foreach ( $certifications as $cert ) {
				$cert_list[] = array(
					'@type' => 'Certification',
					'name' => $cert,
				);
			}
			$org['hasCertification'] = $cert_list;
		}
		
		return $org;
	}
	
	/**
	 * Get collection count
	 */
	private function get_collection_count( $post ) {
		// For category/tag archives
		if ( is_category() || is_tag() ) {
			$term = get_queried_object();
			return $term->count;
		}
		
		// For custom collections
		$count = get_post_meta( $post->ID, '_bw_schema_collection_count', true );
		return ! empty( $count ) ? intval( $count ) : 0;
	}
	
	/**
	 * Get profile entity
	 */
	private function get_profile_entity( $post ) {
		// Check if this is a team member page
		$person_id = get_post_meta( $post->ID, '_bw_schema_person_id', true );
		
		if ( ! empty( $person_id ) ) {
			// Get person data
			$person = array(
				'@type' => 'Person',
			);
			
			$person_name = get_post_meta( $post->ID, '_bw_schema_person_name', true );
			if ( ! empty( $person_name ) ) {
				$person['name'] = $person_name;
			} else {
				$person['name'] = get_the_title( $post );
			}
			
			$job_title = get_post_meta( $post->ID, '_bw_schema_job_title', true );
			if ( ! empty( $job_title ) ) {
				$person['jobTitle'] = $job_title;
			}
			
			$person_description = get_post_meta( $post->ID, '_bw_schema_person_description', true );
			if ( ! empty( $person_description ) ) {
				$person['description'] = $person_description;
			}
			
			$person_image = get_post_meta( $post->ID, '_bw_schema_person_image', true );
			if ( ! empty( $person_image ) ) {
				$person['image'] = $person_image;
			} elseif ( has_post_thumbnail( $post ) ) {
				$person['image'] = get_the_post_thumbnail_url( $post, 'full' );
			}
			
			// Education
			$education = get_post_meta( $post->ID, '_bw_schema_education', true );
			if ( ! empty( $education ) && is_array( $education ) ) {
				$edu_list = array();
				foreach ( $education as $edu ) {
					$edu_item = array(
						'@type' => 'EducationalOrganization',
						'name' => $edu['institution'],
					);
					
					if ( ! empty( $edu['degree'] ) ) {
						$edu_item['degree'] = $edu['degree'];
					}
					
					if ( ! empty( $edu['year'] ) ) {
						$edu_item['endDate'] = $edu['year'];
					}
					
					$edu_list[] = array(
						'@type' => 'EducationalOccupationalCredential',
						'credentialCategory' => $edu['degree'] ?? 'Degree',
						'educationalLevel' => $edu['level'] ?? 'University',
						'provider' => $edu_item,
					);
				}
				$person['hasCredential'] = $edu_list;
			}
			
			// Experience/Expertise
			$expertise = get_post_meta( $post->ID, '_bw_schema_expertise', true );
			if ( ! empty( $expertise ) && is_array( $expertise ) ) {
				$person['knowsAbout'] = $expertise;
			}
			
			// Social profiles
			$social = get_post_meta( $post->ID, '_bw_schema_social_profiles', true );
			if ( ! empty( $social ) && is_array( $social ) ) {
				$person['sameAs'] = array_filter( $social );
			}
			
			// Work for organization
			$person['worksFor'] = $this->get_organization_properties();
			
			return $person;
		}
		
		// Default to organization
		return $this->get_organization_properties();
	}
	
	/**
	 * Get Q&A entity
	 */
	private function get_qa_entity( $post ) {
		$question = get_post_meta( $post->ID, '_bw_schema_question', true );
		$answer = get_post_meta( $post->ID, '_bw_schema_answer', true );
		
		if ( empty( $question ) || empty( $answer ) ) {
			// Try to extract from content
			$content = $post->post_content;
			
			// Look for question in title or first heading
			if ( empty( $question ) ) {
				$question = get_the_title( $post );
			}
			
			// Use content as answer
			if ( empty( $answer ) ) {
				$answer = wp_strip_all_tags( $content );
			}
		}
		
		return array(
			'@type' => 'Question',
			'name' => $question,
			'acceptedAnswer' => array(
				'@type' => 'Answer',
				'text' => $answer,
				'author' => $this->get_author_properties( $post->post_author ),
				'dateCreated' => get_the_date( 'c', $post ),
				'upvoteCount' => get_post_meta( $post->ID, '_bw_schema_upvote_count', true ) ?: 0,
			),
		);
	}
}