<?php
/**
 * Schema Templates for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Templates {
	
	/**
	 * Get predefined schema templates
	 */
	public static function get_templates() {
		return array(
			'hotel' => array(
				'name' => __( 'Hotel', 'bw-ai-schema-pro' ),
				'description' => __( 'Complete hotel schema with amenities and booking info', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'Hotel',
					'name' => '',
					'description' => '',
					'image' => '',
					'url' => '',
					'telephone' => '',
					'email' => '',
					'address' => array(
						'@type' => 'PostalAddress',
						'streetAddress' => '',
						'addressLocality' => '',
						'addressRegion' => '',
						'postalCode' => '',
						'addressCountry' => '',
					),
					'geo' => array(
						'@type' => 'GeoCoordinates',
						'latitude' => '',
						'longitude' => '',
					),
					'amenityFeature' => array(),
					'priceRange' => '$$',
					'starRating' => array(
						'@type' => 'Rating',
						'ratingValue' => '',
					),
					'checkinTime' => '15:00',
					'checkoutTime' => '11:00',
				),
			),
			'resort' => array(
				'name' => __( 'Resort', 'bw-ai-schema-pro' ),
				'description' => __( 'Resort schema with activities and facilities', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'Resort',
					'name' => '',
					'description' => '',
					'image' => '',
					'url' => '',
					'telephone' => '',
					'email' => '',
					'address' => array(
						'@type' => 'PostalAddress',
						'streetAddress' => '',
						'addressLocality' => '',
						'addressRegion' => '',
						'postalCode' => '',
						'addressCountry' => '',
					),
					'amenityFeature' => array(),
					'availableLanguage' => array(),
					'currenciesAccepted' => 'USD',
					'paymentAccepted' => 'Credit Card, Cash',
					'priceRange' => '$$$',
				),
			),
			'law_firm' => array(
				'name' => __( 'Law Firm / Attorney', 'bw-ai-schema-pro' ),
				'description' => __( 'Legal practice schema with areas of expertise', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'Attorney',
					'name' => '',
					'description' => '',
					'image' => '',
					'url' => '',
					'telephone' => '',
					'email' => '',
					'address' => array(
						'@type' => 'PostalAddress',
						'streetAddress' => '',
						'addressLocality' => '',
						'addressRegion' => '',
						'postalCode' => '',
						'addressCountry' => '',
					),
					'areaServed' => array(),
					'knowsAbout' => array(
						'Criminal Law',
						'Family Law',
						'Personal Injury',
						'Business Law',
					),
					'alumni0f' => array(),
					'award' => array(),
					'memberOf' => array(
						array(
							'@type' => 'Organization',
							'name' => 'State Bar Association',
						),
					),
				),
			),
			'expert_author' => array(
				'name' => __( 'Expert Author', 'bw-ai-schema-pro' ),
				'description' => __( 'Enhanced author schema for subject matter experts', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'Person',
					'name' => '',
					'description' => '',
					'image' => '',
					'url' => '',
					'email' => '',
					'jobTitle' => '',
					'worksFor' => array(
						'@type' => 'Organization',
						'name' => '',
					),
					'alumniOf' => array(
						array(
							'@type' => 'EducationalOrganization',
							'name' => '',
						),
					),
					'knowsAbout' => array(),
					'hasCredential' => array(
						array(
							'@type' => 'EducationalOccupationalCredential',
							'name' => '',
							'credentialCategory' => '',
						),
					),
					'award' => array(),
					'sameAs' => array(),
					'publishingPrinciples' => '',
				),
			),
			'medical_practice' => array(
				'name' => __( 'Medical Practice', 'bw-ai-schema-pro' ),
				'description' => __( 'Healthcare provider schema', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'MedicalBusiness',
					'name' => '',
					'description' => '',
					'image' => '',
					'url' => '',
					'telephone' => '',
					'address' => array(
						'@type' => 'PostalAddress',
						'streetAddress' => '',
						'addressLocality' => '',
						'addressRegion' => '',
						'postalCode' => '',
						'addressCountry' => '',
					),
					'medicalSpecialty' => array(),
					'availableService' => array(
						array(
							'@type' => 'MedicalProcedure',
							'name' => '',
						),
					),
					'openingHoursSpecification' => array(),
				),
			),
			'educational_course' => array(
				'name' => __( 'Educational Course', 'bw-ai-schema-pro' ),
				'description' => __( 'Online or offline course schema', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'Course',
					'name' => '',
					'description' => '',
					'provider' => array(
						'@type' => 'Organization',
						'name' => '',
						'sameAs' => '',
					),
					'offers' => array(
						'@type' => 'Offer',
						'price' => '',
						'priceCurrency' => 'USD',
						'availability' => 'https://schema.org/InStock',
						'validFrom' => '',
					),
					'courseMode' => 'online',
					'duration' => 'PT30H',
					'educationalLevel' => 'beginner',
					'teaches' => array(),
					'coursePrerequisites' => array(),
				),
			),
			'software_app' => array(
				'name' => __( 'Software Application', 'bw-ai-schema-pro' ),
				'description' => __( 'Software or mobile app schema', 'bw-ai-schema-pro' ),
				'schema' => array(
					'@context' => 'https://schema.org',
					'@type' => 'SoftwareApplication',
					'name' => '',
					'description' => '',
					'applicationCategory' => '',
					'operatingSystem' => '',
					'offers' => array(
						'@type' => 'Offer',
						'price' => '0',
						'priceCurrency' => 'USD',
					),
					'aggregateRating' => array(
						'@type' => 'AggregateRating',
						'ratingValue' => '',
						'ratingCount' => '',
					),
					'screenshot' => '',
					'featureList' => array(),
					'softwareVersion' => '',
					'releaseNotes' => '',
				),
			),
		);
	}
	
	/**
	 * Get template by key
	 */
	public static function get_template( $key ) {
		$templates = self::get_templates();
		return isset( $templates[$key] ) ? $templates[$key] : null;
	}
	
	/**
	 * Apply template to post
	 */
	public static function apply_template( $post_id, $template_key, $data = array() ) {
		$template = self::get_template( $template_key );
		
		if ( ! $template ) {
			return false;
		}
		
		$schema = $template['schema'];
		
		// Merge with provided data
		if ( ! empty( $data ) ) {
			$schema = self::merge_schema_data( $schema, $data );
		}
		
		// Save as custom schema
		update_post_meta( $post_id, '_bw_schema_custom', wp_json_encode( $schema ) );
		update_post_meta( $post_id, '_bw_schema_template', $template_key );
		
		return true;
	}
	
	/**
	 * Merge schema data recursively
	 */
	private static function merge_schema_data( $schema, $data ) {
		foreach ( $data as $key => $value ) {
			if ( isset( $schema[$key] ) ) {
				if ( is_array( $value ) && is_array( $schema[$key] ) ) {
					$schema[$key] = self::merge_schema_data( $schema[$key], $value );
				} else {
					$schema[$key] = $value;
				}
			} else {
				$schema[$key] = $value;
			}
		}
		
		return $schema;
	}
	
	/**
	 * Get industry-specific recommendations
	 */
	public static function get_industry_recommendations( $industry ) {
		$recommendations = array(
			'hospitality' => array(
				'schema_types' => array( 'Hotel', 'Resort', 'Restaurant' ),
				'ai_properties' => array(
					'amenityFeature' => __( 'List all amenities for better AI understanding', 'bw-ai-schema-pro' ),
					'aggregateRating' => __( 'Include ratings to build trust', 'bw-ai-schema-pro' ),
					'priceRange' => __( 'Help AI understand your market position', 'bw-ai-schema-pro' ),
				),
			),
			'legal' => array(
				'schema_types' => array( 'Attorney', 'LegalService' ),
				'ai_properties' => array(
					'knowsAbout' => __( 'List practice areas for expertise recognition', 'bw-ai-schema-pro' ),
					'alumniOf' => __( 'Educational credentials build authority', 'bw-ai-schema-pro' ),
					'award' => __( 'Professional recognition enhances credibility', 'bw-ai-schema-pro' ),
				),
			),
			'healthcare' => array(
				'schema_types' => array( 'MedicalBusiness', 'Physician', 'Hospital' ),
				'ai_properties' => array(
					'medicalSpecialty' => __( 'Specify medical specialties', 'bw-ai-schema-pro' ),
					'availableService' => __( 'List services for patient matching', 'bw-ai-schema-pro' ),
					'accreditation' => __( 'Include certifications and accreditations', 'bw-ai-schema-pro' ),
				),
			),
			'education' => array(
				'schema_types' => array( 'Course', 'EducationalOrganization', 'School' ),
				'ai_properties' => array(
					'teaches' => __( 'What skills or knowledge is taught', 'bw-ai-schema-pro' ),
					'educationalLevel' => __( 'Target audience level', 'bw-ai-schema-pro' ),
					'coursePrerequisites' => __( 'Required knowledge or skills', 'bw-ai-schema-pro' ),
				),
			),
			'technology' => array(
				'schema_types' => array( 'SoftwareApplication', 'TechArticle', 'Organization' ),
				'ai_properties' => array(
					'applicationCategory' => __( 'Software category for discovery', 'bw-ai-schema-pro' ),
					'featureList' => __( 'Key features for comparison', 'bw-ai-schema-pro' ),
					'softwareRequirements' => __( 'System requirements', 'bw-ai-schema-pro' ),
				),
			),
		);
		
		return isset( $recommendations[$industry] ) ? $recommendations[$industry] : array();
	}
}