<?php
/**
 * Team Member Schema Enhancement for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Team_Member {
	
	/**
	 * Initialize team member enhancements
	 */
	public static function init() {
		// Hook into schema generation
		add_filter( 'bw_schema_rendered_schemas', array( __CLASS__, 'filter_team_member_schemas' ), 20 );
		
		// Add custom meta box for team members
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_team_member_meta_box' ), 15 );
		
		// Save team member schema data
		add_action( 'save_post_sw_teammember', array( __CLASS__, 'save_team_member_schema' ), 10, 2 );
		
		// Disable other plugin schemas on team member pages
		add_action( 'wp', array( __CLASS__, 'disable_conflicting_schemas' ), 5 );
	}
	
	/**
	 * Filter schemas for team member pages
	 */
	public static function filter_team_member_schemas( $schemas ) {
		if ( ! is_singular( 'sw_teammember' ) ) {
			return $schemas;
		}
		
		// Remove any Blog or WebSite schemas
		$filtered_schemas = array();
		foreach ( $schemas as $schema ) {
			if ( isset( $schema['@type'] ) ) {
				$type = $schema['@type'];
				// Skip Blog, WebSite, and CollectionPage schemas
				if ( in_array( $type, array( 'Blog', 'WebSite', 'CollectionPage' ) ) ) {
					continue;
				}
			}
			$filtered_schemas[] = $schema;
		}
		
		return $filtered_schemas;
	}
	
	/**
	 * Add meta box for team member schema fields
	 */
	public static function add_team_member_meta_box() {
		add_meta_box(
			'bw_schema_team_member_fields',
			__( 'Additional Schema Information', 'bw-ai-schema-pro' ),
			array( __CLASS__, 'render_meta_box' ),
			'sw_teammember',
			'normal',
			'default'
		);
	}
	
	/**
	 * Render team member meta box
	 */
	public static function render_meta_box( $post ) {
		wp_nonce_field( 'bw_schema_team_member', 'bw_schema_team_member_nonce' );
		
		// Get existing values
		$expertise = get_post_meta( $post->ID, '_bw_schema_expertise', true );
		$credentials = get_post_meta( $post->ID, '_bw_schema_credentials', true );
		$linkedin = get_post_meta( $post->ID, '_bw_schema_linkedin', true );
		$twitter = get_post_meta( $post->ID, '_bw_schema_twitter', true );
		?>
		<div class="bw-schema-team-member-fields">
			<p>
				<label for="bw_schema_expertise"><?php _e( 'Areas of Expertise', 'bw-ai-schema-pro' ); ?></label><br>
				<input type="text" id="bw_schema_expertise" name="bw_schema_expertise" value="<?php echo esc_attr( $expertise ); ?>" class="widefat" />
				<span class="description"><?php _e( 'Comma-separated list of expertise areas', 'bw-ai-schema-pro' ); ?></span>
			</p>
			
			<p>
				<label for="bw_schema_credentials"><?php _e( 'Credentials', 'bw-ai-schema-pro' ); ?></label><br>
				<input type="text" id="bw_schema_credentials" name="bw_schema_credentials" value="<?php echo esc_attr( $credentials ); ?>" class="widefat" />
				<span class="description"><?php _e( 'Professional credentials or certifications', 'bw-ai-schema-pro' ); ?></span>
			</p>
			
			<p>
				<label for="bw_schema_linkedin"><?php _e( 'LinkedIn Profile URL', 'bw-ai-schema-pro' ); ?></label><br>
				<input type="url" id="bw_schema_linkedin" name="bw_schema_linkedin" value="<?php echo esc_url( $linkedin ); ?>" class="widefat" />
			</p>
			
			<p>
				<label for="bw_schema_twitter"><?php _e( 'Twitter/X Profile URL', 'bw-ai-schema-pro' ); ?></label><br>
				<input type="url" id="bw_schema_twitter" name="bw_schema_twitter" value="<?php echo esc_url( $twitter ); ?>" class="widefat" />
			</p>
		</div>
		<?php
	}
	
	/**
	 * Save team member schema data
	 */
	public static function save_team_member_schema( $post_id, $post ) {
		// Verify nonce
		if ( ! isset( $_POST['bw_schema_team_member_nonce'] ) || 
		     ! wp_verify_nonce( $_POST['bw_schema_team_member_nonce'], 'bw_schema_team_member' ) ) {
			return;
		}
		
		// Check permissions
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}
		
		// Save expertise
		if ( isset( $_POST['bw_schema_expertise'] ) ) {
			$expertise = sanitize_text_field( $_POST['bw_schema_expertise'] );
			if ( ! empty( $expertise ) ) {
				$expertise_array = array_map( 'trim', explode( ',', $expertise ) );
				update_post_meta( $post_id, '_bw_schema_expertise', array_filter( $expertise_array ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_expertise' );
			}
		}
		
		// Save credentials
		if ( isset( $_POST['bw_schema_credentials'] ) ) {
			update_post_meta( $post_id, '_bw_schema_credentials', sanitize_text_field( $_POST['bw_schema_credentials'] ) );
		}
		
		// Save social profiles
		if ( isset( $_POST['bw_schema_linkedin'] ) ) {
			update_post_meta( $post_id, '_bw_schema_linkedin', esc_url_raw( $_POST['bw_schema_linkedin'] ) );
		}
		
		if ( isset( $_POST['bw_schema_twitter'] ) ) {
			update_post_meta( $post_id, '_bw_schema_twitter', esc_url_raw( $_POST['bw_schema_twitter'] ) );
		}
		
		// Build social profiles array
		$social_profiles = array();
		if ( ! empty( $_POST['bw_schema_linkedin'] ) ) {
			$social_profiles[] = esc_url_raw( $_POST['bw_schema_linkedin'] );
		}
		if ( ! empty( $_POST['bw_schema_twitter'] ) ) {
			$social_profiles[] = esc_url_raw( $_POST['bw_schema_twitter'] );
		}
		
		if ( ! empty( $social_profiles ) ) {
			update_post_meta( $post_id, '_bw_schema_social_profiles', $social_profiles );
		} else {
			delete_post_meta( $post_id, '_bw_schema_social_profiles' );
		}
	}
	
	/**
	 * Disable conflicting schemas on team member pages
	 */
	public static function disable_conflicting_schemas() {
		if ( ! is_singular( 'sw_teammember' ) ) {
			return;
		}
		
		// Disable Yoast
		add_filter( 'wpseo_json_ld_output', '__return_false', 999 );
		add_filter( 'wpseo_schema_graph', '__return_empty_array', 999 );
		add_filter( 'wpseo_schema_webpage', '__return_empty_array', 999 );
		
		// Disable Rank Math
		add_filter( 'rank_math/json_ld', '__return_empty_array', 999 );
		
		// Disable All in One SEO
		add_filter( 'aioseo_schema_output', '__return_false', 999 );
		
		// Disable SEOPress
		add_filter( 'seopress_schemas_jsonld_output', '__return_false', 999 );
		
		// Remove generic schema actions
		remove_all_actions( 'wp_head', 60 );
		
		// Add a high priority action to remove late-added schemas
		add_action( 'wp_head', array( __CLASS__, 'remove_remaining_schemas' ), 0 );
	}
	
	/**
	 * Remove any remaining schema outputs
	 */
	public static function remove_remaining_schemas() {
		// Remove any schema-related actions that might have been added late
		global $wp_filter;
		
		if ( isset( $wp_filter['wp_head'] ) ) {
			foreach ( $wp_filter['wp_head'] as $priority => $hooks ) {
				foreach ( $hooks as $hook_key => $hook ) {
					// Check if this might be a schema-related hook
					if ( strpos( $hook_key, 'schema' ) !== false || 
					     strpos( $hook_key, 'structured_data' ) !== false ||
					     strpos( $hook_key, 'json_ld' ) !== false ) {
						remove_action( 'wp_head', $hook_key, $priority );
					}
				}
			}
		}
	}
}