<?php
/**
 * Developer Hooks for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Hooks {
	
	/**
	 * Register all hooks
	 */
	public static function init() {
		// Allow developers to modify schema types
		add_filter( 'bw_schema_types', array( __CLASS__, 'filter_schema_types' ), 10, 1 );
		
		// Allow developers to modify AI properties
		add_filter( 'bw_schema_ai_properties', array( __CLASS__, 'filter_ai_properties' ), 10, 1 );
		
		// Allow developers to add custom schema classes
		add_action( 'bw_schema_init', array( __CLASS__, 'load_custom_schemas' ), 10 );
		
		// Allow developers to modify rendered schemas
		add_filter( 'bw_schema_rendered_schemas', array( __CLASS__, 'filter_rendered_schemas' ), 10, 1 );
		
		// Allow modification of individual schema types
		self::register_schema_filters();
	}
	
	/**
	 * Filter schema types
	 */
	public static function filter_schema_types( $types ) {
		// Allow developers to add custom schema types
		return apply_filters( 'bw_schema_custom_types', $types );
	}
	
	/**
	 * Filter AI properties
	 */
	public static function filter_ai_properties( $properties ) {
		// Allow developers to add custom AI properties
		return apply_filters( 'bw_schema_custom_ai_properties', $properties );
	}
	
	/**
	 * Load custom schemas
	 */
	public static function load_custom_schemas() {
		// Allow developers to load their own schema classes
		do_action( 'bw_schema_load_custom_classes' );
	}
	
	/**
	 * Filter rendered schemas
	 */
	public static function filter_rendered_schemas( $schemas ) {
		// Allow final modification of schemas before output
		return apply_filters( 'bw_schema_final_output', $schemas );
	}
	
	/**
	 * Register filters for each schema type
	 */
	private static function register_schema_filters() {
		$schema_types = array(
			'article',
			'person',
			'organization',
			'localbusiness',
			'product',
			'event',
			'howto',
			'faq',
			'review',
			'recipe',
			'video',
			'course',
			'jobposting'
		);
		
		foreach ( $schema_types as $type ) {
			// Allow modification of each schema type
			add_filter( "bw_schema_{$type}", array( __CLASS__, 'apply_schema_modifications' ), 10, 3 );
			
			// Allow adding fields to each schema type
			add_filter( "bw_schema_{$type}_fields", array( __CLASS__, 'add_custom_fields' ), 10, 2 );
			
			// Allow validation of each schema type
			add_filter( "bw_schema_{$type}_validate", array( __CLASS__, 'validate_schema' ), 10, 2 );
		}
	}
	
	/**
	 * Apply schema modifications
	 */
	public static function apply_schema_modifications( $schema, $post, $type ) {
		// Allow developers to modify schema data
		$schema = apply_filters( 'bw_schema_modify_data', $schema, $post, $type );
		
		// Type-specific modifications
		$schema = apply_filters( "bw_schema_modify_{$type}", $schema, $post );
		
		return $schema;
	}
	
	/**
	 * Add custom fields
	 */
	public static function add_custom_fields( $fields, $post ) {
		// Allow developers to add custom fields
		return apply_filters( 'bw_schema_add_fields', $fields, $post );
	}
	
	/**
	 * Validate schema
	 */
	public static function validate_schema( $is_valid, $schema ) {
		// Allow developers to add custom validation
		return apply_filters( 'bw_schema_custom_validation', $is_valid, $schema );
	}
	
	/**
	 * Helper function to add a custom schema type
	 */
	public static function add_schema_type( $key, $label, $description, $icon, $subtypes ) {
		add_filter( 'bw_schema_custom_types', function( $types ) use ( $key, $label, $description, $icon, $subtypes ) {
			$types[$key] = array(
				'label' => $label,
				'description' => $description,
				'icon' => $icon,
				'subtypes' => $subtypes
			);
			return $types;
		});
	}
	
	/**
	 * Helper function to add AI property
	 */
	public static function add_ai_property( $category, $key, $label ) {
		add_filter( 'bw_schema_custom_ai_properties', function( $properties ) use ( $category, $key, $label ) {
			if ( ! isset( $properties[$category] ) ) {
				$properties[$category] = array();
			}
			$properties[$category][$key] = $label;
			return $properties;
		});
	}
	
	/**
	 * Helper function to modify schema output
	 */
	public static function modify_schema( $type, $callback ) {
		add_filter( "bw_schema_modify_{$type}", $callback, 10, 2 );
	}
}

/**
 * Example usage for developers:
 *
 * // Add a custom schema type
 * BW_Schema_Hooks::add_schema_type( 'podcast', 'Podcast', 'For podcast episodes', 'dashicons-microphone', array(
 *     'PodcastEpisode' => 'Podcast Episode',
 *     'PodcastSeries' => 'Podcast Series'
 * ));
 *
 * // Add a custom AI property
 * BW_Schema_Hooks::add_ai_property( 'expertise', 'hasPublication', 'Publications' );
 *
 * // Modify article schema
 * BW_Schema_Hooks::modify_schema( 'article', function( $schema, $post ) {
 *     $schema['customProperty'] = 'Custom Value';
 *     return $schema;
 * });
 */