<?php
/**
 * BW Schema Author Box
 *
 * Handles the display of author boxes on the frontend
 *
 * @package BW_AI_Schema_Pro
 * @since 1.0.0
 */

class BW_Schema_Author_Box {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Hook into the content to add author box
        add_filter( 'the_content', array( $this, 'maybe_add_author_box' ), 20 );
        
        // Add shortcode support
        add_shortcode( 'bw_author_box', array( $this, 'render_author_box_shortcode' ) );
        
        // Enqueue frontend styles
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );
    }
    
    /**
     * Maybe add author box to post content
     */
    public function maybe_add_author_box( $content ) {
        // Only on single posts/pages
        if ( ! is_singular() ) {
            return $content;
        }
        
        global $post;
        
        // Check if author box is globally enabled
        $author_box_enabled = get_option( 'bw_schema_author_box_enabled', 'yes' );
        if ( $author_box_enabled !== 'yes' ) {
            return $content;
        }
        
        // Check if manual placement is selected
        $author_box_position = get_option( 'bw_schema_author_box_position', 'after_content' );
        if ( $author_box_position === 'manual' ) {
            return $content;
        }
        
        // Check if this post type should show author box by default
        $enabled_post_types = get_option( 'bw_schema_author_box_post_types', array( 'post' ) );
        $show_by_default = in_array( $post->post_type, $enabled_post_types );
        
        // Check if author box is enabled for this specific post
        $show_author_box = get_post_meta( $post->ID, '_bw_schema_show_author_box', true );
        
        // If not set for this post, use the default based on post type
        if ( $show_author_box === '' ) {
            $show_author_box = $show_by_default ? 'yes' : 'no';
        }
        
        if ( $show_author_box !== 'yes' ) {
            return $content;
        }
        
        // Get the author box HTML
        $author_box = $this->get_author_box_html( $post->ID );
        
        // Add based on position setting
        if ( $author_box_position === 'before_content' ) {
            return $author_box . $content;
        } else {
            return $content . $author_box;
        }
    }
    
    /**
     * Get author box HTML
     */
    public function get_author_box_html( $post_id ) {
        $authors = $this->get_post_authors( $post_id );
        
        if ( empty( $authors ) ) {
            return '';
        }
        
        $multiple_class = count( $authors ) > 1 ? ' has-multiple-authors' : '';
        $html = '<div class="bw-author-box-wrapper' . $multiple_class . '">';
        
        foreach ( $authors as $author ) {
            $html .= $this->render_single_author_box( $author );
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Get authors for a post
     */
    private function get_post_authors( $post_id ) {
        $authors = array();
        
        // First check for multiple authors (new system)
        $multiple_authors = get_post_meta( $post_id, '_bw_schema_multiple_authors', true );
        
        if ( ! empty( $multiple_authors ) && is_array( $multiple_authors ) ) {
            $all_custom_authors = get_option( 'bw_schema_custom_authors', array() );
            
            foreach ( $multiple_authors as $author ) {
                if ( empty( $author['type'] ) ) {
                    continue;
                }
                
                switch ( $author['type'] ) {
                    case 'wordpress':
                        if ( ! empty( $author['wordpress_user_id'] ) ) {
                            $author_data = get_userdata( $author['wordpress_user_id'] );
                            if ( $author_data ) {
                                $authors[] = array(
                                    'type' => 'wordpress',
                                    'id' => $author['wordpress_user_id'],
                                    'name' => $author_data->display_name,
                                    'image' => get_avatar_url( $author['wordpress_user_id'], array( 'size' => 150 ) ),
                                    'bio' => get_the_author_meta( 'description', $author['wordpress_user_id'] ),
                                    'url' => get_author_posts_url( $author['wordpress_user_id'] ),
                                    'job_title' => ''
                                );
                            }
                        }
                        break;
                        
                    case 'custom':
                        if ( ! empty( $author['custom_author_id'] ) ) {
                            // Find author by ID in the array
                            $author_data = null;
                            foreach ( $all_custom_authors as $custom_author ) {
                                if ( isset( $custom_author['id'] ) && $custom_author['id'] == $author['custom_author_id'] ) {
                                    $author_data = $custom_author;
                                    break;
                                }
                            }
                            
                            if ( $author_data ) {
                            // Determine author URL
                            $author_url = '';
                            if ( ! empty( $author_data['teamPageUrl'] ) ) {
                                $author_url = $author_data['teamPageUrl'];
                            } elseif ( ! empty( $author_data['teamPageId'] ) && $author_data['teamPageId'] !== 'custom' ) {
                                $author_url = get_permalink( $author_data['teamPageId'] );
                            } elseif ( ! empty( $author_data['social']['website'] ) ) {
                                $author_url = $author_data['social']['website'];
                            }
                            
                            $authors[] = array(
                                'type' => 'custom',
                                'id' => $author['custom_author_id'],
                                'name' => $author_data['name'],
                                'image' => $author_data['image'] ?? '',
                                'bio' => $author_data['description'] ?? '',
                                'url' => $author_url,
                                'job_title' => $author_data['jobTitle'] ?? ''
                            );
                            }
                        }
                        break;
                        
                    case 'external':
                        if ( ! empty( $author['external']['name'] ) ) {
                            $authors[] = array(
                                'type' => 'external',
                                'id' => uniqid(),
                                'name' => $author['external']['name'],
                                'image' => $author['external']['image'] ?? '',
                                'bio' => $author['external']['bio'] ?? '',
                                'url' => $author['external']['website'] ?? '',
                                'job_title' => $author['external']['job_title'] ?? ''
                            );
                        }
                        break;
                }
            }
        }
        
        // Fall back to single custom author (legacy support) or default author
        if ( empty( $authors ) ) {
            $use_custom_authors = get_option( 'bw_schema_use_custom_authors', 'no' );
            $custom_author_id = get_post_meta( $post_id, '_bw_schema_custom_author', true );
            
            if ( $use_custom_authors === 'yes' ) {
                $all_custom_authors = get_option( 'bw_schema_custom_authors', array() );
                
                // If no specific author is set, find the default author
                if ( ! $custom_author_id ) {
                    foreach ( $all_custom_authors as $author ) {
                        if ( ! empty( $author['isDefault'] ) && $author['isDefault'] === true ) {
                            $custom_author_id = $author['id'];
                            break;
                        }
                    }
                }
                
                // Find author by ID in the array
                $author_data = null;
                if ( $custom_author_id ) {
                    foreach ( $all_custom_authors as $author ) {
                        if ( isset( $author['id'] ) && $author['id'] == $custom_author_id ) {
                            $author_data = $author;
                            break;
                        }
                    }
                }
                
                if ( $author_data ) {
                    // Determine author URL
                    $author_url = '';
                    if ( ! empty( $author_data['teamPageUrl'] ) ) {
                        $author_url = $author_data['teamPageUrl'];
                    } elseif ( ! empty( $author_data['teamPageId'] ) && $author_data['teamPageId'] !== 'custom' ) {
                        $author_url = get_permalink( $author_data['teamPageId'] );
                    } elseif ( ! empty( $author_data['social']['website'] ) ) {
                        $author_url = $author_data['social']['website'];
                    }
                    
                    $authors[] = array(
                        'type' => 'custom',
                        'id' => $custom_author_id,
                        'name' => $author_data['name'],
                        'image' => $author_data['image'] ?? '',
                        'bio' => $author_data['description'] ?? '',
                        'url' => $author_url,
                        'job_title' => $author_data['jobTitle'] ?? ''
                    );
                }
            }
        }
        
        // Final fallback to WordPress post author
        if ( empty( $authors ) ) {
            $post = get_post( $post_id );
            $author_id = $post->post_author;
            $author_data = get_userdata( $author_id );
            
            if ( $author_data ) {
                $authors[] = array(
                    'type' => 'wordpress',
                    'id' => $author_id,
                    'name' => $author_data->display_name,
                    'image' => get_avatar_url( $author_id, array( 'size' => 150 ) ),
                    'bio' => get_the_author_meta( 'description', $author_id ),
                    'url' => get_author_posts_url( $author_id ),
                    'job_title' => ''
                );
            }
        }
        
        return $authors;
    }
    
    /**
     * Render single author box
     */
    private function render_single_author_box( $author ) {
        $html = '<div class="bw-author-box">';
        
        // Author image
        $html .= '<div class="bw-author-image">';
        if ( ! empty( $author['image'] ) ) {
            $html .= '<img src="' . esc_url( $author['image'] ) . '" alt="' . esc_attr( $author['name'] ) . '">';
        } else {
            // Default avatar
            $html .= '<div class="bw-author-avatar-placeholder">' . substr( $author['name'], 0, 1 ) . '</div>';
        }
        $html .= '</div>';
        
        // Debug info (remove after testing)
        if ( current_user_can( 'manage_options' ) ) {
            $debug_info = array(
                'type' => $author['type'] ?? 'unknown',
                'id' => $author['id'] ?? 'no-id',
                'name' => $author['name'] ?? 'no-name',
                'has_image' => ! empty( $author['image'] ),
                'image_url' => $author['image'] ?? 'no-image',
                'has_bio' => ! empty( $author['bio'] ),
                'bio_length' => strlen( $author['bio'] ?? '' ),
                'has_job_title' => ! empty( $author['job_title'] ),
                'job_title' => $author['job_title'] ?? 'no-title'
            );
            $html .= '<!-- Author Debug: ' . esc_html( json_encode( $debug_info ) ) . ' -->';
        }
        
        // Author info
        $html .= '<div class="bw-author-info">';
        
        // Name with optional link
        $html .= '<h4 class="bw-author-name">';
        if ( ! empty( $author['url'] ) ) {
            $html .= '<a href="' . esc_url( $author['url'] ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $author['name'] ) . '</a>';
        } else {
            $html .= esc_html( $author['name'] );
        }
        
        // Job title if available
        if ( ! empty( $author['job_title'] ) ) {
            $html .= '<span class="bw-author-title">' . esc_html( $author['job_title'] ) . '</span>';
        }
        $html .= '</h4>';
        
        // Bio/Description
        if ( ! empty( $author['bio'] ) ) {
            $html .= '<div class="bw-author-bio">' . wp_kses_post( $author['bio'] ) . '</div>';
        }
        
        $html .= '</div>'; // .bw-author-info
        $html .= '</div>'; // .bw-author-box
        
        return $html;
    }
    
    /**
     * Render author box shortcode
     */
    public function render_author_box_shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'post_id' => get_the_ID(),
            'author_ids' => '' // Comma-separated custom author IDs
        ), $atts );
        
        if ( ! empty( $atts['author_ids'] ) ) {
            // Override authors for this specific shortcode
            $author_ids = array_map( 'trim', explode( ',', $atts['author_ids'] ) );
            $authors = array();
            $all_custom_authors = get_option( 'bw_schema_custom_authors', array() );
            
            foreach ( $author_ids as $author_id ) {
                // Find author by ID in the array
                $author_data = null;
                foreach ( $all_custom_authors as $custom_author ) {
                    if ( isset( $custom_author['id'] ) && $custom_author['id'] == $author_id ) {
                        $author_data = $custom_author;
                        break;
                    }
                }
                
                if ( $author_data ) {
                    // Determine author URL
                    $author_url = '';
                    if ( ! empty( $author_data['teamPageUrl'] ) ) {
                        $author_url = $author_data['teamPageUrl'];
                    } elseif ( ! empty( $author_data['teamPageId'] ) && $author_data['teamPageId'] !== 'custom' ) {
                        $author_url = get_permalink( $author_data['teamPageId'] );
                    } elseif ( ! empty( $author_data['social']['website'] ) ) {
                        $author_url = $author_data['social']['website'];
                    }
                    
                    $authors[] = array(
                        'type' => 'custom',
                        'id' => $author_id,
                        'name' => $author_data['name'],
                        'image' => $author_data['image'] ?? '',
                        'bio' => $author_data['description'] ?? '',
                        'url' => $author_url,
                        'job_title' => $author_data['jobTitle'] ?? ''
                    );
                }
            }
            
            if ( ! empty( $authors ) ) {
                $html = '<div class="bw-author-box-wrapper bw-author-box-shortcode">';
                foreach ( $authors as $author ) {
                    $html .= $this->render_single_author_box( $author );
                }
                $html .= '</div>';
                return $html;
            }
        }
        
        return $this->get_author_box_html( $atts['post_id'] );
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        if ( is_singular() ) {
            wp_enqueue_style( 
                'bw-schema-author-box', 
                plugin_dir_url( dirname( __FILE__ ) ) . 'assets/css/author-box.css',
                array(),
                '1.0.0'
            );
        }
    }
}

// Initialize
new BW_Schema_Author_Box();