<?php
/**
 * Admin functionality for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Admin {
	
	/**
	 * Add admin menu pages
	 *
	 * All pages are added under Settings menu (options-general.php)
	 */
	public function add_menu_pages() {
		// Main page under Settings menu
		add_options_page(
			__( 'AI Schema Pro', 'bw-ai-schema-pro' ),
			__( 'AI Schema Pro', 'bw-ai-schema-pro' ),
			'manage_options',
			'bw-ai-schema',
			array( $this, 'render_dashboard_page' )
		);

		// Global Settings (hidden - accessed via tab or link from dashboard)
		add_submenu_page(
			null, // Hidden from menu
			__( 'AI Schema Pro - Settings', 'bw-ai-schema-pro' ),
			__( 'Settings', 'bw-ai-schema-pro' ),
			'manage_options',
			'bw-ai-schema-settings',
			array( $this, 'render_settings_page' )
		);

		// Author Profiles (hidden - accessed via tab or link from dashboard)
		add_submenu_page(
			null, // Hidden from menu
			__( 'AI Schema Pro - Author Profiles', 'bw-ai-schema-pro' ),
			__( 'Author Profiles', 'bw-ai-schema-pro' ),
			'manage_options',
			'bw-ai-schema-authors',
			array( $this, 'render_authors_page' )
		);

		// Setup Wizard (hidden from menu but accessible)
		add_submenu_page(
			null, // Hidden from menu
			__( 'AI Schema Pro - Setup Wizard', 'bw-ai-schema-pro' ),
			__( 'Setup Wizard', 'bw-ai-schema-pro' ),
			'manage_options',
			'bw-ai-schema-setup',
			array( $this, 'render_setup_wizard_page' )
		);

		// Test Runner (for development/debugging)
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			add_submenu_page(
				null, // Hidden from menu
				__( 'AI Schema Pro - Test Runner', 'bw-ai-schema-pro' ),
				__( 'Test Runner', 'bw-ai-schema-pro' ),
				'manage_options',
				'bw-ai-schema-tests',
				array( $this, 'render_test_runner_page' )
			);
		}
	}
	
	/**
	 * Render dashboard page
	 */
	public function render_dashboard_page() {
		// Handle form submission
		if ( isset( $_POST['bw_schema_save_dashboard'] ) && wp_verify_nonce( $_POST['bw_schema_dashboard_nonce'], 'bw_schema_dashboard' ) ) {
			$this->save_dashboard_settings();
		}
		
		include BW_SCHEMA_PLUGIN_DIR . 'admin/views/dashboard.php';
	}
	
	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		// Handle form submission
		if ( isset( $_POST['bw_schema_save_settings'] ) && wp_verify_nonce( $_POST['bw_schema_settings_nonce'], 'bw_schema_settings' ) ) {
			$this->save_settings();
		}
		
		include BW_SCHEMA_PLUGIN_DIR . 'admin/views/settings.php';
	}
	
	/**
	 * Render authors page
	 */
	public function render_authors_page() {
		// Handle form submissions
		if ( isset( $_POST['bw_schema_save_author'] ) && wp_verify_nonce( $_POST['bw_schema_author_nonce'], 'bw_schema_author' ) ) {
			$this->save_author();
		}
		
		if ( isset( $_POST['action'] ) && $_POST['action'] === 'bulk-delete' && wp_verify_nonce( $_POST['_wpnonce'], 'bulk-authors' ) ) {
			$this->bulk_delete_authors();
		}
		
		if ( isset( $_GET['action'] ) && $_GET['action'] === 'delete' && wp_verify_nonce( $_GET['_wpnonce'], 'delete_author' ) ) {
			$this->delete_author( $_GET['author'] );
		}
		
		include BW_SCHEMA_PLUGIN_DIR . 'admin/views/author-profiles.php';
	}
	
	/**
	 * Render setup wizard page
	 */
	public function render_setup_wizard_page() {
		// Handle form submission
		if ( isset( $_POST['bw_schema_save_setup'] ) && wp_verify_nonce( $_POST['bw_schema_setup_nonce'], 'bw_schema_setup' ) ) {
			$this->save_setup_wizard();
		}

		include BW_SCHEMA_PLUGIN_DIR . 'admin/views/setup-wizard.php';
	}

	/**
	 * Render test runner page
	 */
	public function render_test_runner_page() {
		include BW_SCHEMA_PLUGIN_DIR . 'admin/views/test-runner.php';
	}
	
	/**
	 * Render post meta box
	 */
	public function render_post_meta_box( $post ) {
		wp_nonce_field( 'bw_schema_meta', 'bw_schema_meta_nonce' );
		
		// Get current values
		$schema_type = get_post_meta( $post->ID, '_bw_schema_type', true );
		$custom_schema = get_post_meta( $post->ID, '_bw_schema_custom', true );
		$disable_schema = get_post_meta( $post->ID, '_bw_schema_disable', true );
		$simple_mode = get_post_meta( $post->ID, '_bw_schema_simple_mode', true );
		if ( empty( $simple_mode ) ) $simple_mode = 'yes'; // Default to simple mode
		
		// Get schema types
		$schema_types = BW_Schema_Core::get_schema_types();
		
		// Get post type default
		$defaults = BW_Schema_Core::get_post_type_defaults();
		$default_type = isset( $defaults[$post->post_type] ) ? $defaults[$post->post_type] : '';

		// Auto-detect suggested schema type
		$suggested_type = $this->auto_detect_schema_type( $post );

		// Get meta box default state setting
		$default_state = get_option( 'bw_schema_metabox_default_state', 'collapsed' );
		$is_expanded = ( $default_state === 'expanded' );
		?>
		<div class="bw-schema-meta-box bw-schema-collapsible-box">
			<!-- Compact Header with Toggle -->
			<div class="bw-schema-box-header">
				<div class="bw-schema-box-summary">
					<span class="bw-schema-current-type">
						<?php
						if ( $schema_type ) {
							$type_parts = explode( ':', $schema_type );
							echo esc_html( isset( $type_parts[1] ) ? $type_parts[1] : $schema_type );
						} else {
							_e( 'Auto-detect', 'bw-ai-schema-pro' );
						}
						?>
					</span>
					<span class="bw-schema-status <?php echo $disable_schema === 'yes' ? 'disabled' : 'enabled'; ?>">
						<?php echo $disable_schema === 'yes' ? __( 'Disabled', 'bw-ai-schema-pro' ) : __( 'Active', 'bw-ai-schema-pro' ); ?>
					</span>
				</div>
				<button type="button" class="bw-schema-toggle-btn" aria-expanded="<?php echo $is_expanded ? 'true' : 'false'; ?>">
					<span class="dashicons dashicons-arrow-down-alt2"></span>
					<span class="screen-reader-text"><?php _e( 'Toggle Schema Settings', 'bw-ai-schema-pro' ); ?></span>
				</button>
			</div>

			<!-- Collapsible Content -->
			<div class="bw-schema-box-content" style="<?php echo $is_expanded ? '' : 'display: none;'; ?>">

			<!-- Mode Switcher -->
			<div class="bw-schema-mode-switcher" style="background: #f0f6fc; padding: 10px; border-radius: 5px; margin-bottom: 20px;">
				<label style="margin-right: 20px;">
					<input type="radio" name="bw_schema_simple_mode" value="yes" <?php checked( $simple_mode, 'yes' ); ?> />
					<strong><?php _e( 'Simple Mode', 'bw-ai-schema-pro' ); ?></strong>
					<span style="color: #666; font-size: 12px;"><?php _e( '(Recommended)', 'bw-ai-schema-pro' ); ?></span>
				</label>
				<label>
					<input type="radio" name="bw_schema_simple_mode" value="no" <?php checked( $simple_mode, 'no' ); ?> />
					<strong><?php _e( 'Advanced Mode', 'bw-ai-schema-pro' ); ?></strong>
					<span style="color: #666; font-size: 12px;"><?php _e( '(JSON Editor)', 'bw-ai-schema-pro' ); ?></span>
				</label>
			</div>
			
			<?php if ( $suggested_type && $suggested_type !== $schema_type ) : ?>
			<div class="bw-schema-suggestion" style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; border-radius: 5px; margin-bottom: 15px;">
				<strong><?php _e( '💡 Suggestion:', 'bw-ai-schema-pro' ); ?></strong>
				<?php printf( __( 'Based on your content, we recommend using "%s" schema type.', 'bw-ai-schema-pro' ), $suggested_type ); ?>
				<button type="button" class="button button-small" id="bw-schema-use-suggestion" data-type="<?php echo esc_attr( $suggested_type ); ?>" style="margin-left: 10px;">
					<?php _e( 'Use This', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>
			<?php endif; ?>
			
			<!-- Simple Mode -->
			<div id="bw-schema-simple-mode" style="<?php echo $simple_mode === 'yes' ? '' : 'display:none;'; ?>">
				<p>
					<label for="bw_schema_type">
						<strong><?php _e( 'What type of content is this?', 'bw-ai-schema-pro' ); ?></strong>
					</label>
					<select name="bw_schema_type" id="bw_schema_type" class="widefat">
						<option value=""><?php _e( '-- Auto-detect (Recommended) --', 'bw-ai-schema-pro' ); ?></option>
						<?php foreach ( $schema_types as $type_key => $type_data ) : ?>
							<optgroup label="<?php echo esc_attr( $type_data['label'] ); ?>">
								<?php foreach ( $type_data['subtypes'] as $subtype_key => $subtype_label ) : ?>
									<?php $value = $type_key . ':' . $subtype_key; ?>
									<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $schema_type, $value ); ?>>
										<?php echo esc_html( $subtype_label ); ?>
									</option>
								<?php endforeach; ?>
							</optgroup>
						<?php endforeach; ?>
					</select>
				</p>
				
				<!-- Dynamic Simple Fields Based on Schema Type -->
				<div id="bw-schema-simple-fields">
					<!-- These will be populated dynamically based on schema type -->
				</div>
				
				<div class="bw-schema-simple-extras" style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 20px;">
					<h4 style="margin-top: 0;"><?php _e( '🤖 AI Optimization', 'bw-ai-schema-pro' ); ?></h4>
					
					<p>
						<label for="bw_schema_about_simple">
							<strong><?php _e( 'What is this content about?', 'bw-ai-schema-pro' ); ?></strong>
						</label>
						<input type="text" name="bw_schema_about_entities" id="bw_schema_about_simple" class="widefat" 
							value="<?php echo esc_attr( implode( ', ', (array) get_post_meta( $post->ID, '_bw_schema_about_entities', true ) ) ); ?>" 
							placeholder="<?php _e( 'e.g., Digital Marketing, SEO Strategy, Business Growth', 'bw-ai-schema-pro' ); ?>" />
						<span class="description"><?php _e( 'Help AI understand your main topics (comma-separated)', 'bw-ai-schema-pro' ); ?></span>
					</p>
					
					<!-- Multiple Authors Section -->
					<div class="bw-schema-multiple-authors" style="margin-top: 20px;">
						<h4 style="margin-bottom: 10px;">
							<?php _e( '👥 Authors', 'bw-ai-schema-pro' ); ?>
						</h4>
						
						<div id="bw-schema-authors-container">
							<?php
							$multiple_authors = get_post_meta( $post->ID, '_bw_schema_multiple_authors', true );
							$disable_default_author = get_post_meta( $post->ID, '_bw_schema_disable_default_author', true );
							
							// If no multiple authors, check for legacy single author
							if ( empty( $multiple_authors ) ) {
								$legacy_author = get_post_meta( $post->ID, '_bw_schema_custom_author', true );
								if ( $legacy_author ) {
									$multiple_authors = array(
										array(
											'type' => 'custom',
											'custom_author_id' => $legacy_author
										)
									);
								}
							}
							
							// Always show at least one author field
							if ( empty( $multiple_authors ) ) {
								$multiple_authors = array( array() );
							}
							
							foreach ( $multiple_authors as $index => $author ) {
								$this->render_author_field( $index, $author );
							}
							?>
						</div>
						
						<button type="button" class="button" id="bw-schema-add-author" style="margin-top: 10px;">
							<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Another Author', 'bw-ai-schema-pro' ); ?>
						</button>
						
						<?php
						// Get global author box settings
						$author_box_enabled = get_option( 'bw_schema_author_box_enabled', 'yes' );
						$enabled_post_types = get_option( 'bw_schema_author_box_post_types', array( 'post' ) );
						$is_supported_post_type = in_array( $post->post_type, $enabled_post_types );
						
						// Get the saved value for this post
						$show_author_box = get_post_meta( $post->ID, '_bw_schema_show_author_box', true );
						
						// Determine default checked state
						if ( $show_author_box === '' ) {
							// No value saved yet - use global settings
							$is_checked = ( $author_box_enabled === 'yes' && $is_supported_post_type );
						} else {
							// Use saved value
							$is_checked = ( $show_author_box === 'yes' );
						}
						?>
						<div style="margin-top: 15px; padding: 15px; background: #f5f5f5; border-radius: 5px;">
							<h4 style="margin: 0 0 10px 0;"><?php _e( 'Author Box Display Settings', 'bw-ai-schema-pro' ); ?></h4>
							
							<?php if ( $author_box_enabled !== 'yes' ) : ?>
								<p style="color: #d63638;">
									<span class="dashicons dashicons-info"></span>
									<?php _e( 'Author box is globally disabled. Enable it in Global Settings to use this feature.', 'bw-ai-schema-pro' ); ?>
								</p>
							<?php elseif ( ! $is_supported_post_type ) : ?>
								<p style="color: #d63638;">
									<span class="dashicons dashicons-info"></span>
									<?php printf( __( 'Author box is not enabled for %s post type in Global Settings.', 'bw-ai-schema-pro' ), $post->post_type ); ?>
								</p>
							<?php endif; ?>
							
							<p>
								<label>
									<input type="checkbox" name="bw_schema_show_author_box" value="yes" <?php checked( $is_checked ); ?> <?php disabled( $author_box_enabled !== 'yes' ); ?>>
									<?php _e( 'Show BW Schema Author Box below content', 'bw-ai-schema-pro' ); ?>
								</label>
							</p>
							<p>
								<label>
									<input type="checkbox" name="bw_schema_disable_default_author" value="1" <?php checked( $disable_default_author, '1' ); ?>>
									<?php _e( 'Disable WordPress default author box (recommended when using custom authors)', 'bw-ai-schema-pro' ); ?>
								</label>
							</p>
							<p class="description">
								<?php _e( 'The BW Schema Author Box supports multiple authors and external contributors. Use the shortcode [bw_author_box] to place it manually.', 'bw-ai-schema-pro' ); ?>
							</p>
						</div>
					</div>
				</div>
			</div>
			
			<!-- Advanced Mode -->
			<div id="bw-schema-advanced-mode" style="<?php echo $simple_mode === 'no' ? '' : 'display:none;'; ?>">
				<div class="bw-schema-advanced">
					<p>
						<label for="bw_schema_custom">
							<strong><?php _e( 'Custom Schema JSON-LD:', 'bw-ai-schema-pro' ); ?></strong>
						</label>
						<textarea name="bw_schema_custom" id="bw_schema_custom" class="widefat" rows="10" placeholder='{"@context": "https://schema.org", "@type": "Article", ...}'><?php echo esc_textarea( $custom_schema ); ?></textarea>
						<span class="description"><?php _e( 'Enter custom JSON-LD schema to override automatic generation.', 'bw-ai-schema-pro' ); ?></span>
					</p>
					
					<div class="bw-schema-ai-fields">
						<h4><?php _e( 'Additional AI Fields', 'bw-ai-schema-pro' ); ?></h4>
						
						<p>
							<label for="bw_schema_fact_checked_by">
								<strong><?php _e( 'Fact Checked By:', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="text" name="bw_schema_fact_checked_by" id="bw_schema_fact_checked_by" class="widefat" 
								value="<?php echo esc_attr( get_post_meta( $post->ID, '_bw_schema_fact_checked_by', true ) ); ?>" 
								placeholder="<?php _e( 'Name of fact checker', 'bw-ai-schema-pro' ); ?>" />
						</p>
						
						<p>
							<label for="bw_schema_last_reviewed">
								<strong><?php _e( 'Last Reviewed Date:', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="date" name="bw_schema_last_reviewed" id="bw_schema_last_reviewed" 
								value="<?php echo esc_attr( get_post_meta( $post->ID, '_bw_schema_last_reviewed', true ) ); ?>" />
						</p>
					</div>
				</div>
			</div>
			
			<?php if ( $post->post_type === 'page' ) : ?>
			<p style="margin-top: 10px;">
				<label for="bw_schema_is_team_page">
					<input type="checkbox" name="bw_schema_is_team_page" id="bw_schema_is_team_page" value="yes" <?php checked( get_post_meta( $post->ID, '_bw_schema_is_team_page', true ), 'yes' ); ?> />
					<strong><?php _e( 'This is a team page', 'bw-ai-schema-pro' ); ?></strong>
				</label>
				<span class="description" style="display: block; margin-top: 5px; margin-left: 24px;">
					<?php _e( 'Check this if this page displays team members using a query loop. The page will output an ItemList of Person schemas instead of a regular WebPage schema.', 'bw-ai-schema-pro' ); ?>
				</span>
			</p>
			<?php endif; ?>
			
			<p style="margin-top: 10px;">
				<label for="bw_schema_disable">
					<input type="checkbox" name="bw_schema_disable" id="bw_schema_disable" value="yes" <?php checked( $disable_schema, 'yes' ); ?> />
					<?php _e( 'Disable all schema markup for this post', 'bw-ai-schema-pro' ); ?>
				</label>
			</p>
			
			<div class="bw-schema-preview" style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #ddd;">
				<button type="button" class="button button-primary" id="bw-schema-preview-btn">
					<span class="dashicons dashicons-search" style="vertical-align: middle;"></span>
					<?php _e( 'Preview in Google Rich Results', 'bw-ai-schema-pro' ); ?>
				</button>
				<button type="button" class="button" id="bw-schema-preview-json-btn">
					<span class="dashicons dashicons-code-standards" style="vertical-align: middle;"></span>
					<?php _e( 'View JSON-LD', 'bw-ai-schema-pro' ); ?>
				</button>
				<div id="bw-schema-preview-output" style="display:none;"></div>
			</div>
			</div><!-- .bw-schema-box-content -->
		</div>
		
		<style>
			/* Collapsible Meta Box Styles */
			.bw-schema-collapsible-box { padding: 0; }
			.bw-schema-box-header {
				display: flex;
				justify-content: space-between;
				align-items: center;
				padding: 12px 15px;
				background: #f6f7f7;
				border-bottom: 1px solid #ddd;
				cursor: pointer;
				margin: -6px -12px 0;
			}
			.bw-schema-box-header:hover { background: #f0f0f1; }
			.bw-schema-box-summary { display: flex; align-items: center; gap: 12px; }
			.bw-schema-current-type {
				font-weight: 600;
				color: #1d2327;
				font-size: 13px;
			}
			.bw-schema-status {
				font-size: 11px;
				padding: 2px 8px;
				border-radius: 3px;
				text-transform: uppercase;
				font-weight: 500;
			}
			.bw-schema-status.enabled { background: #d4edda; color: #155724; }
			.bw-schema-status.disabled { background: #f8d7da; color: #721c24; }
			.bw-schema-toggle-btn {
				background: none;
				border: none;
				cursor: pointer;
				padding: 5px;
				color: #666;
				transition: transform 0.2s ease;
			}
			.bw-schema-toggle-btn:hover { color: #2271b1; }
			.bw-schema-toggle-btn[aria-expanded="true"] .dashicons { transform: rotate(180deg); }
			.bw-schema-box-content { padding: 15px 0 0; }

			/* Original Styles */
			.bw-schema-meta-box p { margin-bottom: 15px; }
			.bw-schema-advanced { margin-top: 20px; }
			.bw-schema-ai-fields { margin-top: 20px; }
			#bw-schema-preview-output { margin-top: 10px; padding: 15px; background: #f5f5f5; border: 1px solid #ddd; border-radius: 5px; }
			.bw-schema-mode-switcher input[type="radio"] { margin-right: 5px; }
			.bw-schema-simple-field { margin-bottom: 15px; }
			.bw-schema-simple-field label { display: block; margin-bottom: 5px; font-weight: 600; }
			.bw-schema-simple-field input,
			.bw-schema-simple-field select,
			.bw-schema-simple-field textarea { width: 100%; }
			.bw-schema-rich-preview { background: #fff; padding: 20px; border: 1px solid #ddd; border-radius: 5px; }
			.bw-schema-rich-preview h3 { color: #1a0dab; font-size: 18px; margin: 0 0 5px 0; }
			.bw-schema-rich-preview .url { color: #006621; font-size: 14px; }
			.bw-schema-rich-preview .description { color: #545454; font-size: 13px; margin-top: 5px; }
			.bw-schema-rich-preview .metadata { color: #70757a; font-size: 12px; margin-top: 5px; }
		</style>
		
		<script>
		jQuery(document).ready(function($) {
			// Collapsible meta box toggle
			$('.bw-schema-box-header').on('click', function() {
				var $content = $(this).siblings('.bw-schema-box-content');
				var $btn = $(this).find('.bw-schema-toggle-btn');
				var isExpanded = $btn.attr('aria-expanded') === 'true';

				if (isExpanded) {
					$content.slideUp(200);
					$btn.attr('aria-expanded', 'false');
				} else {
					$content.slideDown(200);
					$btn.attr('aria-expanded', 'true');
				}
			});

			// Update summary when schema type changes
			$('#bw_schema_type').on('change', function() {
				var val = $(this).val();
				var displayText = val ? val.split(':')[1] || val : '<?php _e( "Auto-detect", "bw-ai-schema-pro" ); ?>';
				$('.bw-schema-current-type').text(displayText);
			});

			// Update status when disable checkbox changes
			$('#bw_schema_disable').on('change', function() {
				var $status = $('.bw-schema-status');
				if ($(this).is(':checked')) {
					$status.removeClass('enabled').addClass('disabled').text('<?php _e( "Disabled", "bw-ai-schema-pro" ); ?>');
				} else {
					$status.removeClass('disabled').addClass('enabled').text('<?php _e( "Active", "bw-ai-schema-pro" ); ?>');
				}
			});

			// Mode switcher
			$('input[name="bw_schema_mode"]').on('change', function() {
				if ($(this).val() === 'simple') {
					$('#bw-schema-simple-mode').show();
					$('#bw-schema-advanced-mode').hide();
					$('input[name="bw_schema_simple_mode"]').val('yes');
				} else {
					$('#bw-schema-simple-mode').hide();
					$('#bw-schema-advanced-mode').show();
					$('input[name="bw_schema_simple_mode"]').val('no');
				}
			});
			
			// Use suggestion
			$('#bw-schema-use-suggestion').on('click', function() {
				var type = $(this).data('type');
				$('#bw_schema_type').val(type).trigger('change');
				$('.bw-schema-suggestion').fadeOut();
			});
			
			// Dynamic simple fields based on schema type
			$('#bw_schema_type').on('change', function() {
				var type = $(this).val();
				if (!type) return;
				
				// Load type-specific simple fields
				$.post(ajaxurl, {
					action: 'bw_schema_get_simple_fields',
					type: type,
					post_id: <?php echo $post->ID; ?>,
					nonce: '<?php echo wp_create_nonce( 'bw_schema_nonce' ); ?>'
				}, function(response) {
					if (response.success) {
						$('#bw-schema-simple-fields').html(response.data);
					}
				});
			});
			
			// Trigger change on load if type is selected
			if ($('#bw_schema_type').val()) {
				$('#bw_schema_type').trigger('change');
			}
			
			// Multiple Authors functionality
			var authorIndex = $('.bw-schema-author-item').length;
			
			// Add author
			$('#bw-schema-add-author').on('click', function() {
				var newAuthor = $('<div class="bw-schema-author-item" style="border: 1px solid #ddd; padding: 15px; margin-bottom: 10px; background: #f9f9f9; position: relative;"></div>');
				
				// Clone the first author item structure but with new index
				var firstAuthor = $('.bw-schema-author-item').first().clone();
				firstAuthor.find('input, select, textarea').each(function() {
					var name = $(this).attr('name');
					if (name) {
						$(this).attr('name', name.replace(/\[\d+\]/, '[' + authorIndex + ']'));
						$(this).val(''); // Clear values
					}
				});
				
				// Reset selections
				firstAuthor.find('select.bw-schema-author-type').val('wordpress');
				firstAuthor.find('.bw-schema-author-fields').hide();
				firstAuthor.find('.bw-schema-author-wordpress').show();
				
				// Add remove button if not present
				if (firstAuthor.find('.bw-schema-remove-author').length === 0) {
					firstAuthor.find('div:first').append('<button type="button" class="button bw-schema-remove-author" style="color: #a00;"><span class="dashicons dashicons-trash"></span> Remove</button>');
				}
				
				$('#bw-schema-authors-container').append(firstAuthor);
				authorIndex++;
				
				// Trigger change event to initialize
				firstAuthor.find('.bw-schema-author-type').trigger('change');
			});
			
			// Remove author
			$(document).on('click', '.bw-schema-remove-author', function() {
				$(this).closest('.bw-schema-author-item').fadeOut(300, function() {
					$(this).remove();
				});
			});
			
			// Toggle author type fields
			$(document).on('change', '.bw-schema-author-type', function() {
				var $container = $(this).closest('.bw-schema-author-item');
				var type = $(this).val();
				
				$container.find('.bw-schema-author-fields').hide();
				$container.find('.bw-schema-author-' + type).show();
			});
			
			// Toggle external author advanced fields
			$(document).on('click', '.bw-schema-toggle-external-fields', function() {
				var $advanced = $(this).siblings('.bw-schema-author-external-advanced');
				
				if ($advanced.is(':visible')) {
					$advanced.slideUp();
					$(this).text('<?php _e( 'Show Advanced Fields', 'bw-ai-schema-pro' ); ?>');
				} else {
					$advanced.slideDown();
					$(this).text('<?php _e( 'Hide Advanced Fields', 'bw-ai-schema-pro' ); ?>');
				}
			});
			
			// Initialize existing author fields
			$('.bw-schema-author-type').trigger('change');
		});
		</script>
		
		<input type="hidden" name="bw_schema_simple_mode" value="<?php echo esc_attr( $simple_mode ); ?>" />
		<?php
	}
	
	/**
	 * Auto-detect schema type based on content
	 */
	private function auto_detect_schema_type( $post ) {
		$content = strtolower( $post->post_content );
		$title = strtolower( $post->post_title );
		$categories = wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) );
		$post_type = strtolower( $post->post_type );

		// Check page mappings first - suggest Organization for mapped "about" pages
		$page_mappings = get_option( 'bw_schema_page_mappings', array() );
		if ( ! empty( $page_mappings ) ) {
			// Pages mapped as About, Services, or Company should use Organization
			$org_mapped_types = array( 'about', 'services', 'company' );
			foreach ( $org_mapped_types as $map_type ) {
				if ( isset( $page_mappings[ $map_type ] ) && intval( $page_mappings[ $map_type ] ) === $post->ID ) {
					return 'organization:Organization';
				}
			}
		}

		// Check for team member post types first
		$team_post_types = array( 'sw_teammember', 'team_member', 'team-member', 'teammember', 'staff', 'team' );
		if ( in_array( $post->post_type, $team_post_types ) ) {
			return 'person:Person';
		}
		
		// Check for landing page post types - these might not need schema
		$landing_page_types = array( 'landing_page', 'landing-page', 'landingpage', 'lp', 'landing', 'sales_page', 'squeeze_page' );
		if ( in_array( $post_type, $landing_page_types ) || 
		     strpos( $post_type, 'landing' ) !== false || 
		     strpos( $post_type, 'lp_' ) === 0 ) {
			// Check if it's a product/service landing page
			if ( strpos( $content, 'price' ) !== false || 
			     strpos( $content, 'buy now' ) !== false || 
			     strpos( $content, 'purchase' ) !== false ||
			     strpos( $title, 'product' ) !== false ||
			     strpos( $title, 'service' ) !== false ) {
				return 'product:Product';
			}
			// Check if the landing page is very minimal or temporary
			if ( strlen( $content ) < 500 || 
			     strpos( $title, 'coming soon' ) !== false ||
			     strpos( $title, 'under construction' ) !== false ||
			     strpos( $content, 'coming soon' ) !== false ) {
				return 'none:None';
			}
			// Default to No Schema for most landing pages
			return 'none:None';
		}
		
		// Check for other post types that typically don't need schema
		$no_schema_types = array( 'attachment', 'revision', 'nav_menu_item', 'custom_css', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block', 'wp_template', 'wp_template_part', 'wp_navigation' );
		if ( in_array( $post_type, $no_schema_types ) ) {
			return 'none:None';
		}
		
		// Check for specific patterns that suggest no schema needed
		if ( strpos( $title, 'thank you' ) !== false || 
		     strpos( $title, 'confirmation' ) !== false ||
		     strpos( $title, 'success' ) !== false ||
		     strpos( $title, 'download' ) !== false && strlen( $content ) < 300 ) {
			return 'none:None';
		}
		
		// Check for specific patterns - but not for landing pages already handled above
		if ( strpos( $title, 'review' ) !== false || strpos( $content, 'rating' ) !== false ) {
			return 'review:Review';
		}
		
		// For HowTo, be more specific - require "how to" not just "guide"
		// Landing pages often have "guide" in the title but aren't instructional
		if ( strpos( $title, 'how to' ) !== false || 
		     ( strpos( $title, 'guide' ) !== false && strpos( $title, 'step' ) !== false ) ||
		     ( strpos( $title, 'tutorial' ) !== false ) ) {
			return 'howto:HowTo';
		}
		
		if ( strpos( $title, 'faq' ) !== false || strpos( $content, 'frequently asked questions' ) !== false ) {
			return 'faq:FAQPage';
		}
		
		if ( $post->post_type === 'post' ) {
			// Check categories for hints
			foreach ( $categories as $cat ) {
				$cat_lower = strtolower( $cat );
				if ( strpos( $cat_lower, 'news' ) !== false ) {
					return 'article:NewsArticle';
				}
				if ( strpos( $cat_lower, 'tech' ) !== false || strpos( $cat_lower, 'development' ) !== false ) {
					return 'article:TechArticle';
				}
			}
			return 'article:BlogPosting';
		}
		
		if ( $post->post_type === 'page' ) {
			// Check for organization/company pages - suggest Organization schema
			// These pages typically describe the company itself
			$org_keywords = array( 'about us', 'about our', 'who we are', 'our story', 'our company', 'company profile', 'our mission', 'our vision' );
			foreach ( $org_keywords as $keyword ) {
				if ( strpos( $title, $keyword ) !== false || strpos( $content, $keyword ) !== false ) {
					return 'organization:Organization';
				}
			}

			// Simple "about" in title - also suggest Organization
			if ( strpos( $title, 'about' ) !== false ) {
				return 'organization:Organization';
			}

			// Contact pages
			if ( strpos( $title, 'contact' ) !== false || strpos( $title, 'get in touch' ) !== false || strpos( $title, 'reach us' ) !== false ) {
				return 'webpage:ContactPage';
			}

			// Team/staff pages
			if ( strpos( $title, 'team' ) !== false || strpos( $title, 'staff' ) !== false || strpos( $title, 'our people' ) !== false ) {
				return 'webpage:ProfilePage';
			}

			// Service pages - suggest LocalBusiness or Organization
			if ( strpos( $title, 'service' ) !== false ) {
				return 'organization:Organization';
			}

			// Default for pages should be WebPage
			return 'webpage:WebPage';
		}
		
		// Default for other post types (custom post types, etc.)
		$default = 'article:Article';
		
		// Allow developers to override auto-detection
		return apply_filters( 'bw_schema_auto_detect_override', $default, $post );
	}
	
	/**
	 * Save settings
	 */
	private function save_settings() {
		// Note: Most settings are now configured in the Setup Wizard.
		// This page only saves advanced organization details and display settings.

		// Get existing organization data to preserve wizard-configured fields
		$org_data = get_option( 'bw_schema_organization', array() );

		// Update only the advanced fields (preserve existing basic fields from wizard)
		$org_data['alternateName'] = sanitize_text_field( $_POST['org_alternate_name'] ?? '' );
		$org_data['legalName'] = sanitize_text_field( $_POST['org_legal_name'] ?? '' );
		$org_data['taxID'] = sanitize_text_field( $_POST['org_tax_id'] ?? '' );
		$org_data['duns'] = sanitize_text_field( $_POST['org_duns'] ?? '' );
		$org_data['slogan'] = sanitize_text_field( $_POST['org_slogan'] ?? '' );

		// Process areas served (comma-separated to array)
		if ( isset( $_POST['org_areas_served'] ) ) {
			if ( ! empty( $_POST['org_areas_served'] ) ) {
				$areas = array_map( 'trim', explode( ',', sanitize_text_field( $_POST['org_areas_served'] ) ) );
				$org_data['areaServed'] = array_filter( $areas );
			} else {
				$org_data['areaServed'] = array();
			}
		}

		// Process services/products offered (comma-separated to array)
		if ( isset( $_POST['org_services'] ) ) {
			if ( ! empty( $_POST['org_services'] ) ) {
				$services = array_map( 'trim', explode( ',', sanitize_textarea_field( $_POST['org_services'] ) ) );
				$org_data['makesOffer'] = array_filter( $services );
			} else {
				$org_data['makesOffer'] = array();
			}
		}

		update_option( 'bw_schema_organization', $org_data );

		// Save author box settings
		update_option( 'bw_schema_author_box_enabled', isset( $_POST['author_box_enabled'] ) ? 'yes' : 'no' );
		
		// Save author box post types
		if ( isset( $_POST['author_box_post_types'] ) && is_array( $_POST['author_box_post_types'] ) ) {
			$post_types = array_map( 'sanitize_text_field', $_POST['author_box_post_types'] );
			update_option( 'bw_schema_author_box_post_types', $post_types );
		} else {
			update_option( 'bw_schema_author_box_post_types', array() );
		}
		
		// Save author box position
		if ( isset( $_POST['author_box_position'] ) ) {
			$allowed_positions = array( 'after_content', 'before_content', 'manual' );
			$position = sanitize_text_field( $_POST['author_box_position'] );
			if ( in_array( $position, $allowed_positions ) ) {
				update_option( 'bw_schema_author_box_position', $position );
			}
		}

		// Save meta box display settings
		if ( isset( $_POST['metabox_position'] ) ) {
			$allowed_positions = array( 'normal', 'side' );
			$position = sanitize_text_field( $_POST['metabox_position'] );
			if ( in_array( $position, $allowed_positions ) ) {
				update_option( 'bw_schema_metabox_position', $position );
			}
		}

		if ( isset( $_POST['metabox_default_state'] ) ) {
			$allowed_states = array( 'collapsed', 'expanded' );
			$state = sanitize_text_field( $_POST['metabox_default_state'] );
			if ( in_array( $state, $allowed_states ) ) {
				update_option( 'bw_schema_metabox_default_state', $state );
			}
		}

		if ( isset( $_POST['metabox_post_types'] ) && is_array( $_POST['metabox_post_types'] ) ) {
			$post_types = array_map( 'sanitize_text_field', $_POST['metabox_post_types'] );
			update_option( 'bw_schema_metabox_post_types', $post_types );
		} else {
			// If none selected, save empty array
			update_option( 'bw_schema_metabox_post_types', array() );
		}

		// Add success message
		add_settings_error( 'bw_schema_settings', 'settings_saved', __( 'Settings saved successfully!', 'bw-ai-schema-pro' ), 'success' );
	}
	
	/**
	 * Save post type settings
	 */
	private function save_post_type_settings() {
		if ( isset( $_POST['post_type_defaults'] ) && is_array( $_POST['post_type_defaults'] ) ) {
			$defaults = array();
			foreach ( $_POST['post_type_defaults'] as $post_type => $schema_type ) {
				if ( ! empty( $schema_type ) ) {
					$defaults[$post_type] = sanitize_text_field( $schema_type );
				}
			}
			update_option( 'bw_schema_post_type_defaults', $defaults );
		}
		
		// Add success message
		add_settings_error( 'bw_schema_post_types', 'settings_saved', __( 'Post type settings saved successfully!', 'bw-ai-schema-pro' ), 'success' );
	}
	
	/**
	 * Save dashboard settings
	 */
	private function save_dashboard_settings() {
		// Save post type defaults
		if ( isset( $_POST['post_type_defaults'] ) && is_array( $_POST['post_type_defaults'] ) ) {
			$defaults = array();
			foreach ( $_POST['post_type_defaults'] as $post_type => $schema_type ) {
				if ( ! empty( $schema_type ) ) {
					$defaults[$post_type] = sanitize_text_field( $schema_type );
				}
			}
			update_option( 'bw_schema_post_type_defaults', $defaults );
		}
		
		// Add success message
		add_settings_error( 'bw_schema_dashboard', 'settings_saved', __( 'Settings saved successfully!', 'bw-ai-schema-pro' ), 'success' );
	}
	
	/**
	 * Save setup wizard data
	 */
	private function save_setup_wizard() {
		$current_step = isset( $_POST['setup_step'] ) ? intval( $_POST['setup_step'] ) : 1;
		
		// Save step 1 - Organization Profile
		if ( $current_step === 1 ) {
			// Get existing org data to preserve sameAs (set in Step 3)
			$existing_org = get_option( 'bw_schema_organization', array() );
			$existing_sameas = isset( $existing_org['sameAs'] ) ? $existing_org['sameAs'] : array();

			// Organization data - preserve existing sameAs
			$org_data = array(
				'name' => sanitize_text_field( $_POST['org_name'] ),
				'url' => esc_url_raw( $_POST['org_url'] ),
				'logo' => esc_url_raw( $_POST['org_logo'] ?? '' ),
				'telephone' => sanitize_text_field( $_POST['org_telephone'] ?? '' ),
				'email' => sanitize_email( $_POST['org_email'] ?? '' ),
				'sameAs' => $existing_sameas,
			);

			update_option( 'bw_schema_organization', $org_data );
			
			// Business details
			update_option( 'bw_schema_business_type', sanitize_text_field( $_POST['business_type'] ) );
			update_option( 'bw_schema_org_description', sanitize_textarea_field( $_POST['org_description'] ?? '' ) );
			update_option( 'bw_schema_founding_date', sanitize_text_field( $_POST['founding_date'] ?? '' ) );
			update_option( 'bw_schema_employee_count', intval( $_POST['employee_count'] ?? 0 ) );
			
			// Price range
			update_option( 'bw_schema_enable_price_range', isset( $_POST['enable_price_range'] ) ? 'yes' : 'no' );
			update_option( 'bw_schema_price_range', sanitize_text_field( $_POST['org_price_range'] ?? '' ) );
		}
		
		// Save step 2 - Leadership & Team
		elseif ( $current_step === 2 ) {
			// Author system preference
			update_option( 'bw_schema_use_custom_authors', sanitize_text_field( $_POST['use_custom_authors'] ?? 'wordpress' ) );
			
			// Founders/Leadership
			$founders = array();
			if ( isset( $_POST['founders'] ) && is_array( $_POST['founders'] ) ) {
				foreach ( $_POST['founders'] as $index => $founder ) {
					if ( ! empty( $founder['name'] ) ) {
						$founder_data = array(
							'name' => sanitize_text_field( $founder['name'] ),
							'jobTitle' => sanitize_text_field( $founder['jobTitle'] ?? '' ),
							'url' => esc_url_raw( $founder['url'] ?? '' ),
							'email' => sanitize_email( $founder['email'] ?? '' ),
							'description' => sanitize_textarea_field( $founder['description'] ?? '' ),
							'isDefault' => $index === 0 ? true : false,
						);
						$founders[] = $founder_data;
					}
				}
			}
			update_option( 'bw_schema_founders', $founders );
			
			// If using custom authors, create them from founders
			if ( $_POST['use_custom_authors'] === 'custom' && ! empty( $founders ) ) {
				$custom_authors = array();
				foreach ( $founders as $founder ) {
					$author = array(
						'id' => sanitize_title( $founder['name'] ) . '_' . time(),
						'name' => $founder['name'],
						'jobTitle' => $founder['jobTitle'],
						'description' => $founder['description'],
						'email' => $founder['email'],
						'url' => $founder['url'],
						'isDefault' => $founder['isDefault'],
					);
					$custom_authors[] = $author;
				}
				update_option( 'bw_schema_custom_authors', $custom_authors );
			}
			
			// Awards & Certifications
			$awards = array_filter( array_map( 'sanitize_text_field', $_POST['awards'] ?? array() ) );
			$certifications = array_filter( array_map( 'sanitize_text_field', $_POST['certifications'] ?? array() ) );
			
			update_option( 'bw_schema_awards', $awards );
			update_option( 'bw_schema_certifications', $certifications );
		}
		
		// Save step 3 - Contact & Social
		elseif ( $current_step === 3 ) {
			// Get existing org_data with proper defaults
			$org_data = get_option( 'bw_schema_organization', array(
				'name' => get_bloginfo( 'name' ),
				'url' => home_url(),
				'logo' => '',
				'telephone' => '',
				'email' => '',
				'sameAs' => array(),
			) );

			// Ensure we have proper structure
			if ( ! isset( $org_data['name'] ) ) {
				$org_data['name'] = get_bloginfo( 'name' );
			}
			if ( ! isset( $org_data['url'] ) ) {
				$org_data['url'] = home_url();
			}

			// Update sameAs (social profiles)
			$org_data['sameAs'] = array();
			if ( isset( $_POST['org_sameas'] ) && is_array( $_POST['org_sameas'] ) ) {
				foreach ( $_POST['org_sameas'] as $url ) {
					$url = trim( $url );
					if ( ! empty( $url ) ) {
						$org_data['sameAs'][] = esc_url_raw( $url );
					}
				}
			}

			update_option( 'bw_schema_organization', $org_data );
			
			// Physical address
			if ( isset( $_POST['address'] ) && is_array( $_POST['address'] ) ) {
				$address = array(
					'street' => sanitize_text_field( $_POST['address']['street'] ?? '' ),
					'city' => sanitize_text_field( $_POST['address']['city'] ?? '' ),
					'state' => sanitize_text_field( $_POST['address']['state'] ?? '' ),
					'postal' => sanitize_text_field( $_POST['address']['postal'] ?? '' ),
					'country' => sanitize_text_field( $_POST['address']['country'] ?? '' ),
				);
				update_option( 'bw_schema_address', $address );
			}
			
			// Contact points
			$contact_points = array();
			if ( isset( $_POST['contact_points'] ) && is_array( $_POST['contact_points'] ) ) {
				foreach ( $_POST['contact_points'] as $contact ) {
					if ( ! empty( $contact['type'] ) || ! empty( $contact['telephone'] ) || ! empty( $contact['email'] ) ) {
						$contact_points[] = array(
							'type' => sanitize_text_field( $contact['type'] ?? '' ),
							'telephone' => sanitize_text_field( $contact['telephone'] ?? '' ),
							'email' => sanitize_email( $contact['email'] ?? '' ),
						);
					}
				}
			}
			update_option( 'bw_schema_contact_points', $contact_points );
		}
		
		// Save step 4 - Page Mapping
		elseif ( $current_step === 4 ) {
			if ( isset( $_POST['page_mappings'] ) && is_array( $_POST['page_mappings'] ) ) {
				$mappings = array();
				foreach ( $_POST['page_mappings'] as $type => $page_id ) {
					$mappings[$type] = intval( $page_id );
				}
				update_option( 'bw_schema_page_mappings', $mappings );
			}
		}
		
		// Save step 5 - Content Defaults
		elseif ( $current_step === 5 ) {
			// Post type defaults
			if ( isset( $_POST['post_type_defaults'] ) && is_array( $_POST['post_type_defaults'] ) ) {
				$defaults = array();
				foreach ( $_POST['post_type_defaults'] as $post_type => $schema_type ) {
					if ( ! empty( $schema_type ) ) {
						$defaults[$post_type] = sanitize_text_field( $schema_type );
					}
				}
				update_option( 'bw_schema_post_type_defaults', $defaults );
			}
			
			// AI Optimization
			update_option( 'bw_schema_publishing_principles', esc_url_raw( $_POST['publishing_principles'] ?? '' ) );
			update_option( 'bw_schema_fact_checking_policy', esc_url_raw( $_POST['fact_checking_policy'] ?? '' ) );
		}
		
		// Save step 6 - Schema Features
		elseif ( $current_step === 6 ) {
			update_option( 'bw_schema_enable_schema', isset( $_POST['enable_schema'] ) ? 'yes' : 'no' );
			update_option( 'bw_schema_enable_breadcrumbs', isset( $_POST['enable_breadcrumbs'] ) ? 'yes' : 'no' );
			update_option( 'bw_schema_enable_sitelinks_search', isset( $_POST['enable_sitelinks_search'] ) ? 'yes' : 'no' );
			update_option( 'bw_schema_enable_cache', isset( $_POST['enable_cache'] ) ? 'yes' : 'no' );

			// Homepage organization schema
			update_option( 'bw_schema_output_org_homepage', isset( $_POST['output_org_homepage'] ) ? 'yes' : 'no' );

			// Team member post type
			update_option( 'bw_schema_team_post_type', sanitize_text_field( $_POST['team_post_type'] ?? '' ) );

			// Conflict management
			$disabled_sources = isset( $_POST['disable_sources'] ) ? array_map( 'sanitize_text_field', $_POST['disable_sources'] ) : array();
			update_option( 'bw_schema_disable_sources', $disabled_sources );
		}
		
		// Save step 7 - Review & Complete
		elseif ( $current_step === 7 ) {
			// Mark setup as complete
			update_option( 'bw_schema_setup_complete', true );
			
			// Clear any existing organization override since we're using proper settings now
			delete_option( 'bw_schema_organization_override' );
			
			// Redirect to dashboard with success message
			if ( isset( $_POST['complete_setup'] ) ) {
				wp_redirect( admin_url( 'options-general.php?page=bw-ai-schema&setup_complete=1' ) );
				exit;
			}
		}

		// Handle navigation
		if ( isset( $_POST['save_and_continue'] ) && $current_step < 7 ) {
			wp_redirect( admin_url( 'options-general.php?page=bw-ai-schema-setup&step=' . ( $current_step + 1 ) ) );
			exit;
		}
	}
	
	/**
	 * Save author
	 */
	private function save_author() {
		$authors = get_option( 'bw_schema_custom_authors', array() );
		
		// Create author data
		$author_data = array(
			'id' => isset( $_POST['author_id'] ) ? sanitize_text_field( $_POST['author_id'] ) : time(),
			'name' => sanitize_text_field( $_POST['author_name'] ),
			'jobTitle' => sanitize_text_field( $_POST['author_job_title'] ?? '' ),
			'credentials' => sanitize_text_field( $_POST['author_credentials'] ?? '' ),
			'description' => sanitize_textarea_field( $_POST['author_description'] ?? '' ),
			'image' => esc_url_raw( $_POST['author_image'] ?? '' ),
			'email' => sanitize_email( $_POST['author_email'] ?? '' ),
			'expertise' => sanitize_text_field( $_POST['author_expertise'] ?? '' ),
			'affiliations' => sanitize_text_field( $_POST['author_affiliations'] ?? '' ),
			'alumniOf' => sanitize_text_field( $_POST['author_alumni'] ?? '' ),
			'social' => array(
				'facebook' => esc_url_raw( $_POST['author_facebook'] ?? '' ),
				'twitter' => esc_url_raw( $_POST['author_twitter'] ?? '' ),
				'linkedin' => esc_url_raw( $_POST['author_linkedin'] ?? '' ),
				'instagram' => esc_url_raw( $_POST['author_instagram'] ?? '' ),
				'website' => esc_url_raw( $_POST['author_website'] ?? '' ),
				'youtube' => esc_url_raw( $_POST['author_youtube'] ?? '' ),
			),
			'teamPageId' => 0,
			'teamPageUrl' => '',
			'isDefault' => isset( $_POST['author_default'] ) ? true : false,
		);
		
		// Handle team page link based on link type
		if ( isset( $_POST['author_link_type'] ) && $_POST['author_link_type'] === 'external' ) {
			// External URL
			$author_data['teamPageUrl'] = esc_url_raw( $_POST['author_team_url'] ?? '' );
			$author_data['teamPageId'] = 'custom'; // Mark as custom URL
		} else {
			// Internal page/post
			$author_data['teamPageId'] = ! empty( $_POST['author_team_page'] ) ? intval( $_POST['author_team_page'] ) : 0;
			$author_data['teamPageUrl'] = ''; // Clear any external URL
		}
		
		// If setting as default, unset all other defaults
		if ( $author_data['isDefault'] ) {
			foreach ( $authors as &$author ) {
				$author['isDefault'] = false;
			}
		}
		
		// Check if updating existing author
		$found = false;
		foreach ( $authors as $key => $author ) {
			if ( $author['id'] == $author_data['id'] ) {
				$authors[$key] = $author_data;
				$found = true;
				break;
			}
		}
		
		// Add new author if not found
		if ( ! $found ) {
			$authors[] = $author_data;
		}
		
		// Save authors
		update_option( 'bw_schema_custom_authors', $authors );
		
		// Save author preference
		if ( isset( $_POST['use_custom_authors'] ) ) {
			update_option( 'bw_schema_use_custom_authors', $_POST['use_custom_authors'] );
		}
		
		// Redirect to prevent form resubmission
		wp_redirect( admin_url( 'options-general.php?page=bw-ai-schema-authors&message=saved' ) );
		exit;
	}

	/**
	 * Delete single author
	 */
	private function delete_author( $author_id ) {
		$authors = get_option( 'bw_schema_custom_authors', array() );

		// Remove author
		foreach ( $authors as $key => $author ) {
			if ( $author['id'] == $author_id ) {
				unset( $authors[$key] );
				break;
			}
		}

		// Re-index array
		$authors = array_values( $authors );

		// Save updated authors
		update_option( 'bw_schema_custom_authors', $authors );

		// Redirect
		wp_redirect( admin_url( 'options-general.php?page=bw-ai-schema-authors&message=deleted' ) );
		exit;
	}

	/**
	 * Bulk delete authors
	 */
	private function bulk_delete_authors() {
		if ( ! isset( $_POST['authors'] ) || ! is_array( $_POST['authors'] ) ) {
			return;
		}

		$authors = get_option( 'bw_schema_custom_authors', array() );
		$to_delete = array_map( 'sanitize_text_field', $_POST['authors'] );

		// Remove selected authors
		foreach ( $authors as $key => $author ) {
			if ( in_array( $author['id'], $to_delete ) ) {
				unset( $authors[$key] );
			}
		}

		// Re-index array
		$authors = array_values( $authors );

		// Save updated authors
		update_option( 'bw_schema_custom_authors', $authors );

		// Redirect
		wp_redirect( admin_url( 'options-general.php?page=bw-ai-schema-authors&message=bulk_deleted' ) );
		exit;
	}
	
	/**
	 * Render a single author field for multiple authors
	 */
	private function render_author_field( $index, $author_data = array() ) {
		$type = isset( $author_data['type'] ) ? $author_data['type'] : 'wordpress';
		$custom_authors = get_option( 'bw_schema_custom_authors', array() );
		$use_custom_authors = get_option( 'bw_schema_use_custom_authors', false );
		?>
		<div class="bw-schema-author-item" style="border: 1px solid #ddd; padding: 15px; margin-bottom: 10px; background: #f9f9f9; position: relative;">
			<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
				<select name="bw_schema_authors[<?php echo $index; ?>][type]" class="bw-schema-author-type">
					<option value="wordpress" <?php selected( $type, 'wordpress' ); ?>><?php _e( 'WordPress User', 'bw-ai-schema-pro' ); ?></option>
					<?php if ( $use_custom_authors && ! empty( $custom_authors ) ) : ?>
					<option value="custom" <?php selected( $type, 'custom' ); ?>><?php _e( 'Custom Author', 'bw-ai-schema-pro' ); ?></option>
					<?php endif; ?>
					<option value="external" <?php selected( $type, 'external' ); ?>><?php _e( 'External Author (One-time)', 'bw-ai-schema-pro' ); ?></option>
				</select>
				
				<?php if ( $index > 0 ) : ?>
				<button type="button" class="button bw-schema-remove-author" style="color: #a00;">
					<span class="dashicons dashicons-trash"></span> <?php _e( 'Remove', 'bw-ai-schema-pro' ); ?>
				</button>
				<?php endif; ?>
			</div>
			
			<!-- WordPress User Fields -->
			<div class="bw-schema-author-fields bw-schema-author-wordpress" style="display: none;">
				<?php
				$selected_user = isset( $author_data['wordpress_user_id'] ) ? $author_data['wordpress_user_id'] : '';
				wp_dropdown_users( array(
					'name' => 'bw_schema_authors[' . $index . '][wordpress_user_id]',
					'selected' => $selected_user,
					'show_option_none' => __( '-- Select User --', 'bw-ai-schema-pro' ),
					'option_none_value' => ''
				) );
				?>
			</div>
			
			<!-- Custom Author Fields -->
			<?php if ( $use_custom_authors && ! empty( $custom_authors ) ) : ?>
			<div class="bw-schema-author-fields bw-schema-author-custom" style="display: none;">
				<select name="bw_schema_authors[<?php echo $index; ?>][custom_author_id]" class="widefat">
					<option value=""><?php _e( '-- Select Custom Author --', 'bw-ai-schema-pro' ); ?></option>
					<?php foreach ( $custom_authors as $author ) : ?>
						<option value="<?php echo esc_attr( $author['id'] ); ?>" 
							<?php selected( isset( $author_data['custom_author_id'] ) ? $author_data['custom_author_id'] : '', $author['id'] ); ?>>
							<?php echo esc_html( $author['name'] ); ?>
							<?php if ( ! empty( $author['jobTitle'] ) ) : ?>
								- <?php echo esc_html( $author['jobTitle'] ); ?>
							<?php endif; ?>
						</option>
					<?php endforeach; ?>
				</select>
			</div>
			<?php endif; ?>
			
			<!-- External Author Fields -->
			<div class="bw-schema-author-fields bw-schema-author-external" style="display: none;">
				<div class="bw-schema-author-external-basic">
					<p>
						<label><?php _e( 'Name*', 'bw-ai-schema-pro' ); ?></label>
						<input type="text" name="bw_schema_authors[<?php echo $index; ?>][external][name]" 
						       value="<?php echo isset( $author_data['external']['name'] ) ? esc_attr( $author_data['external']['name'] ) : ''; ?>" 
						       class="widefat" placeholder="<?php _e( 'John Smith', 'bw-ai-schema-pro' ); ?>">
					</p>
					
					<p>
						<label><?php _e( 'Job Title', 'bw-ai-schema-pro' ); ?></label>
						<input type="text" name="bw_schema_authors[<?php echo $index; ?>][external][job_title]" 
						       value="<?php echo isset( $author_data['external']['job_title'] ) ? esc_attr( $author_data['external']['job_title'] ) : ''; ?>" 
						       class="widefat" placeholder="<?php _e( 'Industry Expert', 'bw-ai-schema-pro' ); ?>">
					</p>
					
					<p>
						<label><?php _e( 'Bio/Description', 'bw-ai-schema-pro' ); ?></label>
						<textarea name="bw_schema_authors[<?php echo $index; ?>][external][bio]" 
						          class="widefat" rows="3"><?php echo isset( $author_data['external']['bio'] ) ? esc_textarea( $author_data['external']['bio'] ) : ''; ?></textarea>
					</p>
				</div>
				
				<button type="button" class="button bw-schema-toggle-external-fields" style="margin: 10px 0;">
					<?php _e( 'Show Advanced Fields', 'bw-ai-schema-pro' ); ?>
				</button>
				
				<div class="bw-schema-author-external-advanced" style="display: none;">
					<p>
						<label><?php _e( 'Profile Image URL', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" name="bw_schema_authors[<?php echo $index; ?>][external][image]" 
						       value="<?php echo isset( $author_data['external']['image'] ) ? esc_attr( $author_data['external']['image'] ) : ''; ?>" 
						       class="widefat" placeholder="https://example.com/author-photo.jpg">
					</p>
					
					<p>
						<label><?php _e( 'Email', 'bw-ai-schema-pro' ); ?></label>
						<input type="email" name="bw_schema_authors[<?php echo $index; ?>][external][email]" 
						       value="<?php echo isset( $author_data['external']['email'] ) ? esc_attr( $author_data['external']['email'] ) : ''; ?>" 
						       class="widefat">
					</p>
					
					<p>
						<label><?php _e( 'Website', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" name="bw_schema_authors[<?php echo $index; ?>][external][website]" 
						       value="<?php echo isset( $author_data['external']['website'] ) ? esc_attr( $author_data['external']['website'] ) : ''; ?>" 
						       class="widefat">
					</p>
					
					<p>
						<label><?php _e( 'LinkedIn', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" name="bw_schema_authors[<?php echo $index; ?>][external][linkedin]" 
						       value="<?php echo isset( $author_data['external']['linkedin'] ) ? esc_attr( $author_data['external']['linkedin'] ) : ''; ?>" 
						       class="widefat" placeholder="https://linkedin.com/in/username">
					</p>
					
					<p>
						<label><?php _e( 'Twitter/X', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" name="bw_schema_authors[<?php echo $index; ?>][external][twitter]" 
						       value="<?php echo isset( $author_data['external']['twitter'] ) ? esc_attr( $author_data['external']['twitter'] ) : ''; ?>" 
						       class="widefat" placeholder="https://twitter.com/username">
					</p>
					
					<p>
						<label><?php _e( 'Areas of Expertise (comma-separated)', 'bw-ai-schema-pro' ); ?></label>
						<input type="text" name="bw_schema_authors[<?php echo $index; ?>][external][expertise]" 
						       value="<?php echo isset( $author_data['external']['expertise'] ) ? esc_attr( $author_data['external']['expertise'] ) : ''; ?>" 
						       class="widefat"
						       placeholder="<?php _e( 'e.g., SEO, Content Marketing, Web Development', 'bw-ai-schema-pro' ); ?>">
					</p>
				</div>
			</div>
		</div>
		<?php
	}
}