<?php
/**
 * Debug Team Schema for BW AI Schema Pro
 * 
 * This file helps debug team page schema generation issues
 */

// Add this to your theme's functions.php or as a mu-plugin to debug
add_action( 'wp_head', function() {
    if ( current_user_can( 'manage_options' ) ) {
        echo "\n<!-- BW AI Schema Pro Debug Info -->\n";
        echo "<!-- Current URL: " . get_permalink() . " -->\n";
        echo "<!-- Is Page: " . ( is_page() ? 'Yes' : 'No' ) . " -->\n";
        echo "<!-- Is Archive: " . ( is_archive() ? 'Yes' : 'No' ) . " -->\n";
        echo "<!-- Post Type: " . get_post_type() . " -->\n";
        
        // Check URL patterns
        $is_team_url = ( strpos( get_permalink(), '/team' ) !== false || strpos( get_permalink(), '/about/team' ) !== false );
        echo "<!-- Has /team in URL: " . ( $is_team_url ? 'Yes' : 'No' ) . " -->\n";
        
        // Check team post type setting
        $team_post_type = get_option( 'bw_schema_team_post_type', 'sw_teammember' );
        echo "<!-- Team Post Type Setting: " . $team_post_type . " -->\n";
        
        // Check if team member post type exists
        $post_types = get_post_types();
        echo "<!-- Team Post Type Exists: " . ( in_array( $team_post_type, $post_types ) ? 'Yes' : 'No' ) . " -->\n";
        
        // List all registered post types
        echo "<!-- Registered Post Types: " . implode( ', ', $post_types ) . " -->\n";
        
        // Check if this would trigger team schema
        $would_trigger_team = is_post_type_archive( $team_post_type ) || 
                            ( is_page() && ( strpos( get_permalink(), '/team' ) !== false || strpos( get_permalink(), '/about/team' ) !== false ) );
        echo "<!-- Would Trigger Team Schema: " . ( $would_trigger_team ? 'Yes' : 'No' ) . " -->\n";
        
        // Check if we're on a team member single page
        if ( is_singular( $team_post_type ) ) {
            echo "<!-- Is Single Team Member: Yes -->\n";
        }
        
        // Debug schema generation
        global $wp_filter;
        if ( isset( $wp_filter['bw_schema_rendered_schemas'] ) ) {
            echo "<!-- Schema Filters Active: Yes -->\n";
            echo "<!-- Filter Priority Count: " . count( $wp_filter['bw_schema_rendered_schemas'] ) . " -->\n";
        }
        
        echo "<!-- / BW AI Schema Pro Debug Info -->\n\n";
    }
}, 5 );

// Function to test team member detection
function bw_debug_team_members() {
    $team_post_type = get_option( 'bw_schema_team_post_type', 'sw_teammember' );
    
    echo "<h3>Team Member Debug Info</h3>";
    echo "<p>Team Post Type Setting: <strong>" . $team_post_type . "</strong></p>";
    
    // Query for team members
    $args = array(
        'post_type' => $team_post_type,
        'posts_per_page' => -1,
        'post_status' => 'publish'
    );
    
    $team_query = new WP_Query( $args );
    
    if ( $team_query->have_posts() ) {
        echo "<p>Found <strong>" . $team_query->found_posts . "</strong> team members.</p>";
        echo "<ol>";
        while ( $team_query->have_posts() ) {
            $team_query->the_post();
            echo "<li>" . get_the_title() . " (ID: " . get_the_ID() . ")</li>";
        }
        echo "</ol>";
        wp_reset_postdata();
    } else {
        echo "<p>No team members found for post type: " . $team_post_type . "</p>";
        
        // Try common team member post types
        $common_types = array( 'team', 'team_member', 'teammember', 'staff', 'employee' );
        foreach ( $common_types as $type ) {
            if ( post_type_exists( $type ) ) {
                $count = wp_count_posts( $type );
                if ( $count->publish > 0 ) {
                    echo "<p>Found post type '<strong>" . $type . "</strong>' with " . $count->publish . " published posts. Consider updating the team post type setting.</p>";
                }
            }
        }
    }
}

// Add admin notice for debugging
add_action( 'admin_notices', function() {
    if ( isset( $_GET['debug_team_schema'] ) && current_user_can( 'manage_options' ) ) {
        ?>
        <div class="notice notice-info">
            <h2>BW AI Schema Pro - Team Schema Debug</h2>
            <?php bw_debug_team_members(); ?>
            <hr>
            <h3>Enhanced Team Page Detection</h3>
            <p>The plugin now detects team pages using multiple methods:</p>
            <ol>
                <li><strong>URL patterns:</strong> /team, /our-team, /meet-the-team, /staff, /about/team, etc.</li>
                <li><strong>Page slugs:</strong> team, our-team, meet-the-team, staff, employees, team-members</li>
                <li><strong>Manual designation:</strong> Check "This is a team page" in the page editor sidebar</li>
                <li><strong>Content detection:</strong> Pages containing team member query blocks or shortcodes</li>
            </ol>
            <hr>
            <p>To fix team schema issues:</p>
            <ol>
                <li>Go to <a href="<?php echo admin_url( 'admin.php?page=bw-schema-settings' ); ?>">Schema Settings</a></li>
                <li>Update the "Team Member Post Type" setting to match your actual team member post type</li>
                <li>For pages that don't match the URL patterns, edit the page and check "This is a team page" in the sidebar</li>
                <li>Make sure team member posts are published and visible</li>
            </ol>
            <hr>
            <p><strong>Current Settings:</strong></p>
            <ul>
                <li>Team Post Type: <code><?php echo esc_html( get_option( 'bw_schema_team_post_type', 'sw_teammember' ) ); ?></code></li>
                <li>Schema Enabled: <?php echo get_option( 'bw_schema_enable_schema', 'yes' ) === 'yes' ? 'Yes' : 'No'; ?></li>
            </ul>
        </div>
        <?php
    }
} );

// Add settings field for team post type if not exists
add_action( 'admin_init', function() {
    // Check if settings section exists
    if ( ! get_option( 'bw_schema_team_post_type' ) ) {
        add_option( 'bw_schema_team_post_type', 'sw_teammember' );
    }
} );

// Add debug link to plugin action links
add_filter( 'plugin_action_links_bw-ai-schema-pro/bw-ai-schema-pro.php', function( $links ) {
    if ( current_user_can( 'manage_options' ) ) {
        $debug_link = '<a href="' . admin_url( 'options-general.php?page=bw-ai-schema&debug_team_schema=1' ) . '">Debug Team Schema</a>';
        array_unshift( $links, $debug_link );
    }
    return $links;
} );