/**
 * BW Schema Author Block Editor Script
 */

(function(blocks, element, editor, components, data) {
    const el = element.createElement;
    const { registerBlockType } = blocks;
    const { RichText, InspectorControls, MediaUpload, MediaUploadCheck } = editor;
    const { PanelBody, PanelRow, SelectControl, TextControl, TextareaControl, ToggleControl, Button, RadioControl, FormTokenField } = components;
    const { Fragment } = element;
    const { useSelect } = data;
    
    registerBlockType('bw-schema/author-schema', {
        title: 'Author Schema',
        icon: 'admin-users',
        category: 'common',
        description: 'Add custom author information with schema markup',
        keywords: ['author', 'schema', 'bio'],
        supports: {
            html: false,
            className: true
        },
        
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const {
                authorType,
                authorId,
                authorName,
                authorJobTitle,
                authorBio,
                authorImage,
                authorUrl,
                teamPageId,
                teamPageUrl,
                showFrontend,
                frontendStyle,
                knowsAbout,
                sameAs,
                alumniOf,
                credentials
            } = attributes;
            
            // Get custom authors from localized data
            const customAuthors = window.bwSchemaData.customAuthors || [];
            const pages = window.bwSchemaData.pages || [];
            
            // Build author options for dropdown
            const authorOptions = [
                { label: '-- Select Author --', value: '' },
                ...customAuthors
            ];
            
            // Build page options
            const pageOptions = [
                { label: '-- Select Team Page --', value: 0 },
                ...pages.map(page => ({
                    label: page.label,
                    value: page.value
                }))
            ];
            
            // Handle author selection
            const onSelectAuthor = (authorId) => {
                setAttributes({ authorId });
                
                // Auto-fill data from selected author
                const selectedAuthor = customAuthors.find(a => a.value === authorId);
                if (selectedAuthor && selectedAuthor.data) {
                    const data = selectedAuthor.data;
                    setAttributes({
                        authorName: data.name || '',
                        authorJobTitle: data.jobTitle || '',
                        authorBio: data.bio || '',
                        authorImage: data.image || '',
                        authorUrl: data.url || '',
                        knowsAbout: data.knowsAbout || [],
                        sameAs: data.sameAs || [],
                        alumniOf: data.alumniOf || '',
                        credentials: data.credentials || []
                    });
                }
            };
            
            // Handle team page selection
            const onSelectTeamPage = (pageId) => {
                setAttributes({ teamPageId: parseInt(pageId) });
                
                // Get page URL
                const selectedPage = pages.find(p => p.value === parseInt(pageId));
                if (selectedPage) {
                    setAttributes({ teamPageUrl: selectedPage.url });
                }
            };
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: 'Author Settings', initialOpen: true },
                        el(RadioControl, {
                            label: 'Author Type',
                            selected: authorType,
                            options: [
                                { label: 'Select Custom Author', value: 'custom' },
                                { label: 'Enter Manually', value: 'manual' }
                            ],
                            onChange: (value) => setAttributes({ authorType: value })
                        }),
                        
                        authorType === 'custom' && el(SelectControl, {
                            label: 'Select Author',
                            value: authorId,
                            options: authorOptions,
                            onChange: onSelectAuthor
                        }),
                        
                        authorType === 'manual' && el(Fragment, {},
                            el(TextControl, {
                                label: 'Author Name',
                                value: authorName,
                                onChange: (value) => setAttributes({ authorName: value })
                            }),
                            
                            el(TextControl, {
                                label: 'Job Title',
                                value: authorJobTitle,
                                onChange: (value) => setAttributes({ authorJobTitle: value })
                            }),
                            
                            el(TextareaControl, {
                                label: 'Biography',
                                value: authorBio,
                                onChange: (value) => setAttributes({ authorBio: value })
                            }),
                            
                            el(MediaUploadCheck, {},
                                el(MediaUpload, {
                                    onSelect: (media) => setAttributes({ authorImage: media.url }),
                                    allowedTypes: ['image'],
                                    value: authorImage,
                                    render: (obj) => {
                                        return el(Fragment, {},
                                            el(Button, {
                                                className: authorImage ? 'image-button' : 'button button-large',
                                                onClick: obj.open
                                            },
                                                !authorImage ? 'Upload Image' : el('img', { src: authorImage })
                                            ),
                                            authorImage && el(Button, {
                                                onClick: () => setAttributes({ authorImage: '' }),
                                                className: 'button',
                                                isDestructive: true,
                                                style: { marginLeft: '10px' }
                                            }, 'Remove')
                                        );
                                    }
                                })
                            )
                        )
                    ),
                    
                    el(PanelBody, { title: 'Link Settings', initialOpen: false },
                        el(SelectControl, {
                            label: 'Link to Team Page',
                            value: teamPageId,
                            options: pageOptions,
                            onChange: onSelectTeamPage,
                            help: 'Select an internal team/staff page to link to'
                        }),
                        
                        !teamPageId && el(TextControl, {
                            label: 'Or Custom URL',
                            value: authorUrl,
                            onChange: (value) => setAttributes({ authorUrl: value }),
                            placeholder: 'https://example.com/team/john-doe'
                        })
                    ),
                    
                    el(PanelBody, { title: 'Advanced Schema', initialOpen: false },
                        el(FormTokenField, {
                            label: 'Areas of Expertise (knowsAbout)',
                            value: knowsAbout || [],
                            onChange: (value) => setAttributes({ knowsAbout: value })
                        }),
                        
                        el(TextControl, {
                            label: 'Education (alumniOf)',
                            value: alumniOf,
                            onChange: (value) => setAttributes({ alumniOf: value }),
                            placeholder: 'Harvard University'
                        }),
                        
                        el(FormTokenField, {
                            label: 'Credentials',
                            value: credentials || [],
                            onChange: (value) => setAttributes({ credentials: value }),
                            help: 'Professional certifications or credentials'
                        }),
                        
                        el('p', { style: { marginTop: '20px', marginBottom: '10px' } }, 
                            el('strong', {}, 'Social Profiles')
                        ),
                        
                        el(TextControl, {
                            label: 'LinkedIn',
                            value: sameAs && sameAs[0] || '',
                            onChange: (value) => {
                                const newSameAs = [...(sameAs || [])];
                                newSameAs[0] = value;
                                setAttributes({ sameAs: newSameAs });
                            },
                            placeholder: 'https://linkedin.com/in/username'
                        }),
                        
                        el(TextControl, {
                            label: 'Twitter/X',
                            value: sameAs && sameAs[1] || '',
                            onChange: (value) => {
                                const newSameAs = [...(sameAs || [])];
                                newSameAs[1] = value;
                                setAttributes({ sameAs: newSameAs });
                            },
                            placeholder: 'https://twitter.com/username'
                        })
                    ),
                    
                    el(PanelBody, { title: 'Display Settings', initialOpen: false },
                        el(ToggleControl, {
                            label: 'Show Author Info on Frontend',
                            checked: showFrontend,
                            onChange: (value) => setAttributes({ showFrontend: value }),
                            help: 'Display author information visually on the page (schema is always added)'
                        }),
                        
                        showFrontend && el(RadioControl, {
                            label: 'Display Style',
                            selected: frontendStyle,
                            options: [
                                { label: 'Card (with image)', value: 'card' },
                                { label: 'Inline (text only)', value: 'inline' },
                                { label: 'Minimal', value: 'minimal' }
                            ],
                            onChange: (value) => setAttributes({ frontendStyle: value })
                        })
                    )
                ),
                
                el('div', { className: 'bw-schema-author-block-editor' },
                    el('div', { className: 'bw-schema-author-preview' },
                        el('h4', {}, 'Author Schema Block'),
                        
                        (authorType === 'custom' && authorId) || (authorType === 'manual' && authorName) ?
                            el('div', { className: 'author-preview-content' },
                                authorImage && el('img', { 
                                    src: authorImage, 
                                    style: { maxWidth: '100px', marginBottom: '10px' } 
                                }),
                                el('p', {}, 
                                    el('strong', {}, authorName || 'Author Name'),
                                    authorJobTitle && el('span', {}, ' - ' + authorJobTitle)
                                ),
                                authorBio && el('p', { style: { fontSize: '14px' } }, 
                                    authorBio.substring(0, 150) + (authorBio.length > 150 ? '...' : '')
                                ),
                                showFrontend && el('p', { 
                                    style: { 
                                        fontSize: '12px', 
                                        color: '#666',
                                        fontStyle: 'italic' 
                                    } 
                                }, 'Frontend display: ' + frontendStyle)
                            ) :
                            el('p', { style: { color: '#666' } }, 
                                'Select an author or enter information manually'
                            )
                    )
                )
            );
        },
        
        save: function() {
            // Server-side rendering
            return null;
        }
    });
})(
    window.wp.blocks,
    window.wp.element,
    window.wp.blockEditor || window.wp.editor,
    window.wp.components,
    window.wp.data
);