<?php
/**
 * Enhanced Setup Wizard for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get current step
$current_step = isset( $_GET['step'] ) ? intval( $_GET['step'] ) : 1;
$total_steps = 7;

// Get saved data
$org_data = BW_Schema_Core::get_organization_schema();
$business_type = get_option( 'bw_schema_business_type', '' );
$price_range = get_option( 'bw_schema_price_range', '' );
$enable_price_range = get_option( 'bw_schema_enable_price_range', 'no' );
$founding_date = get_option( 'bw_schema_founding_date', '' );
$employee_count = get_option( 'bw_schema_employee_count', '' );
$founders = get_option( 'bw_schema_founders', array() );
$awards = get_option( 'bw_schema_awards', array() );
$certifications = get_option( 'bw_schema_certifications', array() );
$contact_points = get_option( 'bw_schema_contact_points', array() );
$page_mappings = get_option( 'bw_schema_page_mappings', array() );
$use_custom_authors = get_option( 'bw_schema_use_custom_authors', 'wordpress' );

// Define steps
$steps = array(
	1 => __( 'Organization', 'bw-ai-schema-pro' ),
	2 => __( 'Leadership', 'bw-ai-schema-pro' ),
	3 => __( 'Contact', 'bw-ai-schema-pro' ),
	4 => __( 'Pages', 'bw-ai-schema-pro' ),
	5 => __( 'Content', 'bw-ai-schema-pro' ),
	6 => __( 'Features', 'bw-ai-schema-pro' ),
	7 => __( 'Review', 'bw-ai-schema-pro' ),
);
?>

<div class="wrap bw-schema-admin-wrap bw-schema-setup-wizard">
	<h1>
		<span class="dashicons dashicons-welcome-learn-more"></span>
		<?php _e( 'BW AI Schema Pro - Complete Setup', 'bw-ai-schema-pro' ); ?>
	</h1>
	
	<!-- Progress Bar -->
	<div class="bw-schema-setup-progress">
		<div class="bw-schema-progress-bar">
			<div class="bw-schema-progress-fill" style="width: <?php echo ( $current_step / $total_steps ) * 100; ?>%;"></div>
		</div>
		<div class="bw-schema-progress-steps">
			<?php foreach ( $steps as $step_num => $step_label ) : ?>
			<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-setup&step=' . $step_num ); ?>"
			   class="step <?php echo $current_step >= $step_num ? 'active' : ''; ?> <?php echo $current_step === $step_num ? 'current' : ''; ?>">
				<?php echo $step_num; ?>. <?php echo esc_html( $step_label ); ?>
			</a>
			<?php endforeach; ?>
		</div>
	</div>
	
	<form method="post" action="" class="bw-schema-wizard-form">
		<?php wp_nonce_field( 'bw_schema_setup', 'bw_schema_setup_nonce' ); ?>
		<input type="hidden" name="bw_schema_save_setup" value="1" />
		<input type="hidden" name="setup_step" value="<?php echo $current_step; ?>" />
		
		<div class="bw-schema-setup-content">
		
		<?php if ( $current_step === 1 ) : ?>
			<!-- Step 1: Organization Profile -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Organization Profile', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Let\'s start with your organization\'s basic information. This forms the foundation of your schema markup.', 'bw-ai-schema-pro' ); ?></p>

				<div class="notice notice-info" style="margin: 15px 0; padding: 12px 15px;">
					<p style="margin: 0;">
						<strong><?php _e( 'Global vs Page-Level Schema:', 'bw-ai-schema-pro' ); ?></strong><br>
						<?php _e( 'This wizard sets up your <strong>global organization identity</strong> that appears site-wide.', 'bw-ai-schema-pro' ); ?>
						<?php _e( 'To mark a specific page (like About Us or Company) as an <strong>Organization page</strong>, select "Organization" from the schema type dropdown when editing that page.', 'bw-ai-schema-pro' ); ?>
					</p>
				</div>

				<div class="bw-schema-setup-fields">
					<!-- Basic Info -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Basic Information', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label for="org_name">
								<strong><?php _e( 'Organization Name', 'bw-ai-schema-pro' ); ?></strong>
								<span class="required">*</span>
							</label>
							<input type="text" id="org_name" name="org_name" value="<?php echo esc_attr( $org_data['name'] ); ?>" class="large-text" required />
							<p class="description"><?php _e( 'Your company or organization\'s official name', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<div class="bw-schema-field">
							<label for="org_url">
								<strong><?php _e( 'Website URL', 'bw-ai-schema-pro' ); ?></strong>
								<span class="required">*</span>
							</label>
							<input type="url" id="org_url" name="org_url" value="<?php echo esc_url( $org_data['url'] ); ?>" class="large-text" required />
						</div>
						
						<div class="bw-schema-field">
							<label for="org_logo">
								<strong><?php _e( 'Logo URL', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<div class="bw-schema-media-field">
								<input type="url" id="org_logo" name="org_logo" value="<?php echo esc_url( $org_data['logo'] ?? '' ); ?>" class="large-text" />
								<button type="button" class="button bw-schema-media-select" data-target="#org_logo">
									<?php _e( 'Select Logo', 'bw-ai-schema-pro' ); ?>
								</button>
							</div>
							<p class="description"><?php _e( 'Recommended: 600x60px minimum, PNG or JPG format', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<div class="bw-schema-field">
							<label for="org_description">
								<strong><?php _e( 'Organization Description', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<textarea id="org_description" name="org_description" rows="3" class="large-text"><?php echo esc_textarea( get_option( 'bw_schema_org_description', get_bloginfo( 'description' ) ) ); ?></textarea>
							<p class="description"><?php _e( 'A brief description of what your organization does', 'bw-ai-schema-pro' ); ?></p>
						</div>

						<div class="bw-schema-field-row">
							<div class="bw-schema-field bw-schema-field-half">
								<label for="org_telephone">
									<strong><?php _e( 'Phone Number', 'bw-ai-schema-pro' ); ?></strong>
								</label>
								<input type="tel" id="org_telephone" name="org_telephone" value="<?php echo esc_attr( $org_data['telephone'] ?? '' ); ?>" class="large-text" placeholder="+1-555-555-5555" />
							</div>
							<div class="bw-schema-field bw-schema-field-half">
								<label for="org_email">
									<strong><?php _e( 'Email Address', 'bw-ai-schema-pro' ); ?></strong>
								</label>
								<input type="email" id="org_email" name="org_email" value="<?php echo esc_attr( $org_data['email'] ?? '' ); ?>" class="large-text" placeholder="info@example.com" />
							</div>
						</div>
					</fieldset>

					<!-- Business Details -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Business Details', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label for="business_type">
								<strong><?php _e( 'Business Type', 'bw-ai-schema-pro' ); ?></strong>
								<span class="required">*</span>
							</label>
							<select id="business_type" name="business_type" required>
								<option value=""><?php _e( '-- Select Business Type --', 'bw-ai-schema-pro' ); ?></option>
								<option value="professional" <?php selected( $business_type, 'professional' ); ?>><?php _e( 'Professional Services (Consulting, Agency, etc.)', 'bw-ai-schema-pro' ); ?></option>
								<option value="medical" <?php selected( $business_type, 'medical' ); ?>><?php _e( 'Medical/Healthcare', 'bw-ai-schema-pro' ); ?></option>
								<option value="hospitality" <?php selected( $business_type, 'hospitality' ); ?>><?php _e( 'Hospitality (Hotel, Restaurant)', 'bw-ai-schema-pro' ); ?></option>
								<option value="education" <?php selected( $business_type, 'education' ); ?>><?php _e( 'Educational Institution', 'bw-ai-schema-pro' ); ?></option>
								<option value="ecommerce" <?php selected( $business_type, 'ecommerce' ); ?>><?php _e( 'E-commerce/Online Store', 'bw-ai-schema-pro' ); ?></option>
								<option value="technology" <?php selected( $business_type, 'technology' ); ?>><?php _e( 'Technology/Software', 'bw-ai-schema-pro' ); ?></option>
								<option value="realestate" <?php selected( $business_type, 'realestate' ); ?>><?php _e( 'Real Estate', 'bw-ai-schema-pro' ); ?></option>
								<option value="financial" <?php selected( $business_type, 'financial' ); ?>><?php _e( 'Financial Services', 'bw-ai-schema-pro' ); ?></option>
								<option value="media" <?php selected( $business_type, 'media' ); ?>><?php _e( 'Media/Publishing', 'bw-ai-schema-pro' ); ?></option>
								<option value="nonprofit" <?php selected( $business_type, 'nonprofit' ); ?>><?php _e( 'Non-Profit Organization', 'bw-ai-schema-pro' ); ?></option>
								<option value="government" <?php selected( $business_type, 'government' ); ?>><?php _e( 'Government Agency', 'bw-ai-schema-pro' ); ?></option>
								<option value="personal" <?php selected( $business_type, 'personal' ); ?>><?php _e( 'Personal/Individual', 'bw-ai-schema-pro' ); ?></option>
							</select>
							<p class="description"><?php _e( 'This determines the default schema types for your content', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<div class="bw-schema-field">
							<label for="founding_date">
								<strong><?php _e( 'Founding Date', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="date" id="founding_date" name="founding_date" value="<?php echo esc_attr( $founding_date ); ?>" />
							<p class="description"><?php _e( 'When was your organization established?', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<div class="bw-schema-field">
							<label for="employee_count">
								<strong><?php _e( 'Number of Employees', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="number" id="employee_count" name="employee_count" value="<?php echo esc_attr( $employee_count ); ?>" min="1" />
							<p class="description"><?php _e( 'Approximate number of people in your organization', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<!-- Price Range -->
						<div class="bw-schema-field" id="price-range-container" style="<?php echo in_array( $business_type, array('professional', 'medical', 'hospitality', 'financial', 'realestate') ) ? '' : 'display: none;'; ?>">
							<label>
								<strong><?php _e( 'Price Range', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<div class="bw-schema-field-group">
								<label class="bw-schema-checkbox">
									<input type="checkbox" name="enable_price_range" value="yes" <?php checked( $enable_price_range, 'yes' ); ?> />
									<?php _e( 'Display price range in schema', 'bw-ai-schema-pro' ); ?>
								</label>
								<select name="org_price_range" id="org_price_range" <?php echo $enable_price_range !== 'yes' ? 'disabled' : ''; ?>>
									<option value=""><?php _e( '-- Select Price Range --', 'bw-ai-schema-pro' ); ?></option>
									<option value="$" <?php selected( $price_range, '$' ); ?>><?php _e( '$ - Inexpensive', 'bw-ai-schema-pro' ); ?></option>
									<option value="$$" <?php selected( $price_range, '$$' ); ?>><?php _e( '$$ - Moderate', 'bw-ai-schema-pro' ); ?></option>
									<option value="$$$" <?php selected( $price_range, '$$$' ); ?>><?php _e( '$$$ - Expensive', 'bw-ai-schema-pro' ); ?></option>
									<option value="$$$$" <?php selected( $price_range, '$$$$' ); ?>><?php _e( '$$$$ - Very Expensive', 'bw-ai-schema-pro' ); ?></option>
								</select>
							</div>
							<p class="description"><?php _e( 'General price range for your services/products', 'bw-ai-schema-pro' ); ?></p>
						</div>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 2 ) : ?>
			<!-- Step 2: Leadership & Team -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Leadership & Team', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Define your organization\'s leadership and team structure. This information enhances credibility and trust.', 'bw-ai-schema-pro' ); ?></p>
				
				<div class="bw-schema-setup-fields">
					<!-- Author System Choice -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Author System', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label><strong><?php _e( 'How do you want to manage authors?', 'bw-ai-schema-pro' ); ?></strong></label>
							<div class="bw-schema-radio-group">
								<label class="bw-schema-radio">
									<input type="radio" name="use_custom_authors" value="wordpress" <?php checked( $use_custom_authors, 'wordpress' ); ?> />
									<span><?php _e( 'Use WordPress Users', 'bw-ai-schema-pro' ); ?></span>
									<span class="description"><?php _e( 'Use existing WordPress user accounts as authors', 'bw-ai-schema-pro' ); ?></span>
								</label>
								<label class="bw-schema-radio">
									<input type="radio" name="use_custom_authors" value="custom" <?php checked( $use_custom_authors, 'custom' ); ?> />
									<span><?php _e( 'Use Custom Authors', 'bw-ai-schema-pro' ); ?></span>
									<span class="description"><?php _e( 'Create custom author profiles (CEO, team members) without WordPress accounts', 'bw-ai-schema-pro' ); ?></span>
								</label>
							</div>
						</div>
					</fieldset>
					
					<!-- Founders/Leadership -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Founders & Leadership', 'bw-ai-schema-pro' ); ?></legend>
						<p class="description"><?php _e( 'Add key people in your organization. The first person will be set as the default author.', 'bw-ai-schema-pro' ); ?></p>
						
						<div class="bw-schema-founders-repeater">
							<?php 
							if ( empty( $founders ) ) {
								$founders = array( array( 'name' => '', 'jobTitle' => '', 'url' => '', 'email' => '', 'description' => '' ) );
							}
							foreach ( $founders as $index => $founder ) : 
							?>
							<div class="bw-schema-founder-item" data-index="<?php echo $index; ?>">
								<div class="bw-schema-founder-header">
									<h4><?php echo $index === 0 ? __( 'Primary Leader (Default Author)', 'bw-ai-schema-pro' ) : sprintf( __( 'Team Member %d', 'bw-ai-schema-pro' ), $index + 1 ); ?></h4>
									<?php if ( $index > 0 ) : ?>
									<button type="button" class="button bw-schema-remove-founder"><?php _e( 'Remove', 'bw-ai-schema-pro' ); ?></button>
									<?php endif; ?>
								</div>
								
								<div class="bw-schema-field-row">
									<div class="bw-schema-field bw-schema-field-half">
										<label><?php _e( 'Name', 'bw-ai-schema-pro' ); ?> <span class="required">*</span></label>
										<input type="text" name="founders[<?php echo $index; ?>][name]" value="<?php echo esc_attr( $founder['name'] ?? '' ); ?>" required />
									</div>
									<div class="bw-schema-field bw-schema-field-half">
										<label><?php _e( 'Job Title', 'bw-ai-schema-pro' ); ?> <span class="required">*</span></label>
										<input type="text" name="founders[<?php echo $index; ?>][jobTitle]" value="<?php echo esc_attr( $founder['jobTitle'] ?? '' ); ?>" placeholder="<?php _e( 'e.g., CEO, Founder', 'bw-ai-schema-pro' ); ?>" required />
									</div>
								</div>
								
								<div class="bw-schema-field-row">
									<div class="bw-schema-field bw-schema-field-half">
										<label><?php _e( 'Email', 'bw-ai-schema-pro' ); ?></label>
										<input type="email" name="founders[<?php echo $index; ?>][email]" value="<?php echo esc_attr( $founder['email'] ?? '' ); ?>" />
									</div>
									<div class="bw-schema-field bw-schema-field-half">
										<label><?php _e( 'Profile URL', 'bw-ai-schema-pro' ); ?></label>
										<input type="url" name="founders[<?php echo $index; ?>][url]" value="<?php echo esc_attr( $founder['url'] ?? '' ); ?>" placeholder="<?php _e( 'LinkedIn or bio page', 'bw-ai-schema-pro' ); ?>" />
									</div>
								</div>
								
								<div class="bw-schema-field">
									<label><?php _e( 'Bio/Description', 'bw-ai-schema-pro' ); ?></label>
									<textarea name="founders[<?php echo $index; ?>][description]" rows="2"><?php echo esc_textarea( $founder['description'] ?? '' ); ?></textarea>
								</div>
								
								<?php if ( $index === 0 ) : ?>
								<input type="hidden" name="founders[0][isDefault]" value="1" />
								<?php endif; ?>
							</div>
							<?php endforeach; ?>
						</div>
						
						<button type="button" class="button" id="bw-schema-add-founder">
							<span class="dashicons dashicons-plus-alt"></span> <?php _e( 'Add Team Member', 'bw-ai-schema-pro' ); ?>
						</button>
					</fieldset>
					
					<!-- Awards & Recognition -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Achievements', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label><?php _e( 'Awards & Recognition', 'bw-ai-schema-pro' ); ?></label>
							<div class="bw-schema-awards-repeater">
								<?php 
								if ( empty( $awards ) ) {
									$awards = array( '' );
								}
								foreach ( $awards as $index => $award ) : 
								?>
								<div class="bw-schema-award-item">
									<input type="text" name="awards[]" value="<?php echo esc_attr( $award ); ?>" placeholder="<?php _e( 'e.g., Best Workplace 2023', 'bw-ai-schema-pro' ); ?>" />
									<button type="button" class="button bw-schema-remove-award"><?php _e( 'Remove', 'bw-ai-schema-pro' ); ?></button>
								</div>
								<?php endforeach; ?>
							</div>
							<button type="button" class="button button-small" id="bw-schema-add-award">
								<span class="dashicons dashicons-plus-alt"></span> <?php _e( 'Add Award', 'bw-ai-schema-pro' ); ?>
							</button>
						</div>
						
						<div class="bw-schema-field">
							<label><?php _e( 'Certifications', 'bw-ai-schema-pro' ); ?></label>
							<div class="bw-schema-certifications-repeater">
								<?php 
								if ( empty( $certifications ) ) {
									$certifications = array( '' );
								}
								foreach ( $certifications as $index => $cert ) : 
								?>
								<div class="bw-schema-cert-item">
									<input type="text" name="certifications[]" value="<?php echo esc_attr( $cert ); ?>" placeholder="<?php _e( 'e.g., ISO 9001:2015', 'bw-ai-schema-pro' ); ?>" />
									<button type="button" class="button bw-schema-remove-cert"><?php _e( 'Remove', 'bw-ai-schema-pro' ); ?></button>
								</div>
								<?php endforeach; ?>
							</div>
							<button type="button" class="button button-small" id="bw-schema-add-cert">
								<span class="dashicons dashicons-plus-alt"></span> <?php _e( 'Add Certification', 'bw-ai-schema-pro' ); ?>
							</button>
						</div>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 3 ) : ?>
			<!-- Step 3: Contact & Social -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Contact & Social Presence', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Provide ways for people and AI to connect with your organization.', 'bw-ai-schema-pro' ); ?></p>
				
				<div class="bw-schema-setup-fields">
					<!-- Social Media -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Social Media Profiles', 'bw-ai-schema-pro' ); ?></legend>
						<p class="description"><?php _e( 'Add your social media profiles to establish credibility and identity.', 'bw-ai-schema-pro' ); ?></p>
						
						<div class="bw-schema-social-repeater">
							<?php 
							$social_profiles = ! empty( $org_data['sameAs'] ) ? $org_data['sameAs'] : array('');
							foreach ( $social_profiles as $index => $profile ) : 
							?>
							<div class="bw-schema-social-item">
								<select name="org_social_type[<?php echo $index; ?>]" class="social-type-select">
									<option value=""><?php _e( '-- Select Platform --', 'bw-ai-schema-pro' ); ?></option>
									<optgroup label="<?php _e( 'Social Media', 'bw-ai-schema-pro' ); ?>">
										<option value="facebook" <?php selected( strpos( $profile, 'facebook.com' ) !== false ); ?>><?php _e( 'Facebook', 'bw-ai-schema-pro' ); ?></option>
										<option value="twitter" <?php selected( strpos( $profile, 'twitter.com' ) !== false || strpos( $profile, 'x.com' ) !== false ); ?>><?php _e( 'Twitter/X', 'bw-ai-schema-pro' ); ?></option>
										<option value="linkedin" <?php selected( strpos( $profile, 'linkedin.com' ) !== false ); ?>><?php _e( 'LinkedIn', 'bw-ai-schema-pro' ); ?></option>
										<option value="instagram" <?php selected( strpos( $profile, 'instagram.com' ) !== false ); ?>><?php _e( 'Instagram', 'bw-ai-schema-pro' ); ?></option>
										<option value="youtube" <?php selected( strpos( $profile, 'youtube.com' ) !== false ); ?>><?php _e( 'YouTube', 'bw-ai-schema-pro' ); ?></option>
									</optgroup>
								</select>
								<input type="url" name="org_sameas[<?php echo $index; ?>]" value="<?php echo esc_url( $profile ); ?>" placeholder="<?php _e( 'Profile URL', 'bw-ai-schema-pro' ); ?>" />
								<button type="button" class="button bw-schema-remove-social"><?php _e( 'Remove', 'bw-ai-schema-pro' ); ?></button>
							</div>
							<?php endforeach; ?>
						</div>
						<button type="button" class="button" id="bw-schema-add-social">
							<span class="dashicons dashicons-plus-alt"></span> <?php _e( 'Add Social Profile', 'bw-ai-schema-pro' ); ?>
						</button>
					</fieldset>
					
					<!-- Physical Address -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Physical Address', 'bw-ai-schema-pro' ); ?></legend>
						<p class="description"><?php _e( 'Your business location. Leave blank if you operate online only.', 'bw-ai-schema-pro' ); ?></p>

						<?php
						$address = get_option( 'bw_schema_address', array() );
						?>

						<div class="bw-schema-field">
							<label for="address_street"><?php _e( 'Street Address', 'bw-ai-schema-pro' ); ?></label>
							<input type="text" id="address_street" name="address[street]" value="<?php echo esc_attr( $address['street'] ?? '' ); ?>" class="large-text" placeholder="123 Main Street" />
						</div>

						<div class="bw-schema-field-row">
							<div class="bw-schema-field bw-schema-field-third">
								<label for="address_city"><?php _e( 'City', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" id="address_city" name="address[city]" value="<?php echo esc_attr( $address['city'] ?? '' ); ?>" />
							</div>
							<div class="bw-schema-field bw-schema-field-third">
								<label for="address_state"><?php _e( 'State/Region', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" id="address_state" name="address[state]" value="<?php echo esc_attr( $address['state'] ?? '' ); ?>" />
							</div>
							<div class="bw-schema-field bw-schema-field-third">
								<label for="address_postal"><?php _e( 'Postal/ZIP Code', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" id="address_postal" name="address[postal]" value="<?php echo esc_attr( $address['postal'] ?? '' ); ?>" />
							</div>
						</div>

						<div class="bw-schema-field">
							<label for="address_country"><?php _e( 'Country', 'bw-ai-schema-pro' ); ?></label>
							<input type="text" id="address_country" name="address[country]" value="<?php echo esc_attr( $address['country'] ?? '' ); ?>" class="regular-text" placeholder="United States" />
						</div>
					</fieldset>

					<!-- Contact Points -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Additional Contact Points', 'bw-ai-schema-pro' ); ?></legend>
						<p class="description"><?php _e( 'Add department-specific contacts (optional). Your main phone/email was set in Step 1.', 'bw-ai-schema-pro' ); ?></p>

						<div class="bw-schema-contact-repeater">
							<?php 
							if ( empty( $contact_points ) ) {
								$contact_points = array( array( 'type' => '', 'telephone' => '', 'email' => '' ) );
							}
							foreach ( $contact_points as $index => $contact ) : 
							?>
							<div class="bw-schema-contact-item">
								<h4><?php echo sprintf( __( 'Contact Point %d', 'bw-ai-schema-pro' ), $index + 1 ); ?></h4>
								<div class="bw-schema-field-row">
									<div class="bw-schema-field bw-schema-field-third">
										<label><?php _e( 'Department', 'bw-ai-schema-pro' ); ?></label>
										<select name="contact_points[<?php echo $index; ?>][type]">
											<option value=""><?php _e( '-- Select Type --', 'bw-ai-schema-pro' ); ?></option>
											<option value="customer service" <?php selected( $contact['type'] ?? '', 'customer service' ); ?>><?php _e( 'Customer Service', 'bw-ai-schema-pro' ); ?></option>
											<option value="technical support" <?php selected( $contact['type'] ?? '', 'technical support' ); ?>><?php _e( 'Technical Support', 'bw-ai-schema-pro' ); ?></option>
											<option value="sales" <?php selected( $contact['type'] ?? '', 'sales' ); ?>><?php _e( 'Sales', 'bw-ai-schema-pro' ); ?></option>
											<option value="billing" <?php selected( $contact['type'] ?? '', 'billing' ); ?>><?php _e( 'Billing', 'bw-ai-schema-pro' ); ?></option>
										</select>
									</div>
									<div class="bw-schema-field bw-schema-field-third">
										<label><?php _e( 'Telephone', 'bw-ai-schema-pro' ); ?></label>
										<input type="tel" name="contact_points[<?php echo $index; ?>][telephone]" value="<?php echo esc_attr( $contact['telephone'] ?? '' ); ?>" />
									</div>
									<div class="bw-schema-field bw-schema-field-third">
										<label><?php _e( 'Email', 'bw-ai-schema-pro' ); ?></label>
										<input type="email" name="contact_points[<?php echo $index; ?>][email]" value="<?php echo esc_attr( $contact['email'] ?? '' ); ?>" />
									</div>
								</div>
								<?php if ( $index > 0 ) : ?>
								<button type="button" class="button bw-schema-remove-contact"><?php _e( 'Remove Contact', 'bw-ai-schema-pro' ); ?></button>
								<?php endif; ?>
							</div>
							<?php endforeach; ?>
						</div>
						<button type="button" class="button" id="bw-schema-add-contact">
							<span class="dashicons dashicons-plus-alt"></span> <?php _e( 'Add Contact Point', 'bw-ai-schema-pro' ); ?>
						</button>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 4 ) : ?>
			<!-- Step 4: Page Mapping -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Page Mapping', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Map your existing pages to appropriate schema types. This ensures each page has the most relevant structured data.', 'bw-ai-schema-pro' ); ?></p>

				<div class="notice notice-success" style="margin: 15px 0; padding: 12px 15px;">
					<p style="margin: 0;">
						<strong><?php _e( 'Smart Organization Detection:', 'bw-ai-schema-pro' ); ?></strong>
						<?php _e( 'Pages you map as "About", "Services", or company-related will automatically be suggested to use <strong>Organization schema</strong> in the post editor for better search engine visibility.', 'bw-ai-schema-pro' ); ?>
					</p>
				</div>

				<div class="bw-schema-setup-fields">
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Assign Schema Types to Your Pages', 'bw-ai-schema-pro' ); ?></legend>
						
						<?php
						// Get all pages
						$pages = get_pages( array( 'post_status' => 'publish' ) );
						$page_options = array( '' => __( '-- Not Set --', 'bw-ai-schema-pro' ) );
						foreach ( $pages as $page ) {
							$page_options[$page->ID] = $page->post_title;
						}
						
						// Define page types
						$page_types = array(
							'home' => array(
								'label' => __( 'Homepage', 'bw-ai-schema-pro' ),
								'description' => __( 'Your main website homepage', 'bw-ai-schema-pro' ),
								'schema' => 'WebSite',
								'icon' => 'dashicons-admin-home'
							),
							'about' => array(
								'label' => __( 'About Page', 'bw-ai-schema-pro' ),
								'description' => __( 'Page describing your organization', 'bw-ai-schema-pro' ),
								'schema' => 'AboutPage',
								'icon' => 'dashicons-info'
							),
							'services' => array(
								'label' => __( 'Services/Products Page', 'bw-ai-schema-pro' ),
								'description' => __( 'Main services or products listing', 'bw-ai-schema-pro' ),
								'schema' => 'CollectionPage',
								'icon' => 'dashicons-cart'
							),
							'contact' => array(
								'label' => __( 'Contact Page', 'bw-ai-schema-pro' ),
								'description' => __( 'Contact information and forms', 'bw-ai-schema-pro' ),
								'schema' => 'ContactPage',
								'icon' => 'dashicons-email'
							),
							'blog' => array(
								'label' => __( 'Blog Index', 'bw-ai-schema-pro' ),
								'description' => __( 'Main blog or news page', 'bw-ai-schema-pro' ),
								'schema' => 'Blog',
								'icon' => 'dashicons-admin-post'
							),
							'team' => array(
								'label' => __( 'Team/Staff Page', 'bw-ai-schema-pro' ),
								'description' => __( 'Team members listing', 'bw-ai-schema-pro' ),
								'schema' => 'CollectionPage',
								'icon' => 'dashicons-groups'
							),
							'terms' => array(
								'label' => __( 'Terms of Service', 'bw-ai-schema-pro' ),
								'description' => __( 'Legal terms and conditions', 'bw-ai-schema-pro' ),
								'schema' => 'WebPage',
								'icon' => 'dashicons-media-document'
							),
							'privacy' => array(
								'label' => __( 'Privacy Policy', 'bw-ai-schema-pro' ),
								'description' => __( 'Privacy and data policies', 'bw-ai-schema-pro' ),
								'schema' => 'WebPage',
								'icon' => 'dashicons-privacy'
							)
						);
						
						foreach ( $page_types as $type_key => $type_info ) :
							$current_value = isset( $page_mappings[$type_key] ) ? $page_mappings[$type_key] : '';
						?>
						<div class="bw-schema-page-mapping">
							<div class="bw-schema-page-type">
								<span class="<?php echo esc_attr( $type_info['icon'] ); ?>"></span>
								<div>
									<strong><?php echo esc_html( $type_info['label'] ); ?></strong>
									<span class="description"><?php echo esc_html( $type_info['description'] ); ?></span>
									<span class="schema-type"><?php echo sprintf( __( 'Schema: %s', 'bw-ai-schema-pro' ), $type_info['schema'] ); ?></span>
								</div>
							</div>
							<div class="bw-schema-page-select">
								<select name="page_mappings[<?php echo esc_attr( $type_key ); ?>]">
									<?php foreach ( $page_options as $page_id => $page_title ) : ?>
									<option value="<?php echo esc_attr( $page_id ); ?>" <?php selected( $current_value, $page_id ); ?>>
										<?php echo esc_html( $page_title ); ?>
									</option>
									<?php endforeach; ?>
								</select>
							</div>
						</div>
						<?php endforeach; ?>
						
						<div class="bw-schema-note">
							<span class="dashicons dashicons-info"></span>
							<p><?php _e( 'Individual team member pages will automatically use ProfilePage schema when they link to team members defined in Step 2.', 'bw-ai-schema-pro' ); ?></p>
						</div>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 5 ) : ?>
			<!-- Step 5: Content Defaults -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Content Defaults', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Set default schema types for different content types and configure AI optimization settings.', 'bw-ai-schema-pro' ); ?></p>
				
				<div class="bw-schema-setup-fields">
					<!-- Post Type Defaults -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Default Schema for Post Types', 'bw-ai-schema-pro' ); ?></legend>
						
						<?php
						$post_types = get_post_types( array( 'public' => true ), 'objects' );
						$schema_types = BW_Schema_Core::get_schema_types();
						$defaults = BW_Schema_Core::get_post_type_defaults();
						
						foreach ( $post_types as $post_type ) :
							if ( in_array( $post_type->name, array( 'attachment' ) ) ) continue;
						?>
						<div class="bw-schema-field">
							<label for="post_type_<?php echo esc_attr( $post_type->name ); ?>">
								<strong><?php echo esc_html( $post_type->label ); ?></strong>
							</label>
							<select name="post_type_defaults[<?php echo esc_attr( $post_type->name ); ?>]" id="post_type_<?php echo esc_attr( $post_type->name ); ?>">
								<option value=""><?php _e( '-- Select Default Schema --', 'bw-ai-schema-pro' ); ?></option>
								<?php foreach ( $schema_types as $key => $type ) : ?>
								<optgroup label="<?php echo esc_attr( $type['label'] ); ?>">
									<?php foreach ( $type['subtypes'] as $subtype_key => $subtype_label ) : ?>
									<option value="<?php echo esc_attr( $key . ':' . $subtype_key ); ?>" <?php selected( isset( $defaults[$post_type->name] ) ? $defaults[$post_type->name] : '', $key . ':' . $subtype_key ); ?>>
										<?php echo esc_html( $subtype_label ); ?>
									</option>
									<?php endforeach; ?>
								</optgroup>
								<?php endforeach; ?>
							</select>
						</div>
						<?php endforeach; ?>
					</fieldset>
					
					<!-- AI Optimization -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'AI Optimization', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label for="publishing_principles">
								<strong><?php _e( 'Publishing Principles URL', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="url" id="publishing_principles" name="publishing_principles" value="<?php echo esc_url( get_option( 'bw_schema_publishing_principles', '' ) ); ?>" class="large-text" />
							<p class="description"><?php _e( 'Link to your editorial guidelines or publishing principles page', 'bw-ai-schema-pro' ); ?></p>
						</div>
						
						<div class="bw-schema-field">
							<label for="fact_checking_policy">
								<strong><?php _e( 'Fact-Checking Policy URL', 'bw-ai-schema-pro' ); ?></strong>
							</label>
							<input type="url" id="fact_checking_policy" name="fact_checking_policy" value="<?php echo esc_url( get_option( 'bw_schema_fact_checking_policy', '' ) ); ?>" class="large-text" />
							<p class="description"><?php _e( 'Link to your fact-checking or review process documentation', 'bw-ai-schema-pro' ); ?></p>
						</div>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 6 ) : ?>
			<!-- Step 6: Schema Features -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Schema Features', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Configure advanced schema features and manage potential conflicts with other plugins.', 'bw-ai-schema-pro' ); ?></p>
				
				<div class="bw-schema-setup-fields">
					<!-- Enable/Disable Features -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Feature Settings', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-field">
							<label class="bw-schema-checkbox">
								<input type="checkbox" name="enable_schema" value="yes" <?php checked( get_option( 'bw_schema_enable_schema', 'yes' ), 'yes' ); ?> />
								<strong><?php _e( 'Enable Schema Markup', 'bw-ai-schema-pro' ); ?></strong>
								<span class="description"><?php _e( 'Master switch for all schema output', 'bw-ai-schema-pro' ); ?></span>
							</label>
						</div>
						
						<div class="bw-schema-field">
							<label class="bw-schema-checkbox">
								<input type="checkbox" name="enable_breadcrumbs" value="yes" <?php checked( get_option( 'bw_schema_enable_breadcrumbs', 'yes' ), 'yes' ); ?> />
								<strong><?php _e( 'Enable Breadcrumb Schema', 'bw-ai-schema-pro' ); ?></strong>
								<span class="description"><?php _e( 'Helps search engines understand your site structure', 'bw-ai-schema-pro' ); ?></span>
							</label>
						</div>
						
						<div class="bw-schema-field">
							<label class="bw-schema-checkbox">
								<input type="checkbox" name="enable_sitelinks_search" value="yes" <?php checked( get_option( 'bw_schema_enable_sitelinks_search', 'yes' ), 'yes' ); ?> />
								<strong><?php _e( 'Enable Sitelinks Search Box', 'bw-ai-schema-pro' ); ?></strong>
								<span class="description"><?php _e( 'Shows search box in Google results', 'bw-ai-schema-pro' ); ?></span>
							</label>
						</div>
						
						<div class="bw-schema-field">
							<label class="bw-schema-checkbox">
								<input type="checkbox" name="enable_cache" value="yes" <?php checked( get_option( 'bw_schema_enable_cache', 'yes' ), 'yes' ); ?> />
								<strong><?php _e( 'Enable Schema Caching', 'bw-ai-schema-pro' ); ?></strong>
								<span class="description"><?php _e( 'Improves performance by caching generated schemas', 'bw-ai-schema-pro' ); ?></span>
							</label>
						</div>

						<div class="bw-schema-field">
							<label class="bw-schema-checkbox">
								<input type="checkbox" name="output_org_homepage" value="yes" <?php checked( get_option( 'bw_schema_output_org_homepage', 'no' ), 'yes' ); ?> />
								<strong><?php _e( 'Output Organization Schema on Homepage', 'bw-ai-schema-pro' ); ?></strong>
								<span class="description"><?php _e( 'Output Organization schema instead of WebSite schema on homepage (recommended for business sites)', 'bw-ai-schema-pro' ); ?></span>
							</label>
						</div>

						<div class="bw-schema-field">
							<label for="team_post_type"><strong><?php _e( 'Team Member Post Type', 'bw-ai-schema-pro' ); ?></strong></label>
							<select id="team_post_type" name="team_post_type">
								<?php
								$team_post_type = get_option( 'bw_schema_team_post_type', '' );
								$post_types = get_post_types( array( 'public' => true ), 'objects' );
								?>
								<option value=""><?php _e( '-- None (Not using team members) --', 'bw-ai-schema-pro' ); ?></option>
								<?php foreach ( $post_types as $pt ) :
									if ( in_array( $pt->name, array( 'attachment', 'revision', 'nav_menu_item' ) ) ) continue;
								?>
								<option value="<?php echo esc_attr( $pt->name ); ?>" <?php selected( $team_post_type, $pt->name ); ?>>
									<?php echo esc_html( $pt->labels->name ); ?>
								</option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php _e( 'Select the post type used for team members to generate Person schema.', 'bw-ai-schema-pro' ); ?></p>
						</div>
					</fieldset>

					<!-- Conflict Management -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Schema Conflict Management', 'bw-ai-schema-pro' ); ?></legend>
						<p class="description"><?php _e( 'Detect and disable schema from other sources to prevent conflicts.', 'bw-ai-schema-pro' ); ?></p>
						
						<?php
						$detected_sources = BW_Schema_Core::detect_schema_sources();
						$disabled_sources = get_option( 'bw_schema_disable_sources', array() );
						
						if ( ! empty( $detected_sources ) ) :
						?>
						<div class="bw-schema-conflict-list">
							<?php foreach ( $detected_sources as $key => $source ) : ?>
							<div class="bw-schema-conflict-item">
								<label class="bw-schema-checkbox">
									<input type="checkbox" name="disable_sources[]" value="<?php echo esc_attr( $key ); ?>" <?php checked( in_array( $key, $disabled_sources ) ); ?> />
									<strong><?php echo esc_html( $source['name'] ); ?></strong>
									<span class="source-type"><?php echo esc_html( ucfirst( $source['type'] ) ); ?></span>
									<span class="description"><?php _e( 'Disable schema output from this source', 'bw-ai-schema-pro' ); ?></span>
								</label>
							</div>
							<?php endforeach; ?>
						</div>
						<?php else : ?>
						<div class="bw-schema-success-message">
							<span class="dashicons dashicons-yes-alt"></span>
							<?php _e( 'No conflicting schema sources detected. You have full control!', 'bw-ai-schema-pro' ); ?>
						</div>
						<?php endif; ?>
					</fieldset>
				</div>
			</div>
			
		<?php elseif ( $current_step === 7 ) : ?>
			<!-- Step 7: Review & Complete -->
			<div class="bw-schema-setup-step">
				<h2><?php _e( 'Review & Complete Setup', 'bw-ai-schema-pro' ); ?></h2>
				<p class="step-description"><?php _e( 'Review your configuration and complete the setup. You can always change these settings later.', 'bw-ai-schema-pro' ); ?></p>
				
				<div class="bw-schema-setup-fields">
					<!-- Configuration Summary -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Configuration Summary', 'bw-ai-schema-pro' ); ?></legend>
						
						<div class="bw-schema-summary">
							<div class="bw-schema-summary-section">
								<h4><?php _e( 'Organization', 'bw-ai-schema-pro' ); ?></h4>
								<ul>
									<li><strong><?php _e( 'Name:', 'bw-ai-schema-pro' ); ?></strong> <?php echo esc_html( $org_data['name'] ); ?></li>
									<li><strong><?php _e( 'Type:', 'bw-ai-schema-pro' ); ?></strong> <?php echo esc_html( ucfirst( $business_type ) ); ?></li>
									<li><strong><?php _e( 'Founded:', 'bw-ai-schema-pro' ); ?></strong> <?php echo esc_html( $founding_date ?: __( 'Not set', 'bw-ai-schema-pro' ) ); ?></li>
									<li><strong><?php _e( 'Employees:', 'bw-ai-schema-pro' ); ?></strong> <?php echo esc_html( $employee_count ?: __( 'Not set', 'bw-ai-schema-pro' ) ); ?></li>
								</ul>
							</div>
							
							<div class="bw-schema-summary-section">
								<h4><?php _e( 'Leadership', 'bw-ai-schema-pro' ); ?></h4>
								<ul>
									<?php if ( ! empty( $founders ) ) : ?>
										<?php foreach ( $founders as $index => $founder ) : ?>
											<?php if ( ! empty( $founder['name'] ) ) : ?>
											<li>
												<?php echo esc_html( $founder['name'] ); ?> - <?php echo esc_html( $founder['jobTitle'] ); ?>
												<?php if ( $index === 0 ) : ?>
												<span class="badge"><?php _e( 'Default Author', 'bw-ai-schema-pro' ); ?></span>
												<?php endif; ?>
											</li>
											<?php endif; ?>
										<?php endforeach; ?>
									<?php else : ?>
										<li><?php _e( 'No team members added', 'bw-ai-schema-pro' ); ?></li>
									<?php endif; ?>
								</ul>
							</div>
							
							<div class="bw-schema-summary-section">
								<h4><?php _e( 'Page Mappings', 'bw-ai-schema-pro' ); ?></h4>
								<ul>
									<?php 
									$mapped_count = 0;
									foreach ( $page_mappings as $type => $page_id ) {
										if ( ! empty( $page_id ) ) $mapped_count++;
									}
									?>
									<li><?php echo sprintf( __( '%d pages mapped to schema types', 'bw-ai-schema-pro' ), $mapped_count ); ?></li>
								</ul>
							</div>
							
							<div class="bw-schema-summary-section">
								<h4><?php _e( 'Features', 'bw-ai-schema-pro' ); ?></h4>
								<ul>
									<li><?php _e( 'Schema Output:', 'bw-ai-schema-pro' ); ?> <strong><?php echo get_option( 'bw_schema_enable_schema', 'yes' ) === 'yes' ? __( 'Enabled', 'bw-ai-schema-pro' ) : __( 'Disabled', 'bw-ai-schema-pro' ); ?></strong></li>
									<li><?php _e( 'Breadcrumbs:', 'bw-ai-schema-pro' ); ?> <strong><?php echo get_option( 'bw_schema_enable_breadcrumbs', 'yes' ) === 'yes' ? __( 'Enabled', 'bw-ai-schema-pro' ) : __( 'Disabled', 'bw-ai-schema-pro' ); ?></strong></li>
									<li><?php _e( 'Price Range:', 'bw-ai-schema-pro' ); ?> <strong><?php echo $enable_price_range === 'yes' ? $price_range ?: __( 'Enabled', 'bw-ai-schema-pro' ) : __( 'Disabled', 'bw-ai-schema-pro' ); ?></strong></li>
								</ul>
							</div>
						</div>
					</fieldset>
					
					<!-- Generated Schema Preview -->
					<fieldset class="bw-schema-fieldset">
						<legend><?php _e( 'Generated Organization Schema Preview', 'bw-ai-schema-pro' ); ?></legend>

						<?php
						// Generate comprehensive preview schema matching renderer output
						$preview_schema = array(
							'@context' => 'https://schema.org',
							'@type' => 'Organization',
							'name' => $org_data['name'],
							'url' => $org_data['url'],
						);

						// Telephone and email
						if ( ! empty( $org_data['telephone'] ) ) {
							$preview_schema['telephone'] = $org_data['telephone'];
						}
						if ( ! empty( $org_data['email'] ) ) {
							$preview_schema['email'] = $org_data['email'];
						}

						// Logo
						if ( ! empty( $org_data['logo'] ) ) {
							$preview_schema['logo'] = array(
								'@type' => 'ImageObject',
								'url' => $org_data['logo'],
							);
						}

						// Social profiles
						if ( ! empty( $org_data['sameAs'] ) ) {
							$preview_schema['sameAs'] = array_filter( $org_data['sameAs'] );
						}

						// Description
						$org_description = get_option( 'bw_schema_org_description', '' );
						if ( ! empty( $org_description ) ) {
							$preview_schema['description'] = $org_description;
						}

						// Founding date
						if ( ! empty( $founding_date ) ) {
							$preview_schema['foundingDate'] = $founding_date;
						}

						// Number of employees
						if ( ! empty( $employee_count ) && $employee_count > 0 ) {
							$preview_schema['numberOfEmployees'] = array(
								'@type' => 'QuantitativeValue',
								'value' => intval( $employee_count )
							);
						}

						// Address
						$address = get_option( 'bw_schema_address', array() );
						if ( ! empty( $address ) && ! empty( $address['street'] ) ) {
							$postal_address = array( '@type' => 'PostalAddress' );
							if ( ! empty( $address['street'] ) ) $postal_address['streetAddress'] = $address['street'];
							if ( ! empty( $address['city'] ) ) $postal_address['addressLocality'] = $address['city'];
							if ( ! empty( $address['state'] ) ) $postal_address['addressRegion'] = $address['state'];
							if ( ! empty( $address['postal'] ) ) $postal_address['postalCode'] = $address['postal'];
							if ( ! empty( $address['country'] ) ) $postal_address['addressCountry'] = $address['country'];
							$preview_schema['address'] = $postal_address;
						}

						// Contact points
						if ( ! empty( $contact_points ) && is_array( $contact_points ) ) {
							$schema_contacts = array();
							foreach ( $contact_points as $contact ) {
								if ( ! empty( $contact['telephone'] ) || ! empty( $contact['email'] ) ) {
									$cp = array( '@type' => 'ContactPoint' );
									if ( ! empty( $contact['type'] ) ) $cp['contactType'] = $contact['type'];
									if ( ! empty( $contact['telephone'] ) ) $cp['telephone'] = $contact['telephone'];
									if ( ! empty( $contact['email'] ) ) $cp['email'] = $contact['email'];
									$schema_contacts[] = $cp;
								}
							}
							if ( ! empty( $schema_contacts ) ) {
								$preview_schema['contactPoint'] = $schema_contacts;
							}
						}

						// Founders
						if ( ! empty( $founders ) && is_array( $founders ) ) {
							$schema_founders = array();
							foreach ( $founders as $founder ) {
								if ( ! empty( $founder['name'] ) ) {
									$person = array( '@type' => 'Person', 'name' => $founder['name'] );
									if ( ! empty( $founder['jobTitle'] ) ) $person['jobTitle'] = $founder['jobTitle'];
									if ( ! empty( $founder['url'] ) ) $person['url'] = $founder['url'];
									$schema_founders[] = $person;
								}
							}
							if ( ! empty( $schema_founders ) ) {
								$preview_schema['founder'] = count( $schema_founders ) === 1 ? $schema_founders[0] : $schema_founders;
							}
						}

						// Awards
						if ( ! empty( $awards ) && is_array( $awards ) ) {
							$filtered_awards = array_filter( $awards );
							if ( ! empty( $filtered_awards ) ) {
								$preview_schema['award'] = array_values( $filtered_awards );
							}
						}

						// Certifications
						if ( ! empty( $certifications ) && is_array( $certifications ) ) {
							$filtered_certs = array_filter( $certifications );
							if ( ! empty( $filtered_certs ) ) {
								$preview_schema['hasCredential'] = array_values( $filtered_certs );
							}
						}

						$schema_json = wp_json_encode( $preview_schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );
						?>

						<div class="bw-schema-code-preview">
							<pre><?php echo esc_html( $schema_json ); ?></pre>
						</div>
						<p class="description" style="margin-top: 10px;">
							<?php _e( 'This preview shows the Organization schema that will be output on your homepage. Test it at', 'bw-ai-schema-pro' ); ?>
							<a href="https://validator.schema.org/" target="_blank">validator.schema.org</a>
						</p>
					</fieldset>
					
					<!-- Complete Setup -->
					<div class="bw-schema-complete-actions">
						<input type="hidden" name="complete_setup" value="1" />
						<p class="description">
							<?php _e( 'Your configuration is ready! Click "Complete Setup" to save all settings and start using BW AI Schema Pro.', 'bw-ai-schema-pro' ); ?>
						</p>
					</div>
				</div>
			</div>
		<?php endif; ?>
		
		</div>
		
		<!-- Navigation -->
		<div class="bw-schema-setup-navigation">
			<?php if ( $current_step > 1 ) : ?>
			<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-setup&step=' . ( $current_step - 1 ) ); ?>" class="button button-secondary">
				<?php _e( '← Previous', 'bw-ai-schema-pro' ); ?>
			</a>
			<?php endif; ?>
			
			<?php if ( $current_step < $total_steps ) : ?>
			<button type="submit" name="save_and_continue" class="button button-primary">
				<?php _e( 'Save & Continue →', 'bw-ai-schema-pro' ); ?>
			</button>
			<?php else : ?>
			<button type="submit" name="complete_setup" class="button button-primary button-hero">
				<?php _e( 'Complete Setup', 'bw-ai-schema-pro' ); ?>
			</button>
			<?php endif; ?>
		</div>
	</form>
</div>

<style>
/* Enhanced Wizard Styles */
.bw-schema-setup-wizard {
	max-width: 900px;
	margin: 20px auto;
}

.bw-schema-setup-progress {
	margin: 30px 0;
}

.bw-schema-progress-bar {
	height: 8px;
	background: #e0e0e0;
	border-radius: 4px;
	overflow: hidden;
	margin-bottom: 20px;
}

.bw-schema-progress-fill {
	height: 100%;
	background: #2271b1;
	transition: width 0.3s ease;
}

.bw-schema-progress-steps {
	display: flex;
	justify-content: space-between;
	flex-wrap: wrap;
	gap: 10px;
}

.bw-schema-progress-steps .step {
	font-size: 13px;
	color: #666;
	padding: 5px 10px;
	border-radius: 3px;
	background: #f0f0f0;
	flex: 1;
	text-align: center;
	min-width: 100px;
	text-decoration: none;
	cursor: pointer;
	transition: all 0.2s ease;
}

.bw-schema-progress-steps .step:hover {
	background: #e0e0e0;
	color: #333;
}

.bw-schema-progress-steps .step.active {
	color: #2271b1;
	font-weight: 600;
	background: #e5f5ff;
}

.bw-schema-progress-steps .step.current {
	background: #2271b1;
	color: white;
}

.bw-schema-setup-content {
	background: white;
	padding: 30px;
	border: 1px solid #ddd;
	border-radius: 8px;
	margin-bottom: 30px;
}

.bw-schema-setup-step h2 {
	margin-top: 0;
	margin-bottom: 10px;
	font-size: 24px;
}

.step-description {
	color: #666;
	margin-bottom: 30px;
	font-size: 15px;
}

.bw-schema-fieldset {
	border: 1px solid #e0e0e0;
	border-radius: 5px;
	padding: 20px;
	margin-bottom: 25px;
}

.bw-schema-fieldset legend {
	font-weight: 600;
	font-size: 16px;
	padding: 0 10px;
	color: #23282d;
}

.bw-schema-field {
	margin-bottom: 20px;
}

.bw-schema-field label {
	display: block;
	margin-bottom: 5px;
}

.bw-schema-field strong {
	font-weight: 600;
}

.bw-schema-field .required {
	color: #d63638;
}

.bw-schema-field input[type="text"],
.bw-schema-field input[type="url"],
.bw-schema-field input[type="email"],
.bw-schema-field input[type="tel"],
.bw-schema-field input[type="number"],
.bw-schema-field input[type="date"],
.bw-schema-field select,
.bw-schema-field textarea {
	width: 100%;
	max-width: 600px;
}

.bw-schema-field .description {
	display: block;
	margin-top: 5px;
	font-size: 13px;
	color: #666;
	font-style: normal;
}

.bw-schema-field-row {
	display: flex;
	gap: 15px;
	margin-bottom: 15px;
}

.bw-schema-field-half {
	flex: 1;
}

.bw-schema-field-third {
	flex: 0 0 31%;
}

.bw-schema-media-field {
	display: flex;
	gap: 10px;
	align-items: center;
}

.bw-schema-media-field input {
	flex: 1;
}

/* Repeater Styles */
.bw-schema-founder-item {
	background: #f8f8f8;
	border: 1px solid #e0e0e0;
	border-radius: 5px;
	padding: 20px;
	margin-bottom: 15px;
}

.bw-schema-founder-header {
	display: flex;
	justify-content: space-between;
	align-items: center;
	margin-bottom: 15px;
}

.bw-schema-founder-header h4 {
	margin: 0;
	font-size: 16px;
}

.bw-schema-social-item,
.bw-schema-award-item,
.bw-schema-cert-item {
	display: flex;
	gap: 10px;
	margin-bottom: 10px;
	align-items: center;
}

.bw-schema-social-item select {
	flex: 0 0 200px;
}

.bw-schema-social-item input {
	flex: 1;
}

/* Checkbox and Radio Styles */
.bw-schema-checkbox,
.bw-schema-radio {
	display: flex;
	align-items: flex-start;
	margin-bottom: 10px;
	cursor: pointer;
}

.bw-schema-checkbox input,
.bw-schema-radio input {
	margin-right: 8px;
	margin-top: 3px;
}

.bw-schema-radio-group {
	display: flex;
	flex-direction: column;
	gap: 15px;
}

.bw-schema-radio span {
	display: block;
}

.bw-schema-radio .description {
	font-size: 13px;
	color: #666;
	margin-top: 3px;
}

/* Page Mapping Styles */
.bw-schema-page-mapping {
	display: flex;
	justify-content: space-between;
	align-items: center;
	padding: 15px;
	border: 1px solid #e0e0e0;
	border-radius: 5px;
	margin-bottom: 10px;
	background: #f9f9f9;
}

.bw-schema-page-type {
	display: flex;
	gap: 15px;
	align-items: center;
	flex: 1;
}

.bw-schema-page-type .dashicons {
	font-size: 24px;
	width: 24px;
	height: 24px;
	color: #2271b1;
}

.bw-schema-page-type strong {
	display: block;
	margin-bottom: 3px;
}

.bw-schema-page-type .description {
	font-size: 13px;
	color: #666;
}

.bw-schema-page-type .schema-type {
	display: block;
	font-size: 11px;
	color: #999;
	margin-top: 3px;
}

.bw-schema-page-select {
	flex: 0 0 250px;
}

.bw-schema-page-select select {
	width: 100%;
}

/* Note Box */
.bw-schema-note {
	background: #f0f8ff;
	border: 1px solid #b3d9ff;
	border-radius: 5px;
	padding: 15px;
	margin-top: 20px;
	display: flex;
	gap: 10px;
	align-items: flex-start;
}

.bw-schema-note .dashicons {
	color: #0073aa;
	flex-shrink: 0;
}

/* Conflict Management */
.bw-schema-conflict-item {
	padding: 10px;
	border: 1px solid #e0e0e0;
	border-radius: 5px;
	margin-bottom: 10px;
	background: #f9f9f9;
}

.bw-schema-conflict-item .source-type {
	display: inline-block;
	background: #e0e0e0;
	padding: 2px 8px;
	border-radius: 3px;
	font-size: 12px;
	margin-left: 10px;
}

/* Summary Styles */
.bw-schema-summary {
	display: grid;
	grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
	gap: 20px;
}

.bw-schema-summary-section {
	background: #f8f8f8;
	padding: 15px;
	border-radius: 5px;
}

.bw-schema-summary-section h4 {
	margin: 0 0 10px 0;
	font-size: 14px;
	color: #23282d;
}

.bw-schema-summary-section ul {
	margin: 0;
	padding-left: 20px;
}

.bw-schema-summary-section li {
	margin-bottom: 5px;
	font-size: 13px;
}

.badge {
	display: inline-block;
	background: #2271b1;
	color: white;
	padding: 2px 8px;
	border-radius: 3px;
	font-size: 11px;
	margin-left: 5px;
}

/* Code Preview */
.bw-schema-code-preview {
	background: #f5f5f5;
	border: 1px solid #ddd;
	border-radius: 5px;
	padding: 15px;
	overflow-x: auto;
	max-width: 100%;
	box-sizing: border-box;
}

.bw-schema-code-preview pre {
	margin: 0;
	font-size: 13px;
	line-height: 1.5;
	white-space: pre-wrap;
	word-wrap: break-word;
	word-break: break-all;
	overflow-wrap: break-word;
}

/* Success Message */
.bw-schema-success-message {
	display: flex;
	align-items: center;
	gap: 10px;
	color: #00a32a;
	font-weight: 500;
}

.bw-schema-success-message .dashicons {
	font-size: 20px;
}

/* Navigation */
.bw-schema-setup-navigation {
	display: flex;
	justify-content: space-between;
	align-items: center;
}

/* Complete Actions */
.bw-schema-complete-actions {
	text-align: center;
	padding: 20px;
	background: #f0f8ff;
	border-radius: 5px;
	margin-top: 20px;
}

.bw-schema-complete-actions .description {
	margin-bottom: 0;
	font-size: 15px;
}

/* Responsive */
@media (max-width: 768px) {
	.bw-schema-field-row {
		flex-direction: column;
	}
	
	.bw-schema-field-half,
	.bw-schema-field-third {
		flex: 1 1 100%;
	}
	
	.bw-schema-page-mapping {
		flex-direction: column;
		align-items: stretch;
		gap: 10px;
	}
	
	.bw-schema-page-select {
		flex: 1;
	}
	
	.bw-schema-progress-steps {
		flex-direction: column;
	}
	
	.bw-schema-progress-steps .step {
		text-align: left;
	}
}
</style>

<script>
jQuery(document).ready(function($) {
	// Business type change handler
	$('#business_type').on('change', function() {
		var businessType = $(this).val();
		var localBusinessTypes = ['professional', 'medical', 'hospitality', 'financial', 'realestate'];
		
		if (localBusinessTypes.includes(businessType)) {
			$('#price-range-container').show();
		} else {
			$('#price-range-container').hide();
		}
	});
	
	// Enable/disable price range
	$('input[name="enable_price_range"]').on('change', function() {
		$('#org_price_range').prop('disabled', !$(this).is(':checked'));
	});
	
	// Founders repeater
	var founderIndex = $('.bw-schema-founder-item').length;
	
	$('#bw-schema-add-founder').on('click', function() {
		var template = $('.bw-schema-founder-item').first().clone();
		
		// Update index
		template.attr('data-index', founderIndex);
		template.find('h4').text('Team Member ' + (founderIndex + 1));
		
		// Clear values and update names
		template.find('input, textarea').each(function() {
			var name = $(this).attr('name');
			if (name) {
				name = name.replace(/\[\d+\]/, '[' + founderIndex + ']');
				$(this).attr('name', name);
				$(this).val('');
			}
		});
		
		// Remove hidden default field
		template.find('input[type="hidden"]').remove();
		
		// Add remove button if not there
		if (!template.find('.bw-schema-remove-founder').length) {
			template.find('.bw-schema-founder-header').append('<button type="button" class="button bw-schema-remove-founder">Remove</button>');
		}
		
		template.appendTo('.bw-schema-founders-repeater');
		founderIndex++;
	});
	
	$(document).on('click', '.bw-schema-remove-founder', function() {
		$(this).closest('.bw-schema-founder-item').remove();
		
		// Reindex
		$('.bw-schema-founder-item').each(function(index) {
			$(this).attr('data-index', index);
			$(this).find('h4').text(index === 0 ? 'Primary Leader (Default Author)' : 'Team Member ' + (index + 1));
			
			$(this).find('input, textarea').each(function() {
				var name = $(this).attr('name');
				if (name) {
					name = name.replace(/\[\d+\]/, '[' + index + ']');
					$(this).attr('name', name);
				}
			});
		});
		
		founderIndex = $('.bw-schema-founder-item').length;
	});
	
	// Social media repeater
	var socialIndex = $('.bw-schema-social-item').length;
	
	$('#bw-schema-add-social').on('click', function() {
		var template = $('.bw-schema-social-item').first().clone();
		
		// Clear values and update names
		template.find('select').val('');
		template.find('input').val('');
		template.find('select, input').each(function() {
			var name = $(this).attr('name');
			if (name) {
				name = name.replace(/\[\d+\]/, '[' + socialIndex + ']');
				$(this).attr('name', name);
			}
		});
		
		template.appendTo('.bw-schema-social-repeater');
		socialIndex++;
	});
	
	$(document).on('click', '.bw-schema-remove-social', function() {
		if ($('.bw-schema-social-item').length > 1) {
			$(this).closest('.bw-schema-social-item').remove();

			// Reindex remaining items
			$('.bw-schema-social-item').each(function(index) {
				$(this).find('select, input').each(function() {
					var name = $(this).attr('name');
					if (name) {
						name = name.replace(/\[\d+\]/, '[' + index + ']');
						$(this).attr('name', name);
					}
				});
			});
			socialIndex = $('.bw-schema-social-item').length;
		}
	});
	
	// Awards repeater
	$('#bw-schema-add-award').on('click', function() {
		var template = '<div class="bw-schema-award-item">' +
			'<input type="text" name="awards[]" value="" placeholder="e.g., Best Workplace 2023" />' +
			' <button type="button" class="button bw-schema-remove-award">Remove</button>' +
			'</div>';
		$('.bw-schema-awards-repeater').append(template);
	});
	
	$(document).on('click', '.bw-schema-remove-award', function() {
		if ($('.bw-schema-award-item').length > 1) {
			$(this).closest('.bw-schema-award-item').remove();
		}
	});
	
	// Certifications repeater
	$('#bw-schema-add-cert').on('click', function() {
		var template = '<div class="bw-schema-cert-item">' +
			'<input type="text" name="certifications[]" value="" placeholder="e.g., ISO 9001:2015" />' +
			' <button type="button" class="button bw-schema-remove-cert">Remove</button>' +
			'</div>';
		$('.bw-schema-certifications-repeater').append(template);
	});
	
	$(document).on('click', '.bw-schema-remove-cert', function() {
		if ($('.bw-schema-cert-item').length > 1) {
			$(this).closest('.bw-schema-cert-item').remove();
		}
	});
	
	// Contact points repeater
	var contactIndex = $('.bw-schema-contact-item').length;
	
	$('#bw-schema-add-contact').on('click', function() {
		var template = $('.bw-schema-contact-item').first().clone();
		
		// Update header
		template.find('h4').text('Contact Point ' + (contactIndex + 1));
		
		// Clear values and update names
		template.find('input, select').each(function() {
			$(this).val('');
			var name = $(this).attr('name');
			if (name) {
				name = name.replace(/\[\d+\]/, '[' + contactIndex + ']');
				$(this).attr('name', name);
			}
		});
		
		// Add remove button if not there
		if (!template.find('.bw-schema-remove-contact').length) {
			template.append('<button type="button" class="button bw-schema-remove-contact">Remove Contact</button>');
		}
		
		template.appendTo('.bw-schema-contact-repeater');
		contactIndex++;
	});
	
	$(document).on('click', '.bw-schema-remove-contact', function() {
		if ($('.bw-schema-contact-item').length > 1) {
			$(this).closest('.bw-schema-contact-item').remove();
			
			// Reindex
			$('.bw-schema-contact-item').each(function(index) {
				$(this).find('h4').text('Contact Point ' + (index + 1));
			});
			contactIndex = $('.bw-schema-contact-item').length;
		}
	});
	
	// Media selector
	$('.bw-schema-media-select').on('click', function(e) {
		e.preventDefault();
		
		var button = $(this);
		var targetInput = $(button.data('target'));
		
		var mediaUploader = wp.media({
			title: 'Select Logo',
			button: {
				text: 'Use this image'
			},
			multiple: false
		});
		
		mediaUploader.on('select', function() {
			var attachment = mediaUploader.state().get('selection').first().toJSON();
			targetInput.val(attachment.url);
		});
		
		mediaUploader.open();
	});
});
</script>