# BW AI Schema Pro - Plugin Documentation

**Version:** 1.0.0
**Author:** Adi Pramono (Bowden Works)
**Last Updated:** January 2026

---

## Table of Contents

1. [Overview](#overview)
2. [Architecture](#architecture)
3. [File Structure](#file-structure)
4. [Core Components](#core-components)
5. [Schema Types Supported](#schema-types-supported)
6. [Author Management System](#author-management-system)
7. [Admin Interface](#admin-interface)
8. [Hooks and Filters](#hooks-and-filters)
9. [Database Storage](#database-storage)
10. [Known Issues & TODOs](#known-issues--todos)
11. [Optimization Opportunities](#optimization-opportunities)

---

## Overview

BW AI Schema Pro is a WordPress plugin designed to generate comprehensive JSON-LD schema markup optimized for AI systems and search engines. The plugin focuses on:

- **AI Optimization**: Schema properties that help AI systems understand content better (knowsAbout, credentials, fact-checking, etc.)
- **Multiple Authors**: Support for multiple authors per post (WordPress users, custom authors, external contributors)
- **Author Box Display**: Frontend display of author information with social links
- **Team Page Detection**: Automatic schema generation for team/staff pages
- **Conflict Management**: Ability to disable schema from other plugins (Yoast, AIOSEO, Rank Math, etc.)
- **Setup Wizard**: 7-step guided configuration process

---

## Architecture

### Design Pattern
- **Singleton Pattern**: Main plugin class (`BW_AI_Schema_Pro`) uses singleton for single instance
- **Autoloader**: SPL autoloader for class loading from `includes/` and `includes/schemas/`
- **Inheritance**: Schema types extend abstract `BW_Schema_Base` class

### Initialization Flow
```
plugins_loaded  BW_AI_Schema_Pro::get_instance()
     init_hooks() (registers all WordPress hooks)
     load_dependencies() (requires class files)
         Core classes
         Schema type classes
         Admin classes (conditional)
         Blocks (Gutenberg)
```

### Key Design Decisions
1. **WordPress Options API**: All settings stored in `wp_options` table (no custom tables)
2. **Post Meta**: Per-post schema settings stored as post meta with `_bw_schema_` prefix
3. **Caching**: Schema output cached via `BW_Schema_Cache` class
4. **Output Location**: Schema injected via `wp_head` hook at priority 1

---

## File Structure

```
bw-ai-schema-pro/
   bw-ai-schema-pro.php          # Main plugin file, entry point
   bw-ai-schema-pro-todo.md      # Improvement plan & changelog
   TEAM-SCHEMA-GUIDE.md          # Team page documentation
   debug-authors.php             # Debug utility for authors
   debug-team-schema.php         # Debug utility for team pages
   test-author-display.php       # Author display testing

   admin/
      assets/
         css/admin.css         # Admin styles (~25KB)
         js/admin.js           # Admin JavaScript (~14KB)
      views/
          dashboard.php         # Main dashboard view
          settings.php          # Global settings page
          author-profiles.php   # Custom author management
          setup-wizard.php      # 7-step setup wizard (~58KB)
          authors.php           # Legacy authors view

   assets/
      css/
         author-box.css        # Frontend author box styles
         frontend.css          # General frontend styles
         frontend-authors.css  # Multiple authors display
      js/
          frontend-authors.js   # JS for author name replacement

   blocks/
      author-schema/            # Gutenberg block for author display
          editor.js
          editor.css
          style.css

   includes/
      class-bw-schema-core.php           # Core utilities & settings (~29KB)
      class-bw-schema-admin.php          # Admin functionality (~54KB)
      class-bw-schema-renderer.php       # Schema output (~19KB)
      class-bw-schema-author-override.php # Author display hooks (~42KB)
      class-bw-schema-author-box.php     # Author box rendering (~17KB)
      class-bw-schema-templates.php      # Template handling (~10KB)
      class-bw-schema-blocks.php         # Gutenberg blocks (~12KB)
      class-bw-schema-cache.php          # Caching system (~3KB)
      class-bw-schema-hooks.php          # Developer hooks (~5KB)
      class-bw-schema-helpers.php        # Helper functions (~11KB)
      class-bw-schema-security.php       # Security features (~11KB)
      class-bw-schema-team-detector.php  # Team page detection (~6KB)
      class-bw-schema-team-member.php    # Team member enhancements (~7KB)

   includes/schemas/
       class-bw-schema-base.php           # Abstract base class (~17KB)
       class-bw-schema-article.php        # Article/BlogPosting (~7KB)
       class-bw-schema-person.php         # Person schema (~15KB)
       class-bw-schema-organization.php   # Organization (~9KB)
       class-bw-schema-localbusiness.php  # LocalBusiness (~17KB)
       class-bw-schema-product.php        # Product (~13KB)
       class-bw-schema-event.php          # Event (~9KB)
       class-bw-schema-howto.php          # HowTo (~13KB)
       class-bw-schema-faq.php            # FAQPage (~7KB)
       class-bw-schema-review.php         # Review (~6KB)
       class-bw-schema-recipe.php         # Recipe (~11KB)
       class-bw-schema-video.php          # VideoObject (~13KB)
       class-bw-schema-course.php         # Course (~14KB)
       class-bw-schema-jobposting.php     # JobPosting (~10KB)
       class-bw-schema-webpage.php        # WebPage variants (~8KB)
```

---

## Core Components

### 1. BW_AI_Schema_Pro (Main Plugin Class)
**File:** `bw-ai-schema-pro.php`

- Entry point for the plugin
- Implements singleton pattern
- Registers all WordPress hooks
- Handles admin menu registration
- Post meta box rendering and saving
- AJAX handlers for preview and simple fields

**Key Methods:**
- `output_schema_markup()` - Main schema output via wp_head
- `add_meta_boxes()` - Registers schema settings meta box
- `save_post_meta()` - Handles saving post-level schema settings
- `maybe_redirect_to_setup()` - Setup wizard redirect logic

### 2. BW_Schema_Core
**File:** `includes/class-bw-schema-core.php`

Central utility class providing:
- Schema type definitions (`get_schema_types()`)
- AI-optimized property definitions (`get_ai_properties()`)
- Organization data retrieval (`get_organization_schema()`)
- Setup completion tracking (`get_setup_completion_status()`)
- Consolidated settings system with migration support
- Schema source conflict detection and disabling

**Settings System:**
```php
// New consolidated approach (v2.0.0)
BW_Schema_Core::get_settings('features', 'enable_schema');
BW_Schema_Core::update_settings('display', 'metabox_position', 'side');

// Legacy support maintained for backward compatibility
get_option('bw_schema_enable_schema'); // Still works
```

### 3. BW_Schema_Renderer
**File:** `includes/class-bw-schema-renderer.php`

Handles all schema output:
- Determines context (singular, archive, author, team page)
- Generates WebSite/Organization schema
- Generates BreadcrumbList schema
- Delegates to specific schema type classes
- Caches generated schema

**Output Structure:**
```html
<!-- BW AI Schema Pro - Structured Data for AI Era -->
<script type="application/ld+json">
{ "@context": "https://schema.org", "@type": "WebSite", ... }
</script>
<script type="application/ld+json">
{ "@context": "https://schema.org", "@type": "Article", ... }
</script>
<!-- / BW AI Schema Pro -->
```

### 4. BW_Schema_Base (Abstract)
**File:** `includes/schemas/class-bw-schema-base.php`

Base class for all schema types:
- `get_common_properties()` - Name, URL, dates, description, image
- `get_multiple_authors_properties()` - Multiple author support
- `get_author_properties()` - Single author with fallbacks
- `get_organization_properties()` - Publisher/organization data
- `add_ai_properties()` - AI-optimized properties (keywords, about, mentions)
- `get_breadcrumb_list()` - Breadcrumb generation

### 5. BW_Schema_Admin
**File:** `includes/class-bw-schema-admin.php`

Admin functionality:
- Menu page registration (under Settings)
- Dashboard rendering
- Settings save handlers
- Setup wizard step handling
- Custom author CRUD operations
- Post meta box rendering

### 6. BW_Schema_Author_Override
**File:** `includes/class-bw-schema-author-override.php`

Comprehensive author display override system:
- Filters for `the_author`, `get_the_author_display_name`
- Avatar filtering
- Author URL filtering
- Kadence theme specific hooks
- Yoast SEO schema integration
- Output buffering for stubborn themes
- JavaScript fallback for client-side author name replacement

### 7. BW_Schema_Author_Box
**File:** `includes/class-bw-schema-author-box.php`

Frontend author box display:
- Automatic insertion via `the_content` filter
- Shortcode support: `[bw_author_box]`
- Multiple author support
- Custom author and external author rendering

---

## Schema Types Supported

| Type | Class | Description |
|------|-------|-------------|
| **Article** | `BW_Schema_Article` | BlogPosting, NewsArticle, TechArticle, ScholarlyArticle |
| **Person** | `BW_Schema_Person` | Team members, author profiles |
| **Organization** | `BW_Schema_Organization` | Company pages, NGOs |
| **LocalBusiness** | `BW_Schema_LocalBusiness` | ProfessionalService, Restaurant, Hotel, etc. |
| **Product** | `BW_Schema_Product` | Products, books, software |
| **Event** | `BW_Schema_Event` | Business, music, sports events |
| **HowTo** | `BW_Schema_HowTo` | Tutorials, guides with steps |
| **FAQ** | `BW_Schema_FAQ` | FAQPage with Q&A pairs |
| **Review** | `BW_Schema_Review` | Reviews with ratings |
| **Recipe** | `BW_Schema_Recipe` | Cooking recipes |
| **Video** | `BW_Schema_Video` | VideoObject, Movie, TVEpisode |
| **Course** | `BW_Schema_Course` | Educational courses |
| **JobPosting** | `BW_Schema_JobPosting` | Job listings |
| **WebPage** | `BW_Schema_WebPage` | AboutPage, ContactPage, ProfilePage |

---

## Author Management System

### Three Author Types

1. **WordPress Users**
   - Standard WordPress user accounts
   - Uses display name, avatar, bio from user profile
   - Links to author archive page

2. **Custom Authors**
   - Stored in `bw_schema_custom_authors` option
   - Full profile: name, job title, credentials, bio, image, social links
   - Can set default author for all posts
   - Links to internal team page or external URL

3. **External Authors**
   - One-time authors defined per-post
   - Name, job title, bio, image, social links
   - For guest contributors without system accounts

### Multiple Authors Per Post
Posts can have multiple authors of any type combination. Stored in post meta `_bw_schema_multiple_authors` as array:

```php
[
    ['type' => 'wordpress', 'wordpress_user_id' => 1],
    ['type' => 'custom', 'custom_author_id' => 'john-smith_123456'],
    ['type' => 'external', 'external' => ['name' => 'Guest', 'bio' => '...']]
]
```

### Author Display Formatting
- 1 author: "John Smith"
- 2 authors: "John Smith and Jane Doe"
- 3 authors: "John Smith, Jane Doe and Bob Wilson"
- 4-5 authors: Full comma-separated list with "and" before last
- 6+ authors: "John Smith, Jane Doe, Bob Wilson and 3 others"

---

## Admin Interface

### Menu Structure
All pages under **Settings > AI Schema Pro**:

| Page | Slug | Purpose |
|------|------|---------|
| Dashboard | `bw-ai-schema` | Overview, completion status, quick config |
| Settings | `bw-ai-schema-settings` | Advanced organization settings, author box |
| Author Profiles | `bw-ai-schema-authors` | Manage custom authors |
| Setup Wizard | `bw-ai-schema-setup` | 7-step guided configuration |

### Setup Wizard Steps
1. **Organization Profile** - Name, URL, logo, type, founding date
2. **Leadership & Team** - Founders, default authors, awards
3. **Contact & Social** - Address, contact points, social profiles
4. **Page Mapping** - Map key pages (about, contact, services)
5. **Content Defaults** - Default schema types per post type
6. **Schema Features** - Enable/disable features, conflict management
7. **Review & Complete** - Final review and completion

### Post Meta Box
Collapsible meta box with:
- **Simple Mode** (default): Schema type dropdown, AI fields, author selection
- **Advanced Mode**: Custom JSON-LD editor
- Schema preview and validation links

---

## Hooks and Filters

### Filters for Customization

```php
// Modify schema types available
add_filter('bw_schema_types', function($types) {
    // Add custom schema types
    return $types;
});

// Modify AI properties
add_filter('bw_schema_ai_properties', function($properties) {
    return $properties;
});

// Modify generated schema before output
add_filter('bw_schema_rendered_schemas', function($schemas) {
    return $schemas;
});

// Modify specific schema types
add_filter('bw_schema_article', function($schema, $post, $type) {
    return $schema;
}, 10, 3);

// Team page URL patterns
add_filter('bw_schema_team_url_patterns', function($patterns) {
    $patterns[] = '/notre-equipe'; // French
    return $patterns;
});

// Team page shortcodes
add_filter('bw_schema_team_shortcodes', function($shortcodes) {
    $shortcodes[] = 'my_custom_team_grid';
    return $shortcodes;
});
```

### Actions
```php
// After schema is rendered
do_action('bw_schema_after_render');

// Before schema is generated for a post
do_action('bw_schema_before_generate', $post);
```

---

## Database Storage

### Options (wp_options)

| Option Key | Description |
|------------|-------------|
| `bw_schema_version` | Plugin version |
| `bw_schema_setup_complete` | Boolean - wizard completed |
| `bw_schema_organization` | Array - org name, URL, logo, sameAs |
| `bw_schema_business_type` | String - business category |
| `bw_schema_founders` | Array - founder/leadership data |
| `bw_schema_custom_authors` | Array - custom author profiles |
| `bw_schema_use_custom_authors` | 'wordpress' or 'custom' |
| `bw_schema_post_type_defaults` | Array - default schemas per post type |
| `bw_schema_page_mappings` | Array - page type to post ID map |
| `bw_schema_enable_schema` | 'yes'/'no' |
| `bw_schema_enable_breadcrumbs` | 'yes'/'no' |
| `bw_schema_disable_sources` | Array - disabled schema sources |
| `bw_schema_settings` | Consolidated settings (v2.0.0+) |

### Post Meta

| Meta Key | Description |
|----------|-------------|
| `_bw_schema_type` | Selected schema type (e.g., 'article:BlogPosting') |
| `_bw_schema_disable` | 'yes' to disable schema for post |
| `_bw_schema_custom` | Custom JSON-LD override |
| `_bw_schema_simple_mode` | 'yes'/'no' - editor mode |
| `_bw_schema_multiple_authors` | Array - multiple author data |
| `_bw_schema_show_author_box` | 'yes'/'no' |
| `_bw_schema_disable_default_author` | '1' to hide theme author box |
| `_bw_schema_is_team_page` | 'yes' - manual team page designation |
| `_bw_schema_about_entities` | Array - main topic entities |
| `_bw_schema_fact_checked_by` | String - fact checker name |
| `_bw_schema_last_reviewed` | Date - last review |

---

## Known Issues & TODOs

### From bw-ai-schema-pro-todo.md

**Completed (Phase 1-5):**
- Collapsible meta box sections
- Setup completion checklist
- Meta box visibility controls per post type
- Option consolidation with migration
- Menu relocated to Settings submenu

**Pending:**
- PHPDoc blocks for all methods
- Unit tests for core functionality
- Translation review
- Performance optimization for large sites

### Identified Issues

1. **Debug Logging**: Many `error_log()` calls in `BW_Schema_Author_Override` - should be conditional on WP_DEBUG
2. **Output Buffering**: Used for stubborn themes - performance impact on large pages
3. **JavaScript Fallback**: Client-side author name replacement adds overhead
4. **Option Proliferation**: ~30+ individual options despite consolidation effort
5. **Large Files**: `setup-wizard.php` (58KB), `admin.css` (25KB) could be optimized

---

## Optimization Opportunities

### Performance
1. **Lazy Loading**: Load admin classes only when needed
2. **Conditional Assets**: Only enqueue CSS/JS on relevant pages
3. **Schema Caching**: Currently basic - could use transients with smarter invalidation
4. **Output Buffering**: Remove or make conditional for author override

### Code Quality
1. **Extract Methods**: Some methods in `BW_Schema_Admin` exceed 100 lines
2. **Reduce Duplication**: Author name formatting repeated in multiple places
3. **Type Hints**: Add PHP 7+ type hints to methods
4. **Constants**: Magic strings could be constants (e.g., author types)

### UX
1. **Inline Help**: More contextual help in admin interfaces
2. **Validation Feedback**: Real-time schema validation
3. **Import/Export**: Settings migration between sites
4. **Bulk Operations**: Bulk update schema types for posts

### Security
1. **Nonce Verification**: Already implemented, good
2. **Capability Checks**: Already implemented, good
3. **Sanitization**: Generally good, some edge cases could be reviewed
4. **Escaping**: Output escaping consistent, good

---

## Summary

BW AI Schema Pro is a well-structured WordPress plugin with comprehensive schema markup functionality. Key strengths include:

- **Comprehensive schema support** for 14+ schema types
- **AI optimization** focus with specialized properties
- **Flexible author system** supporting multiple authors
- **Conflict management** with major SEO plugins
- **Clean architecture** with separation of concerns

Areas for improvement:
- Debug code cleanup
- Performance optimization
- Additional testing
- Documentation completion (PHPDoc)

The plugin is production-ready with the existing TODO list (from `bw-ai-schema-pro-todo.md`) showing phases 1-5 completed. Further optimization and testing would enhance long-term maintainability.
