<?php
/**
 * Article Schema Class for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Article extends BW_Schema_Base {
	
	/**
	 * Generate article schema
	 */
	public function generate( $post, $type = 'Article' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema = array_merge( $schema, $this->get_common_properties( $post ) );
		
		// Add headline (optimized for AI)
		$schema['headline'] = wp_trim_words( get_the_title( $post ), 20 );
		
		// Add authors (supporting multiple authors)
		$authors_schema = $this->get_multiple_authors_properties( $post );
		if ( ! empty( $authors_schema ) ) {
			if ( count( $authors_schema ) === 1 ) {
				$schema['author'] = $authors_schema[0];
			} else {
				$schema['author'] = $authors_schema;
			}
		}
		
		// Add publisher
		$schema['publisher'] = $this->get_organization_properties();
		
		// Add article body
		$content = apply_filters( 'the_content', $post->post_content );
		$schema['articleBody'] = wp_strip_all_tags( $content );
		
		// Add word count (helpful for AI)
		$schema['wordCount'] = str_word_count( wp_strip_all_tags( $content ) );
		
		// Add article section (categories)
		$categories = get_the_category( $post->ID );
		if ( ! empty( $categories ) ) {
			$sections = array();
			foreach ( $categories as $category ) {
				$sections[] = $category->name;
			}
			$schema['articleSection'] = implode( ', ', $sections );
		}
		
		// Add in language
		$schema['inLanguage'] = get_locale();
		
		// Add potential action (read action)
		$schema['potentialAction'] = array(
			'@type' => 'ReadAction',
			'target' => get_permalink( $post ),
		);
		
		// Add main entity of page (for better AI understanding)
		$schema['mainEntityOfPage'] = array(
			'@type' => 'WebPage',
			'@id' => get_permalink( $post ),
		);
		
		// Note: breadcrumb is typically a separate schema, not a property of Article
		// We'll include it in the renderer as a separate schema item
		
		// Add comments info if comments are open
		if ( comments_open( $post ) ) {
			$comment_count = get_comments_number( $post );
			if ( $comment_count > 0 ) {
				$schema['commentCount'] = $comment_count;
				$schema['discussionUrl'] = get_permalink( $post ) . '#comments';
			}
		}
		
		// Add AI-optimized properties
		$schema = $this->add_ai_properties( $schema, $post );
		
		// Add specific properties based on article type
		switch ( $type ) {
			case 'NewsArticle':
				$schema = $this->add_news_article_properties( $schema, $post );
				break;
				
			case 'BlogPosting':
				$schema = $this->add_blog_posting_properties( $schema, $post );
				break;
				
			case 'TechArticle':
				$schema = $this->add_tech_article_properties( $schema, $post );
				break;
				
			case 'ScholarlyArticle':
				$schema = $this->add_scholarly_article_properties( $schema, $post );
				break;
		}
		
		return apply_filters( 'bw_schema_article', $schema, $post, $type );
	}
	
	/**
	 * Add news article specific properties
	 */
	private function add_news_article_properties( $schema, $post ) {
		// Add dateline
		$schema['dateline'] = get_the_date( 'F j, Y', $post );
		
		// Add news specific keywords
		$schema['keywords'] = $this->get_news_keywords( $post );
		
		// Add source organization
		$source_org = get_post_meta( $post->ID, '_bw_schema_source_organization', true );
		if ( ! empty( $source_org ) ) {
			$schema['sourceOrganization'] = array(
				'@type' => 'Organization',
				'name' => $source_org,
			);
		}
		
		return $schema;
	}
	
	/**
	 * Add blog posting specific properties
	 */
	private function add_blog_posting_properties( $schema, $post ) {
		// Add interaction statistics
		$schema['interactionStatistic'] = array(
			array(
				'@type' => 'InteractionCounter',
				'interactionType' => 'https://schema.org/CommentAction',
				'userInteractionCount' => get_comments_number( $post ),
			),
		);
		
		// Add sharing URLs
		$schema['sharedContent'] = array(
			'@type' => 'WebPage',
			'url' => get_permalink( $post ),
		);
		
		return $schema;
	}
	
	/**
	 * Add tech article specific properties
	 */
	private function add_tech_article_properties( $schema, $post ) {
		// Add dependencies/requirements
		$dependencies = get_post_meta( $post->ID, '_bw_schema_dependencies', true );
		if ( ! empty( $dependencies ) ) {
			$schema['dependencies'] = $dependencies;
		}
		
		// Add proficiency level
		$proficiency = get_post_meta( $post->ID, '_bw_schema_proficiency_level', true );
		if ( ! empty( $proficiency ) ) {
			$schema['proficiencyLevel'] = $proficiency;
		}
		
		// Add programming language if applicable
		$programming_lang = get_post_meta( $post->ID, '_bw_schema_programming_language', true );
		if ( ! empty( $programming_lang ) ) {
			$schema['programmingLanguage'] = $programming_lang;
		}
		
		return $schema;
	}
	
	/**
	 * Add scholarly article specific properties
	 */
	private function add_scholarly_article_properties( $schema, $post ) {
		// Add citation
		$citations = get_post_meta( $post->ID, '_bw_schema_citations', true );
		if ( ! empty( $citations ) && is_array( $citations ) ) {
			$citation_list = array();
			foreach ( $citations as $citation ) {
				$citation_list[] = array(
					'@type' => 'CreativeWork',
					'name' => $citation,
				);
			}
			$schema['citation'] = $citation_list;
		}
		
		// Add academic discipline
		$discipline = get_post_meta( $post->ID, '_bw_schema_academic_discipline', true );
		if ( ! empty( $discipline ) ) {
			$schema['about'] = array(
				'@type' => 'Thing',
				'name' => $discipline,
			);
		}
		
		// Add peer review status
		$peer_reviewed = get_post_meta( $post->ID, '_bw_schema_peer_reviewed', true );
		if ( $peer_reviewed === 'yes' ) {
			$schema['peerReviewed'] = true;
		}
		
		return $schema;
	}
	
	/**
	 * Get news keywords
	 */
	private function get_news_keywords( $post ) {
		$keywords = array();
		
		// Get tags
		$tags = wp_get_post_tags( $post->ID );
		foreach ( $tags as $tag ) {
			$keywords[] = $tag->name;
		}
		
		// Get categories
		$categories = get_the_category( $post->ID );
		foreach ( $categories as $category ) {
			$keywords[] = $category->name;
		}
		
		// Get custom news keywords
		$custom_keywords = get_post_meta( $post->ID, '_bw_schema_news_keywords', true );
		if ( ! empty( $custom_keywords ) ) {
			$custom_array = explode( ',', $custom_keywords );
			$keywords = array_merge( $keywords, array_map( 'trim', $custom_array ) );
		}
		
		return implode( ', ', array_unique( $keywords ) );
	}
}