<?php
/**
 * Team Member Schema Enhancement for BW AI Schema Pro
 *
 * Provides meta box and schema enhancement for team member post types.
 * Part of v2.0 Unified People/Author Architecture.
 *
 * @package BW_AI_Schema_Pro
 * @since 1.0.0
 * @since 2.0.0 Added Is Author, Is Leader flags and enhanced Person schema fields
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Team_Member {

	/**
	 * Get the configured team member post type
	 *
	 * @return string|false Post type slug or false if not configured
	 */
	public static function get_team_post_type() {
		$post_type = BW_Schema_Core::get_settings( 'content', 'team_post_type', '' );

		// Fallback to legacy option
		if ( empty( $post_type ) ) {
			$post_type = get_option( 'bw_schema_team_post_type', '' );
		}

		// Verify post type exists
		if ( ! empty( $post_type ) && post_type_exists( $post_type ) ) {
			return $post_type;
		}

		return false;
	}

	/**
	 * Initialize team member enhancements
	 */
	public static function init() {
		// Hook into schema generation
		add_filter( 'bw_schema_rendered_schemas', array( __CLASS__, 'filter_team_member_schemas' ), 20 );

		// Add custom meta box for team members
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_team_member_meta_box' ), 15 );

		// Save team member schema data - dynamic hook based on configured post type
		add_action( 'save_post', array( __CLASS__, 'save_team_member_schema' ), 10, 2 );

		// Disable other plugin schemas on team member pages
		add_action( 'wp', array( __CLASS__, 'disable_conflicting_schemas' ), 5 );

		// Enqueue admin styles for the meta box
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_admin_styles' ) );
	}

	/**
	 * Enqueue admin styles for team member meta box
	 */
	public static function enqueue_admin_styles( $hook ) {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type ) {
			return;
		}

		$screen = get_current_screen();
		if ( $screen && $screen->post_type === $team_post_type && in_array( $hook, array( 'post.php', 'post-new.php' ) ) ) {
			wp_add_inline_style( 'wp-admin', self::get_meta_box_styles() );
		}
	}

	/**
	 * Get inline styles for meta box
	 */
	private static function get_meta_box_styles() {
		return '
			.bw-schema-team-member-fields .bw-schema-section {
				margin-bottom: 20px;
				padding-bottom: 20px;
				border-bottom: 1px solid #ddd;
			}
			.bw-schema-team-member-fields .bw-schema-section:last-child {
				border-bottom: none;
				margin-bottom: 0;
				padding-bottom: 0;
			}
			.bw-schema-team-member-fields .bw-schema-section-title {
				font-size: 14px;
				font-weight: 600;
				margin: 0 0 12px 0;
				color: #1d2327;
			}
			.bw-schema-team-member-fields .bw-schema-checkbox-row {
				display: flex;
				gap: 24px;
				margin-bottom: 12px;
			}
			.bw-schema-team-member-fields .bw-schema-checkbox-row label {
				display: flex;
				align-items: center;
				gap: 6px;
				cursor: pointer;
			}
			.bw-schema-team-member-fields .bw-schema-field {
				margin-bottom: 12px;
			}
			.bw-schema-team-member-fields .bw-schema-field label {
				display: block;
				font-weight: 500;
				margin-bottom: 4px;
			}
			.bw-schema-team-member-fields .bw-schema-field .description {
				color: #646970;
				font-size: 12px;
				margin-top: 4px;
			}
			.bw-schema-team-member-fields .bw-schema-repeater {
				background: #f9f9f9;
				padding: 12px;
				border-radius: 4px;
			}
			.bw-schema-team-member-fields .bw-schema-repeater-item {
				display: flex;
				gap: 8px;
				margin-bottom: 8px;
				align-items: center;
			}
			.bw-schema-team-member-fields .bw-schema-repeater-item:last-child {
				margin-bottom: 0;
			}
			.bw-schema-team-member-fields .bw-schema-repeater-item input {
				flex: 1;
			}
			.bw-schema-team-member-fields .bw-schema-add-item {
				margin-top: 8px;
			}
			.bw-schema-team-member-fields .bw-schema-remove-item {
				color: #b32d2e;
				cursor: pointer;
				padding: 4px;
			}
			.bw-schema-team-member-fields .bw-schema-remove-item:hover {
				color: #a00;
			}
			/* Core fields section */
			.bw-schema-team-member-fields .bw-schema-core-field {
				margin-bottom: 16px;
				padding: 12px;
				background: #f9f9f9;
				border-radius: 4px;
				border-left: 3px solid #2271b1;
			}
			.bw-schema-team-member-fields .bw-schema-core-field:last-child {
				margin-bottom: 0;
			}
			.bw-schema-team-member-fields .bw-schema-core-field-header {
				display: flex;
				justify-content: space-between;
				align-items: center;
				margin-bottom: 8px;
				flex-wrap: wrap;
				gap: 8px;
			}
			.bw-schema-team-member-fields .bw-schema-core-field-header label {
				margin: 0;
			}
			.bw-schema-team-member-fields .bw-schema-detected-value {
				color: #50575e;
				font-size: 13px;
				display: flex;
				align-items: center;
				gap: 4px;
			}
			.bw-schema-team-member-fields .bw-schema-source {
				color: #787c82;
				font-size: 12px;
				font-style: italic;
			}
			.bw-schema-team-member-fields .bw-schema-not-detected {
				color: #996800;
			}
			.bw-schema-team-member-fields .bw-schema-override-row input,
			.bw-schema-team-member-fields .bw-schema-override-row textarea {
				background: #fff;
			}
			.bw-schema-team-member-fields .bw-schema-override-row input::placeholder,
			.bw-schema-team-member-fields .bw-schema-override-row textarea::placeholder {
				color: #a7aaad;
				font-style: italic;
			}
		';
	}

	/**
	 * Filter schemas for team member pages
	 */
	public static function filter_team_member_schemas( $schemas ) {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type || ! is_singular( $team_post_type ) ) {
			return $schemas;
		}

		// Remove any Blog or WebSite schemas
		$filtered_schemas = array();
		foreach ( $schemas as $schema ) {
			if ( isset( $schema['@type'] ) ) {
				$type = $schema['@type'];
				// Skip Blog, WebSite, and CollectionPage schemas
				if ( in_array( $type, array( 'Blog', 'WebSite', 'CollectionPage' ) ) ) {
					continue;
				}
			}
			$filtered_schemas[] = $schema;
		}

		return $filtered_schemas;
	}

	/**
	 * Add meta box for team member schema fields
	 */
	public static function add_team_member_meta_box() {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type ) {
			return;
		}

		add_meta_box(
			'bw_schema_team_member_fields',
			__( 'Person Schema Settings', 'bw-ai-schema-pro' ),
			array( __CLASS__, 'render_meta_box' ),
			$team_post_type,
			'normal',
			'default'
		);
	}

	/**
	 * Detect job title value and its source
	 *
	 * @param int $post_id Post ID
	 * @return array ['value' => string, 'source' => string, 'meta_key' => string]
	 */
	public static function detect_job_title( $post_id ) {
		// Check for configured custom meta key first
		$custom_key = get_option( 'bw_schema_job_title_meta_key', '' );
		if ( ! empty( $custom_key ) ) {
			$value = get_post_meta( $post_id, $custom_key, true );
			if ( ! empty( $value ) ) {
				return array(
					'value'    => $value,
					'source'   => sprintf( __( 'Custom field: %s', 'bw-ai-schema-pro' ), $custom_key ),
					'meta_key' => $custom_key,
				);
			}
		}

		// Try common meta keys
		$common_keys = array(
			'job_title'       => __( 'job_title', 'bw-ai-schema-pro' ),
			'_job_title'      => __( '_job_title', 'bw-ai-schema-pro' ),
			'position'        => __( 'position', 'bw-ai-schema-pro' ),
			'title'           => __( 'title', 'bw-ai-schema-pro' ),
			'role'            => __( 'role', 'bw-ai-schema-pro' ),
			'job-title'       => __( 'job-title', 'bw-ai-schema-pro' ),
			'jobtitle'        => __( 'jobtitle', 'bw-ai-schema-pro' ),
			'team_position'   => __( 'team_position', 'bw-ai-schema-pro' ),
			'member_position' => __( 'member_position', 'bw-ai-schema-pro' ),
		);

		foreach ( $common_keys as $key => $label ) {
			$value = get_post_meta( $post_id, $key, true );
			if ( ! empty( $value ) ) {
				return array(
					'value'    => $value,
					'source'   => sprintf( __( 'Detected from: %s', 'bw-ai-schema-pro' ), $key ),
					'meta_key' => $key,
				);
			}
		}

		return array(
			'value'    => '',
			'source'   => __( 'Not detected', 'bw-ai-schema-pro' ),
			'meta_key' => '',
		);
	}

	/**
	 * Render team member meta box
	 */
	public static function render_meta_box( $post ) {
		wp_nonce_field( 'bw_schema_team_member', 'bw_schema_team_member_nonce' );

		// Get existing values
		$is_author = get_post_meta( $post->ID, '_bw_schema_is_author', true );
		$is_leader = get_post_meta( $post->ID, '_bw_schema_is_leader', true );
		$linked_user = get_post_meta( $post->ID, '_bw_schema_linked_user', true );
		$credentials = get_post_meta( $post->ID, '_bw_schema_credentials', true );
		// Ensure credentials is an array (migrate from old string format)
		if ( ! is_array( $credentials ) ) {
			$credentials = array();
		}
		if ( empty( $credentials ) ) {
			$credentials = array( array( 'category' => '', 'level' => '', 'school' => '', 'description' => '' ) );
		}
		$video = get_post_meta( $post->ID, '_bw_schema_video', true );
		if ( ! is_array( $video ) ) {
			$video = array();
		}
		$knows_about = get_post_meta( $post->ID, '_bw_schema_knows_about', true );
		$same_as = get_post_meta( $post->ID, '_bw_schema_same_as', true );
		$awards = get_post_meta( $post->ID, '_bw_schema_awards', true );
		$alumni_of = get_post_meta( $post->ID, '_bw_schema_alumni_of', true );
		$member_of = get_post_meta( $post->ID, '_bw_schema_member_of', true );

		// Get overrides
		$override_name = get_post_meta( $post->ID, '_bw_schema_override_name', true );
		$override_job_title = get_post_meta( $post->ID, '_bw_schema_override_job_title', true );
		$override_description = get_post_meta( $post->ID, '_bw_schema_override_description', true );
		$override_image = get_post_meta( $post->ID, '_bw_schema_override_image', true );
		$override_url = get_post_meta( $post->ID, '_bw_schema_override_url', true );

		// Detect current values
		$detected_name = $post->post_title;
		$detected_job_title = self::detect_job_title( $post->ID );
		$detected_description = get_the_excerpt( $post );
		$detected_image_id = get_post_thumbnail_id( $post->ID );
		$detected_image_url = $detected_image_id ? wp_get_attachment_image_url( $detected_image_id, 'thumbnail' ) : '';
		$detected_url = get_permalink( $post );

		// Legacy support - migrate old fields
		if ( empty( $knows_about ) ) {
			$legacy_expertise = get_post_meta( $post->ID, '_bw_schema_expertise', true );
			if ( ! empty( $legacy_expertise ) ) {
				$knows_about = is_array( $legacy_expertise ) ? $legacy_expertise : array_map( 'trim', explode( ',', $legacy_expertise ) );
			}
		}

		// Ensure arrays
		$same_as = ! empty( $same_as ) && is_array( $same_as ) ? $same_as : array( '' );
		$awards = ! empty( $awards ) && is_array( $awards ) ? $awards : array( '' );
		$alumni_of = ! empty( $alumni_of ) && is_array( $alumni_of ) ? $alumni_of : array( '' );
		$member_of = ! empty( $member_of ) && is_array( $member_of ) ? $member_of : array( '' );

		// Migrate legacy social profiles to sameAs
		if ( count( $same_as ) === 1 && empty( $same_as[0] ) ) {
			$legacy_linkedin = get_post_meta( $post->ID, '_bw_schema_linkedin', true );
			$legacy_twitter = get_post_meta( $post->ID, '_bw_schema_twitter', true );
			if ( ! empty( $legacy_linkedin ) || ! empty( $legacy_twitter ) ) {
				$same_as = array_filter( array( $legacy_linkedin, $legacy_twitter ) );
				if ( empty( $same_as ) ) {
					$same_as = array( '' );
				}
			}
		}

		?>
		<div class="bw-schema-team-member-fields">

			<!-- Core Fields Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Core Person Fields', 'bw-ai-schema-pro' ); ?></h4>
				<p class="description" style="margin-bottom: 12px;">
					<?php _e( 'These fields are auto-detected from your team member post. Use the override fields to customize schema output.', 'bw-ai-schema-pro' ); ?>
				</p>

				<!-- Name -->
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'Name', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php echo esc_html( $detected_name ); ?>
							<span class="bw-schema-source"><?php _e( '(Post Title)', 'bw-ai-schema-pro' ); ?></span>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<input type="text" name="bw_schema_override_name" value="<?php echo esc_attr( $override_name ); ?>" placeholder="<?php esc_attr_e( 'Override name...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>

				<!-- Job Title -->
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'Job Title', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php if ( ! empty( $detected_job_title['value'] ) ) : ?>
								<?php echo esc_html( $detected_job_title['value'] ); ?>
								<span class="bw-schema-source">(<?php echo esc_html( $detected_job_title['source'] ); ?>)</span>
							<?php else : ?>
								<em class="bw-schema-not-detected"><?php _e( 'Not detected', 'bw-ai-schema-pro' ); ?></em>
							<?php endif; ?>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<input type="text" name="bw_schema_override_job_title" value="<?php echo esc_attr( $override_job_title ); ?>" placeholder="<?php esc_attr_e( 'Override or set job title...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>

				<!-- Description -->
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'Description', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php if ( ! empty( $detected_description ) ) : ?>
								<?php echo esc_html( wp_trim_words( $detected_description, 10, '...' ) ); ?>
								<span class="bw-schema-source"><?php _e( '(Excerpt)', 'bw-ai-schema-pro' ); ?></span>
							<?php else : ?>
								<em class="bw-schema-not-detected"><?php _e( 'No excerpt set', 'bw-ai-schema-pro' ); ?></em>
							<?php endif; ?>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<textarea name="bw_schema_override_description" rows="2" placeholder="<?php esc_attr_e( 'Override description...', 'bw-ai-schema-pro' ); ?>" class="widefat"><?php echo esc_textarea( $override_description ); ?></textarea>
					</div>
				</div>

				<!-- Image -->
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'Image', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php if ( ! empty( $detected_image_url ) ) : ?>
								<img src="<?php echo esc_url( $detected_image_url ); ?>" alt="" style="width: 40px; height: 40px; object-fit: cover; border-radius: 4px; vertical-align: middle; margin-right: 8px;" />
								<span class="bw-schema-source"><?php _e( '(Featured Image)', 'bw-ai-schema-pro' ); ?></span>
							<?php else : ?>
								<em class="bw-schema-not-detected"><?php _e( 'No featured image', 'bw-ai-schema-pro' ); ?></em>
							<?php endif; ?>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<input type="url" name="bw_schema_override_image" value="<?php echo esc_url( $override_image ); ?>" placeholder="<?php esc_attr_e( 'Override image URL...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>

				<!-- URL -->
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'URL', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php echo esc_html( $detected_url ); ?>
							<span class="bw-schema-source"><?php _e( '(Permalink)', 'bw-ai-schema-pro' ); ?></span>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<input type="url" name="bw_schema_override_url" value="<?php echo esc_url( $override_url ); ?>" placeholder="<?php esc_attr_e( 'Override URL...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>

				<!-- Organization (worksFor) -->
				<?php
				$org_data = BW_Schema_Core::get_organization_schema();
				$site_org_name = ! empty( $org_data['name'] ) ? $org_data['name'] : get_bloginfo( 'name' );
				$override_org = get_post_meta( $post->ID, '_bw_schema_override_organization', true );
				?>
				<div class="bw-schema-core-field">
					<div class="bw-schema-core-field-header">
						<label><strong><?php _e( 'Organization (worksFor)', 'bw-ai-schema-pro' ); ?></strong></label>
						<span class="bw-schema-detected-value">
							<?php echo esc_html( $site_org_name ); ?>
							<span class="bw-schema-source"><?php _e( '(Site Organization)', 'bw-ai-schema-pro' ); ?></span>
						</span>
					</div>
					<div class="bw-schema-override-row">
						<input type="text" name="bw_schema_override_organization" value="<?php echo esc_attr( $override_org ); ?>" placeholder="<?php esc_attr_e( 'Override with different organization...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
					<p class="description" style="margin-top: 4px; margin-left: 0;">
						<?php _e( 'Leave blank to use site organization. Enter a different organization name if this person works elsewhere.', 'bw-ai-schema-pro' ); ?>
					</p>
				</div>
			</div>

			<!-- Author Settings Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Author Settings', 'bw-ai-schema-pro' ); ?></h4>

				<div class="bw-schema-checkbox-row">
					<label>
						<input type="checkbox" name="bw_schema_is_author" value="1" <?php checked( $is_author, '1' ); ?> />
						<strong><?php _e( 'Is Author', 'bw-ai-schema-pro' ); ?></strong>
					</label>
					<label>
						<input type="checkbox" name="bw_schema_is_leader" value="1" <?php checked( $is_leader, '1' ); ?> />
						<strong><?php _e( 'Is Founder', 'bw-ai-schema-pro' ); ?></strong>
					</label>
				</div>
				<p class="description">
					<?php _e( 'Authors appear in post author selection. Founders appear in Organization schema "founder" property.', 'bw-ai-schema-pro' ); ?>
				</p>

				<div class="bw-schema-field" style="margin-top: 12px;">
					<label for="bw_schema_linked_user"><?php _e( 'Linked WordPress User', 'bw-ai-schema-pro' ); ?></label>
					<?php
					wp_dropdown_users( array(
						'name'              => 'bw_schema_linked_user',
						'id'                => 'bw_schema_linked_user',
						'selected'          => $linked_user,
						'show_option_none'  => __( '— No linked user —', 'bw-ai-schema-pro' ),
						'option_none_value' => '',
					) );
					?>
					<p class="description">
						<?php _e( 'Link to a WordPress user account. Posts by this user will default to this team member as author.', 'bw-ai-schema-pro' ); ?>
					</p>
				</div>
			</div>

			<!-- Person Schema Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Person Schema Fields', 'bw-ai-schema-pro' ); ?></h4>

				<div class="bw-schema-field-row" style="display: flex; gap: 16px;">
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_honorific_prefix"><?php _e( 'Honorific Prefix', 'bw-ai-schema-pro' ); ?></label>
						<input type="text" id="bw_schema_honorific_prefix" name="bw_schema_honorific_prefix" value="<?php echo esc_attr( get_post_meta( $post->ID, '_bw_schema_honorific_prefix', true ) ); ?>" placeholder="<?php esc_attr_e( 'Dr., Prof., Mr., Ms.', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_honorific_suffix"><?php _e( 'Honorific Suffix', 'bw-ai-schema-pro' ); ?></label>
						<input type="text" id="bw_schema_honorific_suffix" name="bw_schema_honorific_suffix" value="<?php echo esc_attr( get_post_meta( $post->ID, '_bw_schema_honorific_suffix', true ) ); ?>" placeholder="<?php esc_attr_e( 'Jr., III, Esq.', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>

				<div class="bw-schema-field">
					<label for="bw_schema_knows_about"><?php _e( 'Areas of Expertise (knowsAbout)', 'bw-ai-schema-pro' ); ?></label>
					<textarea id="bw_schema_knows_about" name="bw_schema_knows_about" class="widefat" rows="2"><?php echo esc_textarea( is_array( $knows_about ) ? implode( ', ', $knows_about ) : $knows_about ); ?></textarea>
					<p class="description"><?php _e( 'Comma-separated list of expertise areas (e.g., "SEO, Digital Marketing, Content Strategy")', 'bw-ai-schema-pro' ); ?></p>
				</div>
			</div>

			<!-- Credentials Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Credentials & Education (hasCredential)', 'bw-ai-schema-pro' ); ?></h4>
				<p class="description" style="margin-bottom: 12px;">
					<?php _e( 'Degrees, certifications, and professional credentials. Each generates a structured EducationalOccupationalCredential in the schema.', 'bw-ai-schema-pro' ); ?>
				</p>

				<div class="bw-schema-credential-repeater">
					<?php foreach ( $credentials as $index => $credential ) :
						$category = isset( $credential['category'] ) ? $credential['category'] : '';
						$level = isset( $credential['level'] ) ? $credential['level'] : '';
						$school = isset( $credential['school'] ) ? $credential['school'] : '';
						$cred_desc = isset( $credential['description'] ) ? $credential['description'] : '';
					?>
					<div class="bw-schema-credential-item" style="background: #f9f9f9; padding: 12px; margin-bottom: 12px; border-radius: 4px; border-left: 3px solid #72aee6;">
						<div style="display: flex; gap: 12px; margin-bottom: 8px;">
							<div style="flex: 1;">
								<label><?php _e( 'Type', 'bw-ai-schema-pro' ); ?></label>
								<select name="bw_schema_credentials[<?php echo $index; ?>][category]" class="widefat bw-credential-category">
									<option value=""><?php _e( '— Select —', 'bw-ai-schema-pro' ); ?></option>
									<option value="degree" <?php selected( $category, 'degree' ); ?>><?php _e( 'Degree', 'bw-ai-schema-pro' ); ?></option>
									<option value="certification" <?php selected( $category, 'certification' ); ?>><?php _e( 'Certification', 'bw-ai-schema-pro' ); ?></option>
									<option value="professional" <?php selected( $category, 'professional' ); ?>><?php _e( 'Professional License', 'bw-ai-schema-pro' ); ?></option>
								</select>
							</div>
							<div style="flex: 2;">
								<label><?php _e( 'Credential Name', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" name="bw_schema_credentials[<?php echo $index; ?>][level]" value="<?php echo esc_attr( $level ); ?>" placeholder="<?php esc_attr_e( 'e.g., Juris Doctor, MBA, CPA', 'bw-ai-schema-pro' ); ?>" class="widefat" />
							</div>
						</div>
						<div style="display: flex; gap: 12px;">
							<div style="flex: 1;" class="bw-credential-school-wrap">
								<label><?php _e( 'Issued By / School', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" name="bw_schema_credentials[<?php echo $index; ?>][school]" value="<?php echo esc_attr( $school ); ?>" placeholder="<?php esc_attr_e( 'e.g., Harvard University, State Bar', 'bw-ai-schema-pro' ); ?>" class="widefat" />
							</div>
							<div style="flex: 1;" class="bw-credential-desc-wrap">
								<label><?php _e( 'Description (optional)', 'bw-ai-schema-pro' ); ?></label>
								<input type="text" name="bw_schema_credentials[<?php echo $index; ?>][description]" value="<?php echo esc_attr( $cred_desc ); ?>" placeholder="<?php esc_attr_e( 'e.g., Practiced in Federal Courts', 'bw-ai-schema-pro' ); ?>" class="widefat" />
							</div>
							<div style="display: flex; align-items: flex-end;">
								<span class="bw-schema-remove-credential dashicons dashicons-no-alt" title="<?php esc_attr_e( 'Remove', 'bw-ai-schema-pro' ); ?>" style="color: #b32d2e; cursor: pointer; padding: 8px;"></span>
							</div>
						</div>
					</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button bw-schema-add-credential">
					<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Credential', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>

			<!-- Social Profiles Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Social Profiles (sameAs)', 'bw-ai-schema-pro' ); ?></h4>
				<p class="description" style="margin-bottom: 12px;">
					<?php _e( 'Links to official social profiles and authority pages. These help establish identity across the web.', 'bw-ai-schema-pro' ); ?>
				</p>

				<div class="bw-schema-repeater" data-field="same_as">
					<?php foreach ( $same_as as $index => $url ) : ?>
					<div class="bw-schema-repeater-item">
						<input type="url" name="bw_schema_same_as[]" value="<?php echo esc_url( $url ); ?>" placeholder="https://linkedin.com/in/..." class="widefat" />
						<span class="bw-schema-remove-item dashicons dashicons-no-alt" title="<?php esc_attr_e( 'Remove', 'bw-ai-schema-pro' ); ?>"></span>
					</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button bw-schema-add-item" data-field="same_as">
					<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Profile URL', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>

			<!-- Awards Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Awards & Recognition', 'bw-ai-schema-pro' ); ?></h4>

				<div class="bw-schema-repeater" data-field="awards">
					<?php foreach ( $awards as $index => $award ) : ?>
					<div class="bw-schema-repeater-item">
						<input type="text" name="bw_schema_awards[]" value="<?php echo esc_attr( $award ); ?>" placeholder="<?php esc_attr_e( 'Award name or recognition', 'bw-ai-schema-pro' ); ?>" class="widefat" />
						<span class="bw-schema-remove-item dashicons dashicons-no-alt" title="<?php esc_attr_e( 'Remove', 'bw-ai-schema-pro' ); ?>"></span>
					</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button bw-schema-add-item" data-field="awards">
					<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Award', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>

			<!-- Education Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Education (alumniOf)', 'bw-ai-schema-pro' ); ?></h4>

				<div class="bw-schema-repeater" data-field="alumni_of">
					<?php foreach ( $alumni_of as $index => $school ) : ?>
					<div class="bw-schema-repeater-item">
						<input type="text" name="bw_schema_alumni_of[]" value="<?php echo esc_attr( $school ); ?>" placeholder="<?php esc_attr_e( 'University or institution name', 'bw-ai-schema-pro' ); ?>" class="widefat" />
						<span class="bw-schema-remove-item dashicons dashicons-no-alt" title="<?php esc_attr_e( 'Remove', 'bw-ai-schema-pro' ); ?>"></span>
					</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button bw-schema-add-item" data-field="alumni_of">
					<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Education', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>

			<!-- Professional Memberships Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Professional Memberships (memberOf)', 'bw-ai-schema-pro' ); ?></h4>

				<div class="bw-schema-repeater" data-field="member_of">
					<?php foreach ( $member_of as $index => $org ) : ?>
					<div class="bw-schema-repeater-item">
						<input type="text" name="bw_schema_member_of[]" value="<?php echo esc_attr( $org ); ?>" placeholder="<?php esc_attr_e( 'Professional organization name', 'bw-ai-schema-pro' ); ?>" class="widefat" />
						<span class="bw-schema-remove-item dashicons dashicons-no-alt" title="<?php esc_attr_e( 'Remove', 'bw-ai-schema-pro' ); ?>"></span>
					</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button bw-schema-add-item" data-field="member_of">
					<span class="dashicons dashicons-plus-alt2"></span> <?php _e( 'Add Membership', 'bw-ai-schema-pro' ); ?>
				</button>
			</div>

			<!-- Video Section -->
			<div class="bw-schema-section">
				<h4 class="bw-schema-section-title"><?php _e( 'Video (subjectOf)', 'bw-ai-schema-pro' ); ?></h4>
				<p class="description" style="margin-bottom: 12px;">
					<?php _e( 'An introduction or profile video about this person. Generates a VideoObject in the schema.', 'bw-ai-schema-pro' ); ?>
				</p>

				<div class="bw-schema-field">
					<label for="bw_schema_video_name"><?php _e( 'Video Title', 'bw-ai-schema-pro' ); ?></label>
					<input type="text" id="bw_schema_video_name" name="bw_schema_video[name]" value="<?php echo esc_attr( isset( $video['name'] ) ? $video['name'] : '' ); ?>" placeholder="<?php esc_attr_e( 'e.g., Introduction to John Smith', 'bw-ai-schema-pro' ); ?>" class="widefat" />
				</div>

				<div class="bw-schema-field">
					<label for="bw_schema_video_description"><?php _e( 'Video Description', 'bw-ai-schema-pro' ); ?></label>
					<textarea id="bw_schema_video_description" name="bw_schema_video[description]" class="widefat" rows="2" placeholder="<?php esc_attr_e( 'Brief description of the video content', 'bw-ai-schema-pro' ); ?>"><?php echo esc_textarea( isset( $video['description'] ) ? $video['description'] : '' ); ?></textarea>
				</div>

				<div class="bw-schema-field-row" style="display: flex; gap: 16px;">
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_video_thumbnail"><?php _e( 'Thumbnail URL', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" id="bw_schema_video_thumbnail" name="bw_schema_video[thumbnail]" value="<?php echo esc_url( isset( $video['thumbnail'] ) ? $video['thumbnail'] : '' ); ?>" placeholder="https://..." class="widefat" />
					</div>
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_video_upload_date"><?php _e( 'Upload Date', 'bw-ai-schema-pro' ); ?></label>
						<input type="date" id="bw_schema_video_upload_date" name="bw_schema_video[upload_date]" value="<?php echo esc_attr( isset( $video['upload_date'] ) ? $video['upload_date'] : '' ); ?>" class="widefat" />
					</div>
				</div>

				<div class="bw-schema-field-row" style="display: flex; gap: 16px;">
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_video_content_url"><?php _e( 'Video File URL (optional)', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" id="bw_schema_video_content_url" name="bw_schema_video[content_url]" value="<?php echo esc_url( isset( $video['content_url'] ) ? $video['content_url'] : '' ); ?>" placeholder="<?php esc_attr_e( 'Direct link to .mp4 file', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
					<div class="bw-schema-field" style="flex: 1;">
						<label for="bw_schema_video_embed_url"><?php _e( 'Embed URL (YouTube/Vimeo)', 'bw-ai-schema-pro' ); ?></label>
						<input type="url" id="bw_schema_video_embed_url" name="bw_schema_video[embed_url]" value="<?php echo esc_url( isset( $video['embed_url'] ) ? $video['embed_url'] : '' ); ?>" placeholder="<?php esc_attr_e( 'https://youtube.com/embed/...', 'bw-ai-schema-pro' ); ?>" class="widefat" />
					</div>
				</div>
			</div>

		</div>

		<script>
		jQuery(document).ready(function($) {
			// Add repeater item (simple repeaters)
			$('.bw-schema-add-item').on('click', function() {
				var field = $(this).data('field');
				var $repeater = $(this).siblings('.bw-schema-repeater[data-field="' + field + '"]');
				var $template = $repeater.find('.bw-schema-repeater-item').first().clone();
				$template.find('input').val('');
				$repeater.append($template);
			});

			// Remove repeater item (simple repeaters)
			$(document).on('click', '.bw-schema-remove-item', function() {
				var $repeater = $(this).closest('.bw-schema-repeater');
				if ($repeater.find('.bw-schema-repeater-item').length > 1) {
					$(this).closest('.bw-schema-repeater-item').remove();
				} else {
					// Keep one empty field
					$(this).siblings('input').val('');
				}
			});

			// Add credential item
			$('.bw-schema-add-credential').on('click', function() {
				var $repeater = $(this).siblings('.bw-schema-credential-repeater');
				var $items = $repeater.find('.bw-schema-credential-item');
				var newIndex = $items.length;
				var $template = $items.first().clone();

				// Clear values and update name attributes with new index
				$template.find('select, input').each(function() {
					var name = $(this).attr('name');
					if (name) {
						$(this).attr('name', name.replace(/\[\d+\]/, '[' + newIndex + ']'));
					}
					if ($(this).is('select')) {
						$(this).val('');
					} else {
						$(this).val('');
					}
				});

				$repeater.append($template);
			});

			// Remove credential item
			$(document).on('click', '.bw-schema-remove-credential', function() {
				var $repeater = $(this).closest('.bw-schema-credential-repeater');
				if ($repeater.find('.bw-schema-credential-item').length > 1) {
					$(this).closest('.bw-schema-credential-item').remove();
				} else {
					// Keep one empty field
					var $item = $(this).closest('.bw-schema-credential-item');
					$item.find('select').val('');
					$item.find('input').val('');
				}
			});
		});
		</script>
		<?php
	}

	/**
	 * Save team member schema data
	 */
	public static function save_team_member_schema( $post_id, $post ) {
		$team_post_type = self::get_team_post_type();

		// Only process for team member post type
		if ( ! $team_post_type || $post->post_type !== $team_post_type ) {
			return;
		}

		// Verify nonce
		if ( ! isset( $_POST['bw_schema_team_member_nonce'] ) ||
		     ! wp_verify_nonce( $_POST['bw_schema_team_member_nonce'], 'bw_schema_team_member' ) ) {
			return;
		}

		// Check permissions
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Check for autosave
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Save core field overrides
		$override_fields = array(
			'_bw_schema_override_name'         => 'bw_schema_override_name',
			'_bw_schema_override_job_title'    => 'bw_schema_override_job_title',
			'_bw_schema_override_description'  => 'bw_schema_override_description',
			'_bw_schema_override_image'        => 'bw_schema_override_image',
			'_bw_schema_override_url'          => 'bw_schema_override_url',
			'_bw_schema_override_organization' => 'bw_schema_override_organization',
		);

		foreach ( $override_fields as $meta_key => $post_key ) {
			if ( isset( $_POST[ $post_key ] ) ) {
				$value = $post_key === 'bw_schema_override_description'
					? sanitize_textarea_field( $_POST[ $post_key ] )
					: ( in_array( $post_key, array( 'bw_schema_override_image', 'bw_schema_override_url' ) )
						? esc_url_raw( $_POST[ $post_key ] )
						: sanitize_text_field( $_POST[ $post_key ] ) );

				if ( ! empty( $value ) ) {
					update_post_meta( $post_id, $meta_key, $value );
				} else {
					delete_post_meta( $post_id, $meta_key );
				}
			}
		}

		// Save Is Author flag
		$is_author = isset( $_POST['bw_schema_is_author'] ) ? '1' : '';
		update_post_meta( $post_id, '_bw_schema_is_author', $is_author );

		// Save Is Leader flag
		$is_leader = isset( $_POST['bw_schema_is_leader'] ) ? '1' : '';
		update_post_meta( $post_id, '_bw_schema_is_leader', $is_leader );

		// Save linked WordPress user
		$linked_user = isset( $_POST['bw_schema_linked_user'] ) ? absint( $_POST['bw_schema_linked_user'] ) : '';
		if ( ! empty( $linked_user ) ) {
			update_post_meta( $post_id, '_bw_schema_linked_user', $linked_user );
		} else {
			delete_post_meta( $post_id, '_bw_schema_linked_user' );
		}

		// Save credentials (structured array)
		if ( isset( $_POST['bw_schema_credentials'] ) && is_array( $_POST['bw_schema_credentials'] ) ) {
			$credentials = array();
			foreach ( $_POST['bw_schema_credentials'] as $cred ) {
				// Only save if at least category or level is filled
				$category = isset( $cred['category'] ) ? sanitize_text_field( $cred['category'] ) : '';
				$level = isset( $cred['level'] ) ? sanitize_text_field( $cred['level'] ) : '';
				$school = isset( $cred['school'] ) ? sanitize_text_field( $cred['school'] ) : '';
				$description = isset( $cred['description'] ) ? sanitize_text_field( $cred['description'] ) : '';

				if ( ! empty( $category ) || ! empty( $level ) ) {
					$credentials[] = array(
						'category'    => $category,
						'level'       => $level,
						'school'      => $school,
						'description' => $description,
					);
				}
			}
			if ( ! empty( $credentials ) ) {
				update_post_meta( $post_id, '_bw_schema_credentials', $credentials );
			} else {
				delete_post_meta( $post_id, '_bw_schema_credentials' );
			}
		}

		// Save knowsAbout
		if ( isset( $_POST['bw_schema_knows_about'] ) ) {
			$knows_about = sanitize_textarea_field( $_POST['bw_schema_knows_about'] );
			if ( ! empty( $knows_about ) ) {
				$knows_about_array = array_map( 'trim', explode( ',', $knows_about ) );
				update_post_meta( $post_id, '_bw_schema_knows_about', array_filter( $knows_about_array ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_knows_about' );
			}
		}

		// Save sameAs (social profiles)
		if ( isset( $_POST['bw_schema_same_as'] ) && is_array( $_POST['bw_schema_same_as'] ) ) {
			$same_as = array_filter( array_map( 'esc_url_raw', $_POST['bw_schema_same_as'] ) );
			if ( ! empty( $same_as ) ) {
				update_post_meta( $post_id, '_bw_schema_same_as', array_values( $same_as ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_same_as' );
			}
		}

		// Save awards
		if ( isset( $_POST['bw_schema_awards'] ) && is_array( $_POST['bw_schema_awards'] ) ) {
			$awards = array_filter( array_map( 'sanitize_text_field', $_POST['bw_schema_awards'] ) );
			if ( ! empty( $awards ) ) {
				update_post_meta( $post_id, '_bw_schema_awards', array_values( $awards ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_awards' );
			}
		}

		// Save alumniOf
		if ( isset( $_POST['bw_schema_alumni_of'] ) && is_array( $_POST['bw_schema_alumni_of'] ) ) {
			$alumni_of = array_filter( array_map( 'sanitize_text_field', $_POST['bw_schema_alumni_of'] ) );
			if ( ! empty( $alumni_of ) ) {
				update_post_meta( $post_id, '_bw_schema_alumni_of', array_values( $alumni_of ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_alumni_of' );
			}
		}

		// Save memberOf
		if ( isset( $_POST['bw_schema_member_of'] ) && is_array( $_POST['bw_schema_member_of'] ) ) {
			$member_of = array_filter( array_map( 'sanitize_text_field', $_POST['bw_schema_member_of'] ) );
			if ( ! empty( $member_of ) ) {
				update_post_meta( $post_id, '_bw_schema_member_of', array_values( $member_of ) );
			} else {
				delete_post_meta( $post_id, '_bw_schema_member_of' );
			}
		}

		// Save honorific prefix
		if ( isset( $_POST['bw_schema_honorific_prefix'] ) ) {
			$honorific_prefix = sanitize_text_field( $_POST['bw_schema_honorific_prefix'] );
			if ( ! empty( $honorific_prefix ) ) {
				update_post_meta( $post_id, '_bw_schema_honorific_prefix', $honorific_prefix );
			} else {
				delete_post_meta( $post_id, '_bw_schema_honorific_prefix' );
			}
		}

		// Save honorific suffix
		if ( isset( $_POST['bw_schema_honorific_suffix'] ) ) {
			$honorific_suffix = sanitize_text_field( $_POST['bw_schema_honorific_suffix'] );
			if ( ! empty( $honorific_suffix ) ) {
				update_post_meta( $post_id, '_bw_schema_honorific_suffix', $honorific_suffix );
			} else {
				delete_post_meta( $post_id, '_bw_schema_honorific_suffix' );
			}
		}

		// Save video (subjectOf)
		if ( isset( $_POST['bw_schema_video'] ) && is_array( $_POST['bw_schema_video'] ) ) {
			$video = array(
				'name'        => isset( $_POST['bw_schema_video']['name'] ) ? sanitize_text_field( $_POST['bw_schema_video']['name'] ) : '',
				'description' => isset( $_POST['bw_schema_video']['description'] ) ? sanitize_textarea_field( $_POST['bw_schema_video']['description'] ) : '',
				'thumbnail'   => isset( $_POST['bw_schema_video']['thumbnail'] ) ? esc_url_raw( $_POST['bw_schema_video']['thumbnail'] ) : '',
				'upload_date' => isset( $_POST['bw_schema_video']['upload_date'] ) ? sanitize_text_field( $_POST['bw_schema_video']['upload_date'] ) : '',
				'content_url' => isset( $_POST['bw_schema_video']['content_url'] ) ? esc_url_raw( $_POST['bw_schema_video']['content_url'] ) : '',
				'embed_url'   => isset( $_POST['bw_schema_video']['embed_url'] ) ? esc_url_raw( $_POST['bw_schema_video']['embed_url'] ) : '',
			);

			// Only save if at least name is filled
			if ( ! empty( $video['name'] ) ) {
				update_post_meta( $post_id, '_bw_schema_video', $video );
			} else {
				delete_post_meta( $post_id, '_bw_schema_video' );
			}
		}
	}

	/**
	 * Get team members flagged as authors
	 *
	 * @return WP_Post[] Array of team member posts
	 */
	public static function get_author_team_members() {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type ) {
			return array();
		}

		return get_posts( array(
			'post_type'      => $team_post_type,
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'   => '_bw_schema_is_author',
					'value' => '1',
				),
			),
			'orderby'        => 'title',
			'order'          => 'ASC',
		) );
	}

	/**
	 * Get team members flagged as leaders
	 *
	 * @return WP_Post[] Array of team member posts
	 */
	public static function get_leader_team_members() {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type ) {
			return array();
		}

		return get_posts( array(
			'post_type'      => $team_post_type,
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'   => '_bw_schema_is_leader',
					'value' => '1',
				),
			),
			'orderby'        => 'title',
			'order'          => 'ASC',
		) );
	}

	/**
	 * Get team member by linked WordPress user ID
	 *
	 * @param int $user_id WordPress user ID
	 * @return WP_Post|null Team member post or null
	 */
	public static function get_team_member_by_user( $user_id ) {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type ) {
			return null;
		}

		$posts = get_posts( array(
			'post_type'      => $team_post_type,
			'posts_per_page' => 1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'   => '_bw_schema_linked_user',
					'value' => $user_id,
				),
			),
		) );

		return ! empty( $posts ) ? $posts[0] : null;
	}

	/**
	 * Get Person schema data for a team member
	 *
	 * @param WP_Post|int $team_member Team member post object or ID
	 * @return array Person schema data
	 */
	public static function get_person_schema_data( $team_member ) {
		if ( is_numeric( $team_member ) ) {
			$team_member = get_post( $team_member );
		}

		if ( ! $team_member ) {
			return array();
		}

		// Check for overrides first, fall back to detected values
		$override_name = get_post_meta( $team_member->ID, '_bw_schema_override_name', true );
		$override_description = get_post_meta( $team_member->ID, '_bw_schema_override_description', true );
		$override_image = get_post_meta( $team_member->ID, '_bw_schema_override_image', true );
		$override_url = get_post_meta( $team_member->ID, '_bw_schema_override_url', true );
		$override_job_title = get_post_meta( $team_member->ID, '_bw_schema_override_job_title', true );

		$data = array(
			'name'        => ! empty( $override_name ) ? $override_name : $team_member->post_title,
			'url'         => ! empty( $override_url ) ? $override_url : get_permalink( $team_member ),
			'description' => ! empty( $override_description ) ? $override_description : get_the_excerpt( $team_member ),
			'image'       => ! empty( $override_image ) ? $override_image : get_the_post_thumbnail_url( $team_member, 'full' ),
		);

		// Get job title: override > detected from meta
		$job_title = $override_job_title;
		if ( empty( $job_title ) ) {
			$detected = self::detect_job_title( $team_member->ID );
			$job_title = $detected['value'];
		}
		if ( ! empty( $job_title ) ) {
			$data['jobTitle'] = $job_title;
		}

		// Get worksFor organization: override > site organization
		$override_org = get_post_meta( $team_member->ID, '_bw_schema_override_organization', true );
		if ( ! empty( $override_org ) ) {
			// Custom organization
			$data['worksFor'] = array(
				'@type' => 'Organization',
				'name'  => $override_org,
			);
		} else {
			// Site organization with @id reference
			$org_data = BW_Schema_Core::get_organization_schema();
			$data['worksFor'] = array(
				'@type' => 'Organization',
				'name'  => ! empty( $org_data['name'] ) ? $org_data['name'] : get_bloginfo( 'name' ),
				'@id'   => home_url( '/#organization' ),
			);
		}

		// Honorific fields
		$honorific_prefix = get_post_meta( $team_member->ID, '_bw_schema_honorific_prefix', true );
		if ( ! empty( $honorific_prefix ) ) {
			$data['honorificPrefix'] = $honorific_prefix;
		}

		$honorific_suffix = get_post_meta( $team_member->ID, '_bw_schema_honorific_suffix', true );
		if ( ! empty( $honorific_suffix ) ) {
			$data['honorificSuffix'] = $honorific_suffix;
		}

		// Schema enhancement fields

		// Credentials (structured EducationalOccupationalCredential)
		$credentials = get_post_meta( $team_member->ID, '_bw_schema_credentials', true );
		if ( ! empty( $credentials ) && is_array( $credentials ) ) {
			$credential_schemas = array();
			foreach ( $credentials as $cred ) {
				if ( empty( $cred['level'] ) && empty( $cred['category'] ) ) {
					continue;
				}

				$cred_schema = array(
					'@type' => 'EducationalOccupationalCredential',
				);

				// Set credentialCategory
				if ( ! empty( $cred['category'] ) ) {
					$cred_schema['credentialCategory'] = $cred['category'];
				}

				// For degrees, use educationalLevel; for others, use name
				if ( $cred['category'] === 'degree' ) {
					if ( ! empty( $cred['level'] ) ) {
						$cred_schema['educationalLevel'] = $cred['level'];
					}
					// School becomes recognizedBy with CollegeOrUniversity type
					if ( ! empty( $cred['school'] ) ) {
						$cred_schema['recognizedBy'] = array(
							'@type' => 'CollegeOrUniversity',
							'name'  => $cred['school'],
						);
					}
				} else {
					// For certifications and professional licenses
					if ( ! empty( $cred['level'] ) ) {
						$cred_schema['name'] = $cred['level'];
					}
					// School/issuer becomes recognizedBy with Organization type
					if ( ! empty( $cred['school'] ) ) {
						$cred_schema['recognizedBy'] = array(
							'@type' => 'Organization',
							'name'  => $cred['school'],
						);
					}
				}

				// Add description if present
				if ( ! empty( $cred['description'] ) ) {
					$cred_schema['description'] = $cred['description'];
				}

				$credential_schemas[] = $cred_schema;
			}

			if ( ! empty( $credential_schemas ) ) {
				$data['hasCredential'] = $credential_schemas;
			}
		}

		$knows_about = get_post_meta( $team_member->ID, '_bw_schema_knows_about', true );
		if ( ! empty( $knows_about ) ) {
			$data['knowsAbout'] = $knows_about;
		}

		$same_as = get_post_meta( $team_member->ID, '_bw_schema_same_as', true );
		if ( ! empty( $same_as ) ) {
			$data['sameAs'] = $same_as;
		}

		$awards = get_post_meta( $team_member->ID, '_bw_schema_awards', true );
		if ( ! empty( $awards ) ) {
			$data['award'] = $awards;
		}

		$alumni_of = get_post_meta( $team_member->ID, '_bw_schema_alumni_of', true );
		if ( ! empty( $alumni_of ) ) {
			$data['alumniOf'] = array_map( function( $name ) {
				return array(
					'@type' => 'EducationalOrganization',
					'name'  => $name,
				);
			}, $alumni_of );
		}

		$member_of = get_post_meta( $team_member->ID, '_bw_schema_member_of', true );
		if ( ! empty( $member_of ) ) {
			$data['memberOf'] = array_map( function( $name ) {
				return array(
					'@type' => 'Organization',
					'name'  => $name,
				);
			}, $member_of );
		}

		// Video (subjectOf VideoObject)
		$video = get_post_meta( $team_member->ID, '_bw_schema_video', true );
		if ( ! empty( $video ) && is_array( $video ) && ! empty( $video['name'] ) ) {
			$video_schema = array(
				'@type' => 'VideoObject',
				'name'  => $video['name'],
			);

			if ( ! empty( $video['description'] ) ) {
				$video_schema['description'] = $video['description'];
			}

			if ( ! empty( $video['thumbnail'] ) ) {
				$video_schema['thumbnailUrl'] = $video['thumbnail'];
			}

			if ( ! empty( $video['upload_date'] ) ) {
				// Format as ISO 8601
				$video_schema['uploadDate'] = $video['upload_date'] . 'T00:00:00+00:00';
			}

			if ( ! empty( $video['content_url'] ) ) {
				$video_schema['contentUrl'] = $video['content_url'];
			}

			if ( ! empty( $video['embed_url'] ) ) {
				$video_schema['embedUrl'] = $video['embed_url'];
			}

			$data['subjectOf'] = $video_schema;
		}

		return apply_filters( 'bw_schema_person_data', $data, $team_member );
	}

	/**
	 * Disable conflicting schemas on team member pages
	 */
	public static function disable_conflicting_schemas() {
		$team_post_type = self::get_team_post_type();

		if ( ! $team_post_type || ! is_singular( $team_post_type ) ) {
			return;
		}

		// Disable Yoast
		add_filter( 'wpseo_json_ld_output', '__return_false', 999 );
		add_filter( 'wpseo_schema_graph', '__return_empty_array', 999 );
		add_filter( 'wpseo_schema_webpage', '__return_empty_array', 999 );

		// Disable Rank Math
		add_filter( 'rank_math/json_ld', '__return_empty_array', 999 );

		// Disable All in One SEO
		add_filter( 'aioseo_schema_output', '__return_false', 999 );

		// Disable SEOPress
		add_filter( 'seopress_schemas_jsonld_output', '__return_false', 999 );

		// Remove generic schema actions
		remove_all_actions( 'wp_head', 60 );

		// Add a high priority action to remove late-added schemas
		add_action( 'wp_head', array( __CLASS__, 'remove_remaining_schemas' ), 0 );
	}

	/**
	 * Remove any remaining schema outputs from other plugins
	 */
	public static function remove_remaining_schemas() {
		// Remove any schema-related actions that might have been added late
		// BUT preserve our own BW AI Schema Pro hooks
		global $wp_filter;

		if ( isset( $wp_filter['wp_head'] ) ) {
			foreach ( $wp_filter['wp_head'] as $priority => $hooks ) {
				foreach ( $hooks as $hook_key => $hook ) {
					// Skip our own hooks - check for class names AND method names
					// Object callbacks generate keys like "hash + method_name"
					if ( strpos( $hook_key, 'BW_AI_Schema_Pro' ) !== false ||
					     strpos( $hook_key, 'BW_Schema' ) !== false ||
					     strpos( $hook_key, 'output_schema_markup' ) !== false ) {
						continue;
					}

					// Check if this might be a schema-related hook from other plugins
					if ( strpos( $hook_key, 'schema' ) !== false ||
					     strpos( $hook_key, 'structured_data' ) !== false ||
					     strpos( $hook_key, 'json_ld' ) !== false ) {
						remove_action( 'wp_head', $hook_key, $priority );
					}
				}
			}
		}
	}
}
