# Design Document: Unified People/Author Architecture (v2.0)

**Status:** Approved
**Authors:** Rian, Claude
**Date:** 2026-01-15

---

## Problem Statement

Currently, a person like "Tom" can be defined in 4 separate places:
1. **WordPress User** - For login and posting
2. **Team Member CPT** - Public profile page
3. **AI Schema Pro Custom Author** - Schema-specific data
4. **Per-post override** - Edge case overrides

This leads to:
- Data duplication and inconsistency
- Confusion about which source is used where
- Bylines linking to WP author archives instead of team pages
- Schema data disconnected from public profiles

---

## Goals

1. **Single source of truth** for people who appear publicly on the site
2. **Bylines link to team pages** automatically
3. **Schema lives with the profile** (not in a separate admin section)
4. **Support different roles**: team members, authors, admins, external contributors
5. **Make the plugin portable** (configurable team member CPT)

---

## Solution Overview

### Person Types

```
┌─────────────────────────────────────────────────────────────┐
│                      PERSON TYPES                           │
├──────────────────────────┬──────────────────────────────────┤
│     TEAM MEMBER          │      EXTERNAL AUTHOR             │
│     (CPT Post)           │      (Plugin Storage)            │
├──────────────────────────┼──────────────────────────────────┤
│ • Has public profile page│ • No public page                 │
│ • Full Person schema     │ • Basic Person schema            │
│ • Can be flagged "Author"│ • Always an author               │
│ • Can link to WP User    │ • Reusable across posts          │
│ • Source of truth        │ • For guest contributors         │
└──────────────────────────┴──────────────────────────────────┘
```

### Relationships

```
┌─────────────────┐         ┌─────────────────┐
│  TEAM MEMBER    │◄───────►│  WORDPRESS USER │
│  (Tom's page)   │ optional│  (Tom's login)  │
└────────┬────────┘         └────────┬────────┘
         │                           │
         │ if "Is Author"            │ "Default Author"
         ▼                           ▼
┌─────────────────┐         ┌─────────────────┐
│  AUTHOR         │◄────────│  POSTS BY USER  │
│  (in selection) │ default │  (Jenny posts)  │
└─────────────────┘         └─────────────────┘
```

---

## Detailed Design

### 1. Plugin Settings

**New Settings:**

```php
// Team Member CPT configuration
'team_member_post_type' => 'sw_teammember', // Configurable

// Feature toggles
'use_team_as_authors' => true,  // Enable new architecture
'allow_external_authors' => true,
```

**Admin UI:**
- Settings > AI Schema Pro > General
- Dropdown: "Team Member Post Type" (lists all CPTs)
- If no team CPT selected, show warning

---

### 2. Team Member Schema Meta Box

**Location:** Edit screen for configured Team Member CPT

**Fields Added:**

| Field | Type | Description |
|-------|------|-------------|
| **Author Settings** | Section | |
| Is Author | Checkbox | Include in author selection |
| Linked WP User | User dropdown | Optional user account link |
| **Person Schema** | Section | |
| Job Title | Text | (may already exist in CPT) |
| Credentials | Text | Degrees, certifications |
| Knows About | Textarea | Areas of expertise |
| Awards | Repeater | Awards and recognition |
| Same As | Repeater | Social profile URLs |
| Alumni Of | Repeater | Education history |
| Member Of | Repeater | Professional memberships |

**Note:** Some fields may already exist in the Team CPT (via ACF or theme). The plugin should detect and use existing fields where possible, or add its own.

---

### 3. Authors Section Redesign

**Current:** Separate "Custom Authors" with full form

**New:**

```
AI Schema Pro > Authors
├── Team Member Authors (from Team CPT where Is Author = true)
│   ├── Tom Smith [Edit] → links to Team Member edit screen
│   ├── Jane Doe [Edit] → links to Team Member edit screen
│   └── + Add Author → links to "Add New Team Member"
│
└── External Authors (stored in plugin)
    ├── Guest Writer 1 [Edit] → inline form
    ├── Guest Writer 2 [Edit] → inline form
    └── + Add External Author → inline form
```

**"Edit" for Team Member:** Redirects to `/wp-admin/post.php?post={team_member_id}&action=edit`

---

### 4. WordPress User Settings

**New User Profile Fields:**

| Field | Type | Description |
|-------|------|-------------|
| Default Author | Dropdown | Team member or external author |
| | | "When I publish, default author to..." |

**Logic:**
```php
// On post save, if author not explicitly set:
$default_author = get_user_meta($current_user_id, 'bw_default_author', true);
if ($default_author && !get_post_meta($post_id, '_bw_schema_author', true)) {
    update_post_meta($post_id, '_bw_schema_author', $default_author);
}
```

---

### 5. Post Author Selection

**Current:**
- Dropdown with: WordPress Users, Custom Authors, External (inline)

**New:**
- Dropdown with: Team Member Authors, External Authors
- "Add new external author" expands inline form
- Default pre-selected based on current user's settings

**UI Mockup:**
```
Author *
┌─────────────────────────────────────────┐
│ ▼ Tom Smith (Team Member)               │
├─────────────────────────────────────────┤
│   Team Members                          │
│   ├── Tom Smith                         │
│   ├── Jane Doe                          │
│   └── Bob Wilson                        │
│   External Authors                      │
│   ├── Guest Writer 1                    │
│   └── Guest Writer 2                    │
│   ──────────────────────────────────    │
│   + Add new external author             │
└─────────────────────────────────────────┘
```

---

### 6. Schema Generation

#### 6a. Team Member Page Schema (ProfilePage + Person)

**Important:** The team member page itself should use `ProfilePage` as the top-level type, with the `Person` as the `mainEntity`. This tells Google "This page is a profile about this person."

```php
function get_team_member_page_schema($team_member) {
    $person_schema = get_person_schema($team_member);

    return [
        '@context' => 'https://schema.org',
        '@type' => 'ProfilePage',
        'mainEntity' => $person_schema,
        'name' => $team_member->post_title . ' | Team',
        'url' => get_permalink($team_member),
    ];
}
```

#### 6b. Person Schema (for author references)

```php
function get_person_schema($team_member) {
    $org_data = BW_Schema_Core::get_organization_schema();

    return [
        '@type' => 'Person',
        'name' => $team_member->post_title,
        'url' => get_permalink($team_member), // Team page URL!
        'jobTitle' => get_post_meta($team_member->ID, 'job_title', true),
        'description' => get_the_excerpt($team_member),
        'image' => get_the_post_thumbnail_url($team_member),

        // Reciprocal linking - connects Person to Organization
        'worksFor' => [
            '@type' => 'Organization',
            'name' => $org_data['name'],
            '@id' => home_url('/#organization'),
        ],

        // Schema enhancement fields
        'knowsAbout' => get_post_meta($team_member->ID, '_bw_schema_knows_about', true),
        'hasCredential' => get_post_meta($team_member->ID, '_bw_schema_credentials', true),
        'sameAs' => get_post_meta($team_member->ID, '_bw_schema_same_as', true),
        'memberOf' => get_post_meta($team_member->ID, '_bw_schema_member_of', true),
        'alumniOf' => get_post_meta($team_member->ID, '_bw_schema_alumni_of', true),
        // ... etc
    ];
}

function get_author_schema($author_ref) {
    if ($author_ref['type'] === 'team_member') {
        $team_member = get_post($author_ref['id']);
        return get_person_schema($team_member);
    }
}
```

**Key Points:**
- Team member pages use `ProfilePage` with `mainEntity: Person`
- Person schema includes `worksFor` linking back to Organization (reciprocal linking)
- The `url` is the team member permalink, not WP author archive
- Organization schema should include `employee` array referencing team members

---

### 7. Byline Display

**Current Logic:** Filter `the_author` and related, try to override name

**New Logic:**
```php
function filter_author_link($link, $author_id) {
    $post_author = get_post_author_data(get_the_ID());

    if ($post_author['type'] === 'team_member') {
        // Link to team page
        return get_permalink($post_author['id']);
    } elseif ($post_author['type'] === 'external') {
        // No link, or link to their website if provided
        return $post_author['website'] ?? '';
    }
}
```

---

### 8. Data Storage

**Team Member Post Meta (new fields):**
```
_bw_schema_is_author = '1' | ''
_bw_schema_is_leader = '1' | ''    // Founder/Leadership team
_bw_schema_linked_user = {user_id} | ''
_bw_schema_credentials = 'PhD, MBA'
_bw_schema_knows_about = ['SEO', 'Marketing', 'WordPress']
_bw_schema_same_as = ['https://linkedin.com/...', 'https://twitter.com/...']
_bw_schema_awards = [...]
_bw_schema_alumni_of = [...]
_bw_schema_member_of = [...]
```

**User Meta (new):**
```
bw_default_author = ['type' => 'team_member', 'id' => 123]
```

**External Authors (plugin option):**
```php
// bw_schema_external_authors option
[
    'ext_001' => [
        'id' => 'ext_001',
        'name' => 'Guest Writer',
        'job_title' => 'Freelance Journalist',
        'bio' => '...',
        'image' => 'https://...',
        'website' => 'https://...',
        'social' => [...],
        'created' => '2026-01-15',
    ],
    // ...
]
```

**Post Meta (author assignment):**
```
_bw_schema_authors = [
    ['type' => 'team_member', 'id' => 123],
    ['type' => 'external', 'id' => 'ext_001'],
]
```

---

## Migration Plan

### From v1.x to v2.0

1. **Detect existing custom authors**
   - Check `bw_schema_custom_authors` option

2. **For each custom author:**
   - If matching team member exists (by name): Link them
   - If no match: Offer to create team member OR convert to external

3. **Update post references:**
   - Change `['type' => 'custom', 'id' => 'xxx']`
   - To `['type' => 'team_member', 'id' => 123]` or `['type' => 'external', 'id' => 'ext_xxx']`

4. **Preserve backward compatibility:**
   - Keep old author resolution as fallback for 1 version
   - Log warnings for unmigrated data

---

## UI/UX Considerations

### For Content Editors (Jenny)

**Before:**
- Confusing: "Do I pick from Users? Custom Authors? What's the difference?"

**After:**
- Simple: "Pick an author from the dropdown. It defaults to Tom."
- Clear categories: Team Members vs External Authors

### For Admins (Rian)

**Before:**
- Edit Tom's info in multiple places
- Schema separate from profile

**After:**
- Edit Tom's team page = edit everything
- Schema fields right there on the same screen

---

## Resolved Design Questions

1. **Field detection:** How to detect if team CPT already has job_title, bio, etc. via ACF?
   - **Decision:** Check common field names, allow mapping in settings

2. **Multiple team CPTs:** What if site has both "staff" and "leadership" CPTs?
   - **Decision:** Allow multiple CPTs to be selected as "person" sources

3. **Author archives:** Should we auto-redirect `/author/tom/` to team page?
   - **Decision:** YES - Auto-redirect by default
   - **Rationale:** Default WordPress author archives (`/author/username`) are low-quality lists of posts. High-quality Team Member pages should be the canonical URL for a person. Splitting link equity between the WP archive and Team page dilutes authority.
   - **Implementation:** Add setting "Redirect Author Archives to Team Member pages" (enabled by default)

---

## Additional Schema Requirements (from review)

### Author Archive Redirects

**New Setting:**
```php
'redirect_author_archives' => true,  // Enabled by default
```

**Logic:**
```php
// On author archive template_redirect
function maybe_redirect_author_archive() {
    if (!is_author() || !get_option('bw_schema_redirect_author_archives', true)) {
        return;
    }

    $author = get_queried_object();
    $team_member_id = get_team_member_by_user($author->ID);

    if ($team_member_id) {
        wp_redirect(get_permalink($team_member_id), 301);
        exit;
    }
}
```

### Organization → Employee Reciprocal Link

Organization schema should reference team members as employees:

```php
function get_organization_schema() {
    $team_members = get_posts([
        'post_type' => get_option('bw_schema_team_post_type'),
        'meta_query' => [
            ['key' => '_bw_schema_is_author', 'value' => '1']
        ],
    ]);

    $employees = array_map(function($member) {
        return [
            '@type' => 'Person',
            'name' => $member->post_title,
            'url' => get_permalink($member),
            '@id' => get_permalink($member) . '#person',
        ];
    }, $team_members);

    return [
        '@type' => 'Organization',
        '@id' => home_url('/#organization'),
        'name' => $org_name,
        'employee' => $employees,
        // ... other org fields
    ];
}
```

This creates a bidirectional knowledge graph:
- **Organization** has `employee` → references Person
- **Person** has `worksFor` → references Organization

---

## Setup Wizard Changes

### Current Wizard (v1.x)
1. Organization Profile
2. Leadership & Team ← **Remove** (leadership moves to People)
3. Contact & Social
4. Page Mapping
5. Content Defaults
6. Schema Features ← Has team post type setting (move to People)
7. Review & Complete

### New Wizard (v2.0)
1. Organization Profile
2. Contact & Social (was step 3)
3. Page Mapping (was step 4)
4. Content Defaults (was step 5)
5. **People** ← **NEW** (consolidates leadership + team + authors)
6. Schema Features (simplified)
7. Review & Complete

### Step 5: People (New)

**Purpose:** Single "at a glance" view of all people on the site.

**Sections:**

#### 5a. Team Configuration
```
Team Member Post Type: [dropdown: sw_teammember ▼]
```

#### 5b. Team Members Overview
| Team Member | Is Author | Is Leader | Linked WP User |
|-------------|-----------|-----------|----------------|
| Tom Smith | ☑ | ☑ | tom@... |
| Jane Doe | ☑ | ☐ | - |
| Bob Wilson | ☐ | ☑ | - |
| [+ Add Team Member] |

- Checkboxes update team member post meta directly
- "Is Leader" = appears in founders/leadership schema
- "Is Author" = appears in post author dropdown

#### 5c. WordPress Users & Default Authors
| WordPress User | Role | Default Author |
|----------------|------|----------------|
| admin@... | Administrator | Tom Smith ▼ |
| jenny@... | Editor | Tom Smith ▼ |
| tom@... | Author | Tom Smith ▼ |

- Dropdown shows team members flagged as authors
- When user creates post, this author is pre-selected

**Benefits:**
- See all people relationships in one place
- No need for separate "Leadership" step
- No need for "Authors" in Schema Features
- Clear visual of who posts as whom

---

## Implementation Order

### Phase 1: Foundation
1. [ ] Add Team CPT setting to plugin
2. [ ] Create meta box for team member edit screen
3. [ ] Add "Is Author" and "Is Leader" flag logic

### Phase 2: Schema Generation
4. [ ] Update team member page schema (ProfilePage + Person)
5. [ ] Add `worksFor` property to Person schema
6. [ ] Update Organization schema with `employee` array (reciprocal link)
7. [ ] Update schema generation for article authors

### Phase 3: Author Management
8. [ ] Redesign Authors admin page
9. [ ] Add User profile "Default Author" field
10. [ ] Update post author selection UI

### Phase 4: URL & Redirect Handling
11. [ ] Update byline display to link to team pages
12. [ ] Add author archive redirect setting
13. [ ] Implement 301 redirect from /author/username/ to team page

### Phase 5: Wizard & Migration
14. [ ] **Redesign Setup Wizard Step 5 (People)**
15. [ ] **Remove old Leadership step from wizard**
16. [ ] Build migration tool for existing custom authors

### Phase 6: Testing & Documentation
17. [ ] Testing and documentation
18. [ ] Validate with Google Rich Results Test

---

## Success Criteria

### Core Functionality
- [ ] Tom exists in ONE place (team member)
- [ ] Tom's byline links to his team page
- [ ] Tom's schema pulls from team member + enhancements
- [ ] Jenny can post and it defaults to Tom as author
- [ ] External guest authors work for one-off contributors
- [ ] Existing sites can migrate without data loss

### Schema Compliance (from review)
- [ ] Team member pages output `ProfilePage` with `mainEntity: Person`
- [ ] Person schema includes `worksFor` referencing Organization
- [ ] Organization schema includes `employee` array referencing team members
- [ ] `/author/username/` redirects 301 to team member page (when linked)
- [ ] Schema validates in Google Rich Results Test
