# BW AI Schema Pro - Architecture Reference

**Version:** 1.0.5 | **Last Updated:** 2026-01-15

Technical documentation for developers working on the plugin.

---

## Design Patterns

### Singleton Pattern
Main plugin class uses singleton for single instance:
```php
class BW_AI_Schema_Pro {
    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
}
```

### Abstract Factory
Schema types extend abstract base class:
```
BW_Schema_Base (abstract)
├── BW_Schema_Article
├── BW_Schema_Person
├── BW_Schema_Organization
├── BW_Schema_LocalBusiness
├── BW_Schema_Product
├── BW_Schema_Event
├── BW_Schema_HowTo
├── BW_Schema_FAQ
├── BW_Schema_Review
├── BW_Schema_Recipe
├── BW_Schema_Video
├── BW_Schema_Course
├── BW_Schema_JobPosting
└── BW_Schema_WebPage
```

### Autoloader
SPL autoloader for class loading:
```php
spl_autoload_register(function($class) {
    if (strpos($class, 'BW_Schema_') === 0) {
        // Load from includes/ or includes/schemas/
    }
});
```

---

## File Structure

```
bw-ai-schema-pro/
├── bw-ai-schema-pro.php           # Main plugin file, entry point
├── CLAUDE.md                      # AI development guide
│
├── docs/                          # Documentation
│   ├── SPEC.md                    # Feature specifications
│   ├── ARCHITECTURE.md            # This file
│   ├── CHANGELOG.md               # Version history
│   ├── ROADMAP.md                 # Future plans
│   ├── SESSION-LOG.md             # Session handoffs
│   ├── TESTING.md                 # Test checklists
│   ├── KNOWN-ISSUES.md            # Bug tracking
│   └── guides/
│       └── team-schema.md         # Team page guide
│
├── admin/
│   ├── assets/
│   │   ├── css/admin.css          # Admin styles (~25KB)
│   │   └── js/admin.js            # Admin JavaScript (~14KB)
│   └── views/
│       ├── dashboard.php          # Main dashboard
│       ├── settings.php           # Global settings
│       ├── author-profiles.php    # Custom author management
│       ├── setup-wizard.php       # 7-step wizard (~58KB)
│       └── authors.php            # Legacy view
│
├── assets/
│   ├── css/
│   │   ├── author-box.css         # Author box styles
│   │   ├── frontend.css           # General frontend
│   │   └── frontend-authors.css   # Multiple authors display
│   └── js/
│       └── frontend-authors.js    # Author name replacement
│
├── blocks/
│   └── author-schema/             # Gutenberg block
│       ├── editor.js
│       ├── editor.css
│       └── style.css
│
├── includes/
│   ├── class-bw-schema-core.php           # Core utilities (~29KB)
│   ├── class-bw-schema-admin.php          # Admin functionality (~54KB)
│   ├── class-bw-schema-renderer.php       # Schema output (~19KB)
│   ├── class-bw-schema-author-override.php # Author display (~42KB)
│   ├── class-bw-schema-author-box.php     # Author box (~17KB)
│   ├── class-bw-schema-templates.php      # Template handling (~10KB)
│   ├── class-bw-schema-blocks.php         # Gutenberg (~12KB)
│   ├── class-bw-schema-cache.php          # Caching (~3KB)
│   ├── class-bw-schema-hooks.php          # Developer hooks (~5KB)
│   ├── class-bw-schema-helpers.php        # Helpers (~11KB)
│   ├── class-bw-schema-security.php       # Security (~11KB)
│   ├── class-bw-schema-team-detector.php  # Team detection (~6KB)
│   └── class-bw-schema-team-member.php    # Team members (~7KB)
│
└── includes/schemas/
    ├── class-bw-schema-base.php           # Abstract base (~17KB)
    ├── class-bw-schema-article.php        # Article types (~7KB)
    ├── class-bw-schema-person.php         # Person (~15KB)
    ├── class-bw-schema-organization.php   # Organization (~9KB)
    ├── class-bw-schema-localbusiness.php  # LocalBusiness (~17KB)
    ├── class-bw-schema-product.php        # Product (~13KB)
    ├── class-bw-schema-event.php          # Event (~9KB)
    ├── class-bw-schema-howto.php          # HowTo (~13KB)
    ├── class-bw-schema-faq.php            # FAQ (~7KB)
    ├── class-bw-schema-review.php         # Review (~6KB)
    ├── class-bw-schema-recipe.php         # Recipe (~11KB)
    ├── class-bw-schema-video.php          # Video (~13KB)
    ├── class-bw-schema-course.php         # Course (~14KB)
    ├── class-bw-schema-jobposting.php     # JobPosting (~10KB)
    └── class-bw-schema-webpage.php        # WebPage (~8KB)
```

---

## Initialization Flow

```
plugins_loaded hook
└── BW_AI_Schema_Pro::get_instance()
    ├── init_hooks()
    │   ├── wp_head (priority 1) → output_schema_markup()
    │   ├── admin_menu → add_menu_pages()
    │   ├── add_meta_boxes → add_meta_boxes()
    │   └── save_post → save_post_meta()
    │
    └── load_dependencies()
        ├── BW_Schema_Core::init()
        ├── BW_Schema_Renderer
        ├── BW_Schema_Admin (admin only)
        ├── BW_Schema_Author_Override
        ├── BW_Schema_Author_Box
        ├── BW_Schema_Cache
        ├── BW_Schema_Hooks
        ├── BW_Schema_Security
        ├── BW_Schema_Team_Detector
        ├── BW_Schema_Team_Member
        └── BW_Schema_Blocks
```

---

## Class Responsibilities

### BW_AI_Schema_Pro (Main)
- Plugin entry point and orchestrator
- WordPress hook registration
- Admin menu registration
- Post meta box rendering and saving
- AJAX handlers (preview, simple fields)

### BW_Schema_Core
- Schema type definitions
- AI property definitions
- Settings management (get/update)
- Option migration
- Conflict detection and disabling
- Setup completion tracking

### BW_Schema_Renderer
- Context detection (singular, archive, team, author)
- Schema generation orchestration
- WebSite/Organization schema
- Breadcrumb schema
- Caching integration

### BW_Schema_Admin
- Admin page rendering
- Settings save handlers
- Setup wizard step handling
- Custom author CRUD

### BW_Schema_Author_Override
- 40+ WordPress filters for author display
- Kadence theme integration
- Yoast SEO schema integration
- Output buffering fallback
- JavaScript DOM replacement

### BW_Schema_Author_Box
- Frontend author box rendering
- `[bw_author_box]` shortcode
- Multiple author support

### BW_Schema_Base (Abstract)
Base class providing:
- `get_common_properties()` - name, URL, dates, description
- `get_multiple_authors_properties()` - multiple author support
- `get_author_properties()` - single author
- `get_organization_properties()` - publisher
- `add_ai_properties()` - AI optimization

---

## Data Storage

### Options (wp_options)

| Key | Type | Description |
|-----|------|-------------|
| `bw_schema_settings` | array | Consolidated settings (v2.0.0+) |
| `bw_schema_organization` | array | Organization name, URL, logo, sameAs |
| `bw_schema_business_type` | string | Business category |
| `bw_schema_founders` | array | Founder/leadership data |
| `bw_schema_custom_authors` | array | Custom author profiles |
| `bw_schema_contact_points` | array | Contact information |
| `bw_schema_page_mappings` | array | Page type to post ID map |
| `bw_schema_post_type_defaults` | array | Default schemas per post type |
| `bw_schema_setup_complete` | bool | Wizard completed |
| `bw_schema_disable_sources` | array | Disabled schema sources |

### Post Meta

| Key | Type | Description |
|-----|------|-------------|
| `_bw_schema_type` | string | Selected schema type (e.g., 'article:BlogPosting') |
| `_bw_schema_disable` | string | 'yes' to disable schema |
| `_bw_schema_custom` | string | Custom JSON-LD override |
| `_bw_schema_simple_mode` | string | 'yes'/'no' for editor mode |
| `_bw_schema_multiple_authors` | array | Multiple author data |
| `_bw_schema_show_author_box` | string | 'yes'/'no' |
| `_bw_schema_disable_default_author` | string | '1' to hide theme author |
| `_bw_schema_is_team_page` | string | 'yes' for team page |
| `_bw_schema_about_entities` | array | Main topic entities |
| `_bw_schema_fact_checked_by` | string | Fact checker name |
| `_bw_schema_last_reviewed` | string | Review date |

### Multiple Authors Structure

```php
[
    [
        'type' => 'wordpress',
        'wordpress_user_id' => 1
    ],
    [
        'type' => 'custom',
        'custom_author_id' => 'john-smith_123456'
    ],
    [
        'type' => 'external',
        'external' => [
            'name' => 'Guest Author',
            'job_title' => 'Contributor',
            'bio' => '...',
            'image' => 'https://...',
            'email' => 'guest@example.com',
            'website' => 'https://...',
            'linkedin' => 'https://...',
            'twitter' => 'https://...',
            'expertise' => 'Topic 1, Topic 2'
        ]
    ]
]
```

---

## Developer Hooks

### Schema Modification Filters

```php
// Modify specific schema type
add_filter('bw_schema_article', function($schema, $post, $type) {
    $schema['customProperty'] = 'value';
    return $schema;
}, 10, 3);

// Modify all schemas before output
add_filter('bw_schema_rendered_schemas', function($schemas) {
    // $schemas is array of schema arrays
    return $schemas;
});

// Modify schema types available
add_filter('bw_schema_types', function($types) {
    $types['custom'] = [
        'label' => 'Custom Type',
        'subtypes' => ['SubType1' => 'Sub Type 1']
    ];
    return $types;
});

// Modify AI properties
add_filter('bw_schema_ai_properties', function($properties) {
    return $properties;
});
```

### Team Page Filters

```php
// Add custom URL patterns for team detection
add_filter('bw_schema_team_url_patterns', function($patterns) {
    $patterns[] = '/notre-equipe'; // French
    $patterns[] = '/unser-team';   // German
    return $patterns;
});

// Add custom shortcodes for team detection
add_filter('bw_schema_team_shortcodes', function($shortcodes) {
    $shortcodes[] = 'my_custom_team_grid';
    return $shortcodes;
});
```

### Helper Methods

```php
// Add custom schema type
BW_Schema_Hooks::add_schema_type(
    'podcast',           // ID
    'Podcast',           // Label
    'For podcast episodes', // Description
    'dashicons-microphone', // Icon
    ['PodcastEpisode' => 'Episode'] // Subtypes
);

// Add AI property
BW_Schema_Hooks::add_ai_property(
    'credibility',       // Category
    'verifiedBy',        // Key
    'Verified By'        // Label
);
```

### Actions

```php
// After schema is rendered
do_action('bw_schema_after_render');

// Before schema is generated for a post
do_action('bw_schema_before_generate', $post);
```

---

## Schema Output

### Output Location
- Hook: `wp_head`
- Priority: 1 (very early)

### Output Format
```html
<!-- BW AI Schema Pro - Structured Data for AI Era -->
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "WebSite",
  "name": "Site Name",
  "url": "https://example.com"
}
</script>
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "Article",
  "headline": "...",
  "author": {...}
}
</script>
<!-- / BW AI Schema Pro -->
```

### Context Detection

| Context | Method | Schema Output |
|---------|--------|---------------|
| Homepage | `is_front_page()` | WebSite or Organization |
| Single Post | `is_singular('post')` | Article + Breadcrumb |
| Single Page | `is_singular('page')` | WebPage + Breadcrumb |
| Author Archive | `is_author()` | ProfilePage with Person |
| Category | `is_category()` | CollectionPage |
| Team Page | Custom detection | ItemList with Person |

---

## Conflict Management

### Supported Sources

| Plugin | Hooks Disabled |
|--------|----------------|
| Yoast SEO | `wpseo_json_ld_output`, `wpseo_schema_graph_pieces` |
| All in One SEO | `aioseo_schema_disable`, `aioseo_schema_graphs` |
| SEOPress | `seopress_schemas_jsonld_output` |
| Rank Math | `rank_math/json_ld`, `rank_math/snippet` |
| WooCommerce | `woocommerce_structured_data_*` |
| Themes | `theme_structured_data`, `get_schema_markup` |

### Implementation
```php
// In BW_Schema_Core::disable_conflicting_schema()
$sources_to_disable = get_option('bw_schema_disable_sources', []);

if (in_array('yoast', $sources_to_disable)) {
    add_filter('wpseo_json_ld_output', '__return_empty_array', 999);
    add_filter('wpseo_schema_graph_pieces', '__return_empty_array', 999);
}
```

---

## Caching

### Implementation
- Uses WordPress transients
- Key format: `bw_schema_post_{post_id}`
- Expiration: 24 hours

### Cache Invalidation
- On post save/update
- On settings update
- On author meta update
- Manual via admin action

---

## Security

### Nonce Verification
All admin forms use WordPress nonces:
```php
wp_nonce_field('bw_schema_save_settings', 'bw_schema_nonce');
// Verification
if (!wp_verify_nonce($_POST['bw_schema_nonce'], 'bw_schema_save_settings')) {
    wp_die('Security check failed');
}
```

### Capability Checks
```php
if (!current_user_can('manage_options')) {
    wp_die('Unauthorized');
}
```

### Output Escaping
- Text: `esc_html()`
- Attributes: `esc_attr()`
- URLs: `esc_url()`
- HTML content: `wp_kses_post()`

---

## Performance Considerations

### Large Files
These files are candidates for splitting:
- `setup-wizard.php` (58KB)
- `class-bw-schema-admin.php` (54KB)
- `class-bw-schema-author-override.php` (42KB)

### Output Buffering
`BW_Schema_Author_Override` uses output buffering as fallback:
- Started at `get_header` hook
- Ended at `wp_footer` hook
- Can impact performance on large pages

### Admin Asset Loading
Assets are conditionally loaded:
```php
if ($screen->id === 'settings_page_bw-ai-schema') {
    wp_enqueue_style('bw-schema-admin');
    wp_enqueue_script('bw-schema-admin');
}
```
