/**
 * BW AI Schema Pro - Frontend Multiple Authors Display
 */
(function() {
    'use strict';
    
    // Helper function to create linked author names
    function createLinkedAuthors() {
        if (!bwSchemaAuthors.authorsInfo || bwSchemaAuthors.authorsInfo.length === 0) {
            return bwSchemaAuthors.formattedNames;
        }
        
        var linkedAuthors = [];
        bwSchemaAuthors.authorsInfo.forEach(function(author) {
            if (author.url && author.url !== '') {
                // Fix any URL issues (like triple slashes)
                var cleanUrl = author.url.replace(/([^:]\/)\/+/g, '$1');
                linkedAuthors.push('<a href="' + cleanUrl + '" rel="author" target="_blank">' + author.name + '</a>');
            } else {
                linkedAuthors.push(author.name);
            }
        });
        
        // Format the linked authors
        if (linkedAuthors.length === 1) {
            return linkedAuthors[0];
        } else if (linkedAuthors.length === 2) {
            return linkedAuthors[0] + ' and ' + linkedAuthors[1];
        } else if (linkedAuthors.length === 3) {
            return linkedAuthors[0] + ', ' + linkedAuthors[1] + ' and ' + linkedAuthors[2];
        } else {
            var othersCount = linkedAuthors.length - 2;
            return linkedAuthors[0] + ', ' + linkedAuthors[1] + ' and ' + othersCount + ' ' + (othersCount === 1 ? 'other' : 'others');
        }
    }
    
    // Wait for DOM to be ready
    function updateMultipleAuthors() {
        // Check if we have multiple authors data
        if (typeof bwSchemaAuthors === 'undefined' || !bwSchemaAuthors.hasMultiple) {
            return;
        }
        
        var linkedAuthorsHTML = createLinkedAuthors();
        console.log('[BW Schema] Attempting to update authors with links:', linkedAuthorsHTML);
        
        // Method 1: Update the entire posted-by span content
        var postedBySpans = document.querySelectorAll('.entry-meta .posted-by');
        postedBySpans.forEach(function(span) {
            // Find the author link within
            var authorLink = span.querySelector('.author.vcard a');
            if (authorLink) {
                // Store the original href
                var originalHref = authorLink.href;
                
                // Replace the entire author vcard content
                var authorVcard = span.querySelector('.author.vcard');
                if (authorVcard) {
                    authorVcard.innerHTML = linkedAuthorsHTML;
                    console.log('[BW Schema] Updated author vcard with links');
                }
            }
        });
        
        // Method 2: If Method 1 didn't work, try more specific selectors
        var authorVcards = document.querySelectorAll('.entry-meta .posted-by .author.vcard');
        authorVcards.forEach(function(vcard) {
            // Replace entire vcard content with linked authors
            vcard.innerHTML = linkedAuthorsHTML;
            console.log('[BW Schema] Replaced author vcard content');
        });
    }
    
    // Multiple attempts to ensure we catch the content
    function tryMultipleTimes() {
        updateMultipleAuthors();
        
        // Try again after various delays
        setTimeout(updateMultipleAuthors, 100);
        setTimeout(updateMultipleAuthors, 500);
        setTimeout(updateMultipleAuthors, 1000);
        setTimeout(updateMultipleAuthors, 2000);
    }
    
    // Run when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', tryMultipleTimes);
    } else {
        tryMultipleTimes();
    }
    
    // Also listen for any dynamic content changes
    if (window.MutationObserver) {
        var observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    var hasAuthorContent = false;
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1 && (node.classList && node.classList.contains('entry-meta') || node.querySelector && node.querySelector('.posted-by'))) {
                            hasAuthorContent = true;
                        }
                    });
                    if (hasAuthorContent) {
                        console.log('[BW Schema] Detected new author content, updating...');
                        updateMultipleAuthors();
                    }
                }
            });
        });
        
        // Start observing when body is available
        function startObserving() {
            if (document.body) {
                observer.observe(document.body, { childList: true, subtree: true });
            } else {
                setTimeout(startObserving, 100);
            }
        }
        startObserving();
    }
})();