#!/bin/bash
# BW AI Schema Pro - Release Script
# Usage: ./release.sh [version]
# Example: ./release.sh 2.1.0

set -e

PLUGIN_DIR="/srv/apps/bwgeo/wp-content/plugins/bw-ai-schema-pro"
UPDATE_DIR="/srv/apps/bwgeo/wp-content/uploads/plugin-updates"
PLUGIN_SLUG="bw-ai-schema-pro"

# Get version from argument or read from plugin file
if [ -n "$1" ]; then
    VERSION="$1"
else
    VERSION=$(grep "Version:" "$PLUGIN_DIR/$PLUGIN_SLUG.php" | head -1 | sed 's/.*Version: *//' | tr -d '[:space:]')
fi

echo "=== Releasing $PLUGIN_SLUG v$VERSION ==="

# Verify version in plugin file matches
PLUGIN_VERSION=$(grep "Version:" "$PLUGIN_DIR/$PLUGIN_SLUG.php" | head -1 | sed 's/.*Version: *//' | tr -d '[:space:]')
if [ "$VERSION" != "$PLUGIN_VERSION" ]; then
    echo "ERROR: Version mismatch!"
    echo "  Argument: $VERSION"
    echo "  Plugin file: $PLUGIN_VERSION"
    echo "Please update the version in $PLUGIN_SLUG.php first."
    exit 1
fi

# Create zip (excluding dev files)
echo "Creating zip archive..."
cd /srv/apps/bwgeo/wp-content/plugins
zip -rq "/tmp/$PLUGIN_SLUG-$VERSION.zip" $PLUGIN_SLUG \
    -x "$PLUGIN_SLUG/docs/*" \
    -x "$PLUGIN_SLUG/.git/*" \
    -x "$PLUGIN_SLUG/debug-*.php" \
    -x "$PLUGIN_SLUG/test-*.php"

# Copy to updates directory
cp "/tmp/$PLUGIN_SLUG-$VERSION.zip" "$UPDATE_DIR/"
rm "/tmp/$PLUGIN_SLUG-$VERSION.zip"

# Update JSON metadata
cat > "$UPDATE_DIR/$PLUGIN_SLUG.json" << EOFJ
{
    "name": "BW AI Schema Pro",
    "version": "$VERSION",
    "download_url": "https://bwgeo.demoing.info/wp-content/uploads/plugin-updates/$PLUGIN_SLUG-$VERSION.zip",
    "author": "Adi Pramono",
    "author_homepage": "https://bowdenworks.com",
    "requires": "5.0",
    "tested": "6.7",
    "requires_php": "7.4",
    "sections": {
        "description": "Advanced Schema Markup plugin optimized for AI era - Enhance your content's visibility and understanding by AI systems with comprehensive schema markup.",
        "changelog": "<h4>$VERSION</h4><ul><li>See CHANGELOG.md for details</li></ul>"
    }
}
EOFJ

# Fix permissions
chmod 644 "$UPDATE_DIR/$PLUGIN_SLUG-$VERSION.zip"
chmod 644 "$UPDATE_DIR/$PLUGIN_SLUG.json"

echo ""
echo "=== Release Complete ==="
echo "Version: $VERSION"
echo "Zip: $UPDATE_DIR/$PLUGIN_SLUG-$VERSION.zip"
echo "JSON: $UPDATE_DIR/$PLUGIN_SLUG.json"
echo ""
echo "Client sites will see the update within 12 hours (or force check in wp-admin)."
