<?php
/**
 * Author Profiles Admin Page (v2.0 Architecture)
 *
 * Shows team member authors and external authors in a unified view.
 *
 * @package BW_AI_Schema_Pro
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get team post type
$team_post_type = BW_Schema_Team_Member::get_team_post_type();

// Get team members flagged as authors
$team_member_authors = array();
if ( $team_post_type ) {
	$team_member_authors = get_posts( array(
		'post_type'      => $team_post_type,
		'posts_per_page' => -1,
		'meta_query'     => array(
			array(
				'key'   => '_bw_schema_is_author',
				'value' => '1',
			),
		),
		'orderby'        => 'title',
		'order'          => 'ASC',
	) );
}

// Get external authors (stored in plugin option)
$external_authors = get_option( 'bw_schema_external_authors', array() );

// Handle search
$search_query = isset( $_GET['s'] ) ? sanitize_text_field( $_GET['s'] ) : '';
if ( $search_query ) {
	$team_member_authors = array_filter( $team_member_authors, function( $member ) use ( $search_query ) {
		$job_title_data = BW_Schema_Team_Member::detect_job_title( $member );
		$job_title      = is_array( $job_title_data ) ? ( $job_title_data['value'] ?? '' ) : $job_title_data;
		return stripos( $member->post_title, $search_query ) !== false ||
		       stripos( $job_title, $search_query ) !== false;
	} );

	$external_authors = array_filter( $external_authors, function( $author ) use ( $search_query ) {
		return stripos( $author['name'] ?? '', $search_query ) !== false ||
		       stripos( $author['jobTitle'] ?? '', $search_query ) !== false;
	} );
}

// Get external author to edit
$edit_author    = null;
$edit_author_id = null;
if ( isset( $_GET['action'] ) && $_GET['action'] === 'edit' && isset( $_GET['external_id'] ) ) {
	$edit_author_id = sanitize_text_field( $_GET['external_id'] );
	if ( isset( $external_authors[ $edit_author_id ] ) ) {
		$edit_author = $external_authors[ $edit_author_id ];
	}
}

// Get plugin-wide default author (v2.0)
$plugin_default_author = get_option( 'bw_schema_plugin_default_author', array() );
$plugin_default_type   = $plugin_default_author['type'] ?? '';
$plugin_default_id     = $plugin_default_author['id'] ?? '';

// Get default author display name for the info box
$default_author_display = '';
if ( $plugin_default_type === 'team_member' && $plugin_default_id ) {
	$default_team_member = get_post( intval( $plugin_default_id ) );
	if ( $default_team_member ) {
		$job_title_data = BW_Schema_Team_Member::detect_job_title( $default_team_member );
		$job_title      = is_array( $job_title_data ) ? ( $job_title_data['value'] ?? '' ) : $job_title_data;
		$default_author_display = $default_team_member->post_title . ( $job_title ? ' (' . $job_title . ')' : '' );
	}
} elseif ( $plugin_default_type === 'external' && $plugin_default_id && isset( $external_authors[ $plugin_default_id ] ) ) {
	$ext = $external_authors[ $plugin_default_id ];
	$default_author_display = $ext['name'] . ( ! empty( $ext['jobTitle'] ) ? ' (' . $ext['jobTitle'] . ')' : '' );
}
?>

<div class="wrap bw-schema-admin-wrap">
	<h1><?php _e( 'BW AI Schema Pro', 'bw-ai-schema-pro' ); ?></h1>

	<!-- Tab Navigation -->
	<nav class="nav-tab-wrapper bw-schema-nav-tabs">
		<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema' ); ?>" class="nav-tab">
			<span class="dashicons dashicons-dashboard"></span>
			<?php _e( 'Dashboard', 'bw-ai-schema-pro' ); ?>
		</a>
		<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-settings' ); ?>" class="nav-tab">
			<span class="dashicons dashicons-admin-generic"></span>
			<?php _e( 'Settings', 'bw-ai-schema-pro' ); ?>
		</a>
		<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-authors' ); ?>" class="nav-tab nav-tab-active">
			<span class="dashicons dashicons-admin-users"></span>
			<?php _e( 'Authors', 'bw-ai-schema-pro' ); ?>
		</a>
		<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-setup' ); ?>" class="nav-tab">
			<span class="dashicons dashicons-welcome-learn-more"></span>
			<?php _e( 'Setup Wizard', 'bw-ai-schema-pro' ); ?>
		</a>
	</nav>

	<!-- Plugin-wide Default Author Section (v2.0) -->
	<div class="bw-schema-default-author-section" style="margin: 20px 0; padding: 20px; background: linear-gradient(135deg, #f0f6fc 0%, #e8f4f8 100%); border: 1px solid #c3c4c7; border-left: 4px solid #2271b1; border-radius: 4px;">
		<form method="post" action="" style="margin: 0;">
			<?php wp_nonce_field( 'bw_schema_plugin_default_author', 'bw_schema_plugin_default_author_nonce' ); ?>
			<input type="hidden" name="bw_schema_save_plugin_default" value="1">

			<div style="display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 15px;">
				<div style="flex: 1; min-width: 200px;">
					<h3 style="margin: 0 0 5px 0; font-size: 14px; font-weight: 600;">
						<span class="dashicons dashicons-admin-users" style="vertical-align: middle; color: #2271b1;"></span>
						<?php _e( 'Plugin Default Author', 'bw-ai-schema-pro' ); ?>
					</h3>
					<p class="description" style="margin: 0; color: #50575e;">
						<?php _e( 'Used when a post has no author assigned and the WordPress user has no personal default set.', 'bw-ai-schema-pro' ); ?>
					</p>
				</div>

				<div style="display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
					<select name="bw_schema_plugin_default_type" id="plugin-default-type" style="min-width: 150px;">
						<option value=""><?php _e( '-- No Default --', 'bw-ai-schema-pro' ); ?></option>
						<?php if ( ! empty( $team_member_authors ) ) : ?>
							<option value="team_member" <?php selected( $plugin_default_type, 'team_member' ); ?>><?php _e( 'Team Member', 'bw-ai-schema-pro' ); ?></option>
						<?php endif; ?>
						<?php if ( ! empty( $external_authors ) ) : ?>
							<option value="external" <?php selected( $plugin_default_type, 'external' ); ?>><?php _e( 'External Author', 'bw-ai-schema-pro' ); ?></option>
						<?php endif; ?>
					</select>

					<!-- Team Member Dropdown -->
					<select name="bw_schema_plugin_default_team_member" id="plugin-default-team-member" style="min-width: 200px; display: <?php echo $plugin_default_type === 'team_member' ? 'inline-block' : 'none'; ?>;">
						<option value=""><?php _e( '-- Select --', 'bw-ai-schema-pro' ); ?></option>
						<?php foreach ( $team_member_authors as $member ) :
							$job_title_data = BW_Schema_Team_Member::detect_job_title( $member );
							$job_title      = is_array( $job_title_data ) ? ( $job_title_data['value'] ?? '' ) : $job_title_data;
						?>
							<option value="<?php echo esc_attr( $member->ID ); ?>" <?php selected( $plugin_default_type === 'team_member' ? $plugin_default_id : '', $member->ID ); ?>>
								<?php echo esc_html( $member->post_title ); ?>
								<?php if ( $job_title ) : ?> - <?php echo esc_html( $job_title ); ?><?php endif; ?>
							</option>
						<?php endforeach; ?>
					</select>

					<!-- External Author Dropdown -->
					<select name="bw_schema_plugin_default_external" id="plugin-default-external" style="min-width: 200px; display: <?php echo $plugin_default_type === 'external' ? 'inline-block' : 'none'; ?>;">
						<option value=""><?php _e( '-- Select --', 'bw-ai-schema-pro' ); ?></option>
						<?php foreach ( $external_authors as $ext_id => $ext ) : ?>
							<option value="<?php echo esc_attr( $ext_id ); ?>" <?php selected( $plugin_default_type === 'external' ? $plugin_default_id : '', $ext_id ); ?>>
								<?php echo esc_html( $ext['name'] ); ?>
								<?php if ( ! empty( $ext['jobTitle'] ) ) : ?> - <?php echo esc_html( $ext['jobTitle'] ); ?><?php endif; ?>
							</option>
						<?php endforeach; ?>
					</select>

					<button type="submit" class="button button-primary"><?php _e( 'Save Default', 'bw-ai-schema-pro' ); ?></button>
				</div>
			</div>

			<?php if ( $default_author_display ) : ?>
			<div style="margin-top: 10px; padding: 8px 12px; background: #fff; border-radius: 3px; display: inline-block;">
				<span style="color: #2271b1;">
					<span class="dashicons dashicons-yes-alt" style="vertical-align: middle;"></span>
					<?php printf( __( 'Current default: <strong>%s</strong>', 'bw-ai-schema-pro' ), esc_html( $default_author_display ) ); ?>
				</span>
			</div>
			<?php endif; ?>
		</form>
	</div>

	<script>
	jQuery(document).ready(function($) {
		$('#plugin-default-type').on('change', function() {
			var type = $(this).val();
			$('#plugin-default-team-member, #plugin-default-external').hide();
			if (type === 'team_member') {
				$('#plugin-default-team-member').show();
			} else if (type === 'external') {
				$('#plugin-default-external').show();
			}
		});
	});
	</script>

	<!-- Search Box -->
	<form method="get" action="" style="margin-top: 20px;">
		<input type="hidden" name="page" value="bw-ai-schema-authors">
		<p class="search-box">
			<label class="screen-reader-text" for="author-search-input">Search Authors:</label>
			<input type="search" id="author-search-input" name="s" value="<?php echo esc_attr( $search_query ); ?>">
			<input type="submit" id="search-submit" class="button" value="Search Authors">
		</p>
	</form>

	<?php
	// Migration Tool for Legacy Custom Authors
	$legacy_custom_authors = get_option( 'bw_schema_custom_authors', array() );
	if ( ! empty( $legacy_custom_authors ) ) :
		// Count posts using each legacy author
		$legacy_author_post_counts = array();
		foreach ( $legacy_custom_authors as $author ) {
			if ( empty( $author['id'] ) ) continue;
			$count = 0;
			// Count posts using this custom author in multiple_authors
			$posts_with_author = get_posts( array(
				'post_type'      => 'any',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => '_bw_schema_multiple_authors',
						'value'   => $author['id'],
						'compare' => 'LIKE',
					),
				),
			) );
			$legacy_author_post_counts[ $author['id'] ] = count( $posts_with_author );
		}

		// Find potential team member matches
		$team_matches = array();
		if ( $team_post_type ) {
			$all_team_members = get_posts( array(
				'post_type'      => $team_post_type,
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			) );
			foreach ( $all_team_members as $tm ) {
				$team_matches[ strtolower( trim( $tm->post_title ) ) ] = $tm->ID;
			}
		}
	?>
	<!-- Legacy Custom Authors Migration Tool -->
	<div class="bw-schema-migration-section" style="margin: 20px 0; padding: 20px; background: #fcf9e8; border: 1px solid #dba617; border-left: 4px solid #dba617; border-radius: 4px;">
		<h2 style="margin: 0 0 15px 0; color: #8a6d14;">
			<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
			<?php _e( 'Migrate Legacy Custom Authors', 'bw-ai-schema-pro' ); ?>
		</h2>
		<p class="description" style="margin-bottom: 15px;">
			<?php printf( __( 'You have %d legacy Custom Authors from the old system. Migrate them to the v2.0 architecture by converting each to a Team Member or External Author.', 'bw-ai-schema-pro' ), count( $legacy_custom_authors ) ); ?>
		</p>

		<form method="post" action="">
			<?php wp_nonce_field( 'bw_schema_migrate_authors', 'bw_schema_migrate_authors_nonce' ); ?>
			<input type="hidden" name="bw_schema_migrate_authors" value="1">

			<table class="wp-list-table widefat fixed striped" style="background: #fff;">
				<thead>
					<tr>
						<th style="width: 25%;"><?php _e( 'Legacy Author', 'bw-ai-schema-pro' ); ?></th>
						<th style="width: 10%; text-align: center;"><?php _e( 'Posts', 'bw-ai-schema-pro' ); ?></th>
						<th style="width: 20%;"><?php _e( 'Action', 'bw-ai-schema-pro' ); ?></th>
						<th style="width: 45%;"><?php _e( 'Target', 'bw-ai-schema-pro' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $legacy_custom_authors as $idx => $author ) :
						if ( empty( $author['id'] ) ) continue;
						$author_id = $author['id'];
						$author_name = $author['name'] ?? 'Unknown';
						$author_job = $author['jobTitle'] ?? '';
						$post_count = $legacy_author_post_counts[ $author_id ] ?? 0;

						// Check for auto-match
						$name_lower = strtolower( trim( $author_name ) );
						$auto_match = isset( $team_matches[ $name_lower ] ) ? $team_matches[ $name_lower ] : '';
					?>
					<tr>
						<td>
							<strong><?php echo esc_html( $author_name ); ?></strong>
							<?php if ( $author_job ) : ?>
							<br><small style="color: #666;"><?php echo esc_html( $author_job ); ?></small>
							<?php endif; ?>
						</td>
						<td style="text-align: center;">
							<?php if ( $post_count > 0 ) : ?>
							<span class="count" style="background: #2271b1; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 12px;">
								<?php echo $post_count; ?>
							</span>
							<?php else : ?>
							<span style="color: #999;">0</span>
							<?php endif; ?>
						</td>
						<td>
							<select name="migrate_action[<?php echo esc_attr( $author_id ); ?>]" class="migration-action" data-author="<?php echo esc_attr( $author_id ); ?>" style="width: 100%;">
								<option value="skip"><?php _e( 'Skip (keep legacy)', 'bw-ai-schema-pro' ); ?></option>
								<?php if ( $team_post_type ) : ?>
								<option value="link_team" <?php selected( ! empty( $auto_match ) ); ?>><?php _e( 'Link to Team Member', 'bw-ai-schema-pro' ); ?></option>
								<?php endif; ?>
								<option value="to_external"><?php _e( 'Convert to External Author', 'bw-ai-schema-pro' ); ?></option>
								<option value="delete"><?php _e( 'Delete', 'bw-ai-schema-pro' ); ?></option>
							</select>
						</td>
						<td>
							<?php if ( $team_post_type && ! empty( $all_team_members ) ) : ?>
							<select name="migrate_target[<?php echo esc_attr( $author_id ); ?>]" class="migration-target" data-author="<?php echo esc_attr( $author_id ); ?>" style="width: 100%; display: <?php echo ! empty( $auto_match ) ? 'block' : 'none'; ?>;">
								<option value=""><?php _e( '-- Select Team Member --', 'bw-ai-schema-pro' ); ?></option>
								<?php foreach ( $all_team_members as $tm ) : ?>
								<option value="<?php echo $tm->ID; ?>" <?php selected( $auto_match, $tm->ID ); ?>>
									<?php echo esc_html( $tm->post_title ); ?>
								</option>
								<?php endforeach; ?>
							</select>
							<?php endif; ?>
							<?php if ( ! empty( $auto_match ) ) : ?>
							<small style="color: #00a32a; display: block; margin-top: 5px;">
								<span class="dashicons dashicons-yes" style="font-size: 14px; vertical-align: middle;"></span>
								<?php _e( 'Auto-matched by name!', 'bw-ai-schema-pro' ); ?>
							</small>
							<?php endif; ?>
						</td>
					</tr>
					<?php endforeach; ?>
				</tbody>
			</table>

			<div style="margin-top: 15px; display: flex; gap: 10px; align-items: center;">
				<button type="submit" class="button button-primary">
					<span class="dashicons dashicons-migrate" style="vertical-align: middle;"></span>
					<?php _e( 'Run Migration', 'bw-ai-schema-pro' ); ?>
				</button>
				<span class="description"><?php _e( 'Post references will be automatically updated to use the new author format.', 'bw-ai-schema-pro' ); ?></span>
			</div>
		</form>

		<script>
		jQuery(document).ready(function($) {
			$('.migration-action').on('change', function() {
				var authorId = $(this).data('author');
				var action = $(this).val();
				var targetSelect = $('.migration-target[data-author="' + authorId + '"]');
				if (action === 'link_team') {
					targetSelect.show();
				} else {
					targetSelect.hide();
				}
			});
		});
		</script>
	</div>
	<?php endif; ?>

	<!-- Team Member Authors Section -->
	<div class="bw-schema-authors-section" style="margin: 20px 0;">
		<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
			<h2 style="margin: 0;">
				<span class="dashicons dashicons-groups" style="vertical-align: middle;"></span>
				<?php _e( 'Team Member Authors', 'bw-ai-schema-pro' ); ?>
			</h2>
			<?php if ( $team_post_type ) : ?>
				<a href="<?php echo admin_url( 'post-new.php?post_type=' . $team_post_type ); ?>" class="button button-primary">
					<span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
					<?php _e( 'Add Team Member', 'bw-ai-schema-pro' ); ?>
				</a>
			<?php endif; ?>
		</div>

		<?php if ( ! $team_post_type ) : ?>
			<div class="notice notice-warning" style="margin: 0;">
				<p>
					<strong><?php _e( 'Team Post Type Not Configured', 'bw-ai-schema-pro' ); ?></strong><br>
					<?php _e( 'Please configure a team member post type in the Setup Wizard to use team member authors.', 'bw-ai-schema-pro' ); ?>
					<a href="<?php echo admin_url( 'options-general.php?page=bw-ai-schema-setup&step=6' ); ?>"><?php _e( 'Configure Now', 'bw-ai-schema-pro' ); ?></a>
				</p>
			</div>
		<?php elseif ( empty( $team_member_authors ) ) : ?>
			<div class="notice notice-info" style="margin: 0;">
				<p>
					<?php _e( 'No team members are flagged as authors yet.', 'bw-ai-schema-pro' ); ?><br>
					<?php _e( 'To add an author, edit a team member and check "Is Author" in the Person Schema Settings.', 'bw-ai-schema-pro' ); ?>
				</p>
			</div>
		<?php else : ?>
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<th class="manage-column" style="width: 60px;">Image</th>
						<th class="manage-column">Name</th>
						<th class="manage-column">Job Title</th>
						<th class="manage-column">Linked WP User</th>
						<th class="manage-column" style="width: 120px;">Actions</th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $team_member_authors as $member ) :
						$job_title_data = BW_Schema_Team_Member::detect_job_title( $member );
						$job_title      = is_array( $job_title_data ) ? ( $job_title_data['value'] ?? '' ) : $job_title_data;
						$image_url      = get_the_post_thumbnail_url( $member, 'thumbnail' );
						$linked_user    = get_post_meta( $member->ID, '_bw_schema_linked_user', true );
						$user_info      = $linked_user ? get_userdata( $linked_user ) : null;
						?>
						<tr>
							<td>
								<?php if ( $image_url ) : ?>
									<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $member->post_title ); ?>" style="width: 50px; height: 50px; object-fit: cover; border-radius: 50%;">
								<?php else : ?>
									<div style="width: 50px; height: 50px; background: #ddd; border-radius: 50%; display: flex; align-items: center; justify-content: center;">
										<span style="font-size: 20px; color: #666;"><?php echo esc_html( substr( $member->post_title, 0, 1 ) ); ?></span>
									</div>
								<?php endif; ?>
							</td>
							<td>
								<strong><?php echo esc_html( $member->post_title ); ?></strong>
								<?php
								$honorific_suffix = get_post_meta( $member->ID, '_bw_schema_honorific_suffix', true );
								if ( $honorific_suffix ) :
									?>
									<span style="color: #666;"><?php echo esc_html( $honorific_suffix ); ?></span>
								<?php endif; ?>
								<br>
								<a href="<?php echo get_permalink( $member ); ?>" target="_blank" style="font-size: 12px;">View Profile</a>
							</td>
							<td><?php echo esc_html( $job_title ); ?></td>
							<td>
								<?php if ( $user_info ) : ?>
									<span class="dashicons dashicons-yes-alt" style="color: #46b450; vertical-align: middle;"></span>
									<?php echo esc_html( $user_info->display_name ); ?>
									<br><small style="color: #666;"><?php echo esc_html( $user_info->user_email ); ?></small>
								<?php else : ?>
									<span style="color: #999;">—</span>
								<?php endif; ?>
							</td>
							<td>
								<a href="<?php echo admin_url( 'post.php?post=' . $member->ID . '&action=edit' ); ?>" class="button button-small">
									<?php _e( 'Edit', 'bw-ai-schema-pro' ); ?>
								</a>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		<?php endif; ?>

		<p class="description" style="margin-top: 10px;">
			<span class="dashicons dashicons-info-outline" style="vertical-align: middle;"></span>
			<?php _e( 'Team member authors are managed from their team member profile. Check "Is Author" on any team member to make them available as a post author.', 'bw-ai-schema-pro' ); ?>
		</p>
	</div>

	<!-- External Authors Section -->
	<div class="bw-schema-authors-section" style="margin: 30px 0;">
		<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
			<h2 style="margin: 0;">
				<span class="dashicons dashicons-businessman" style="vertical-align: middle;"></span>
				<?php _e( 'External Authors', 'bw-ai-schema-pro' ); ?>
			</h2>
			<a href="#add-external-author" class="button button-secondary">
				<span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
				<?php _e( 'Add External Author', 'bw-ai-schema-pro' ); ?>
			</a>
		</div>

		<p class="description" style="margin-bottom: 15px;">
			<?php _e( 'External authors are for guest contributors who don\'t have a team member profile. They\'re reusable across multiple posts.', 'bw-ai-schema-pro' ); ?>
		</p>

		<?php if ( empty( $external_authors ) ) : ?>
			<div class="notice notice-info" style="margin: 0;">
				<p><?php _e( 'No external authors yet. Add one for guest contributors without team profiles.', 'bw-ai-schema-pro' ); ?></p>
			</div>
		<?php else : ?>
			<form method="post" action="">
				<?php wp_nonce_field( 'bulk-external-authors' ); ?>
				<input type="hidden" name="action" value="bulk-delete-external">

				<table class="wp-list-table widefat fixed striped">
					<thead>
						<tr>
							<td class="manage-column column-cb check-column">
								<input type="checkbox" id="cb-select-all-external">
							</td>
							<th class="manage-column" style="width: 60px;">Image</th>
							<th class="manage-column">Name</th>
							<th class="manage-column">Job Title</th>
							<th class="manage-column">Website</th>
							<th class="manage-column" style="width: 150px;">Actions</th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $external_authors as $author_id => $author ) : ?>
							<tr>
								<th scope="row" class="check-column">
									<input type="checkbox" name="external_ids[]" value="<?php echo esc_attr( $author_id ); ?>">
								</th>
								<td>
									<?php if ( ! empty( $author['image'] ) ) : ?>
										<img src="<?php echo esc_url( $author['image'] ); ?>" alt="<?php echo esc_attr( $author['name'] ); ?>" style="width: 50px; height: 50px; object-fit: cover; border-radius: 50%;">
									<?php else : ?>
										<div style="width: 50px; height: 50px; background: #e0e0e0; border-radius: 50%; display: flex; align-items: center; justify-content: center;">
											<span style="font-size: 20px; color: #666;"><?php echo esc_html( substr( $author['name'] ?? 'E', 0, 1 ) ); ?></span>
										</div>
									<?php endif; ?>
								</td>
								<td>
									<strong><?php echo esc_html( $author['name'] ?? '' ); ?></strong>
									<?php if ( ! empty( $author['credentials'] ) ) : ?>
										<br><small style="color: #666;"><?php echo esc_html( $author['credentials'] ); ?></small>
									<?php endif; ?>
								</td>
								<td><?php echo esc_html( $author['jobTitle'] ?? '' ); ?></td>
								<td>
									<?php if ( ! empty( $author['website'] ) ) : ?>
										<a href="<?php echo esc_url( $author['website'] ); ?>" target="_blank"><?php echo esc_html( parse_url( $author['website'], PHP_URL_HOST ) ); ?></a>
									<?php else : ?>
										<span style="color: #999;">—</span>
									<?php endif; ?>
								</td>
								<td>
									<a href="?page=bw-ai-schema-authors&action=edit&external_id=<?php echo esc_attr( $author_id ); ?>#add-external-author" class="button button-small"><?php _e( 'Edit', 'bw-ai-schema-pro' ); ?></a>
									<?php
									$delete_url = wp_nonce_url(
										'?page=bw-ai-schema-authors&action=delete-external&external_id=' . $author_id,
										'delete_external_author'
									);
									?>
									<a href="<?php echo esc_url( $delete_url ); ?>" class="button button-small" onclick="return confirm('<?php _e( 'Are you sure you want to delete this external author?', 'bw-ai-schema-pro' ); ?>');"><?php _e( 'Delete', 'bw-ai-schema-pro' ); ?></a>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>

				<div class="tablenav bottom">
					<div class="alignleft actions bulkactions">
						<select name="bulk_action">
							<option value=""><?php _e( 'Bulk Actions', 'bw-ai-schema-pro' ); ?></option>
							<option value="delete"><?php _e( 'Delete', 'bw-ai-schema-pro' ); ?></option>
						</select>
						<input type="submit" class="button action" value="<?php _e( 'Apply', 'bw-ai-schema-pro' ); ?>" onclick="return confirm('<?php _e( 'Are you sure you want to delete selected external authors?', 'bw-ai-schema-pro' ); ?>');">
					</div>
				</div>
			</form>
		<?php endif; ?>
	</div>

	<!-- Add/Edit External Author Form -->
	<div id="add-external-author" style="margin-top: 30px; padding: 20px; background: #fff; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
		<h2><?php echo $edit_author ? __( 'Edit External Author', 'bw-ai-schema-pro' ) : __( 'Add External Author', 'bw-ai-schema-pro' ); ?></h2>

		<p class="description" style="margin-bottom: 20px;">
			<?php _e( 'External authors are for guest contributors who don\'t have a team member profile on your site.', 'bw-ai-schema-pro' ); ?>
		</p>

		<form method="post" action="">
			<?php wp_nonce_field( 'bw_schema_external_author', 'bw_schema_external_author_nonce' ); ?>
			<input type="hidden" name="bw_schema_save_external_author" value="1">
			<?php if ( $edit_author ) : ?>
				<input type="hidden" name="external_author_id" value="<?php echo esc_attr( $edit_author_id ); ?>">
			<?php endif; ?>

			<table class="form-table">
				<tr>
					<th scope="row"><label for="external_name"><?php _e( 'Name', 'bw-ai-schema-pro' ); ?> <span class="required">*</span></label></th>
					<td>
						<input type="text" id="external_name" name="external_name" class="regular-text" required value="<?php echo $edit_author ? esc_attr( $edit_author['name'] ?? '' ) : ''; ?>">
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="external_job_title"><?php _e( 'Job Title', 'bw-ai-schema-pro' ); ?></label></th>
					<td>
						<input type="text" id="external_job_title" name="external_job_title" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['jobTitle'] ?? '' ) : ''; ?>" placeholder="<?php _e( 'e.g., Freelance Writer, Industry Expert', 'bw-ai-schema-pro' ); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="external_description"><?php _e( 'Bio', 'bw-ai-schema-pro' ); ?></label></th>
					<td>
						<textarea id="external_description" name="external_description" rows="4" class="large-text"><?php echo $edit_author ? esc_textarea( $edit_author['description'] ?? '' ) : ''; ?></textarea>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="external_image"><?php _e( 'Profile Image URL', 'bw-ai-schema-pro' ); ?></label></th>
					<td>
						<input type="url" id="external_image" name="external_image" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['image'] ?? '' ) : ''; ?>">
						<button type="button" class="button" id="upload_external_image"><?php _e( 'Upload Image', 'bw-ai-schema-pro' ); ?></button>
						<div id="external_image_preview" style="margin-top: 10px;">
							<?php if ( $edit_author && ! empty( $edit_author['image'] ) ) : ?>
								<img src="<?php echo esc_url( $edit_author['image'] ); ?>" style="max-width: 100px; height: auto; border-radius: 50%;">
							<?php endif; ?>
						</div>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="external_website"><?php _e( 'Website', 'bw-ai-schema-pro' ); ?></label></th>
					<td>
						<input type="url" id="external_website" name="external_website" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['website'] ?? '' ) : ''; ?>" placeholder="https://example.com">
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="external_credentials"><?php _e( 'Credentials', 'bw-ai-schema-pro' ); ?></label></th>
					<td>
						<input type="text" id="external_credentials" name="external_credentials" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['credentials'] ?? '' ) : ''; ?>" placeholder="<?php _e( 'e.g., Ph.D., MBA', 'bw-ai-schema-pro' ); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row"><?php _e( 'Social Profiles', 'bw-ai-schema-pro' ); ?></th>
					<td>
						<p>
							<label for="external_linkedin"><?php _e( 'LinkedIn:', 'bw-ai-schema-pro' ); ?></label><br>
							<input type="url" id="external_linkedin" name="external_linkedin" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['social']['linkedin'] ?? '' ) : ''; ?>" placeholder="https://linkedin.com/in/...">
						</p>
						<p>
							<label for="external_twitter"><?php _e( 'Twitter/X:', 'bw-ai-schema-pro' ); ?></label><br>
							<input type="url" id="external_twitter" name="external_twitter" class="regular-text" value="<?php echo $edit_author ? esc_attr( $edit_author['social']['twitter'] ?? '' ) : ''; ?>" placeholder="https://x.com/...">
						</p>
					</td>
				</tr>
			</table>

			<p class="submit">
				<input type="submit" class="button button-primary" value="<?php echo $edit_author ? __( 'Update External Author', 'bw-ai-schema-pro' ) : __( 'Add External Author', 'bw-ai-schema-pro' ); ?>">
				<?php if ( $edit_author ) : ?>
					<a href="?page=bw-ai-schema-authors" class="button"><?php _e( 'Cancel', 'bw-ai-schema-pro' ); ?></a>
				<?php endif; ?>
			</p>
		</form>
	</div>
</div>

<script>
jQuery(document).ready(function($) {
	// Select all checkboxes for external authors
	$('#cb-select-all-external').change(function() {
		$('input[name="external_ids[]"]').prop('checked', $(this).prop('checked'));
	});

	// Media uploader for external author image
	if (typeof wp !== 'undefined' && typeof wp.media !== 'undefined') {
		$('#upload_external_image').click(function(e) {
			e.preventDefault();

			var mediaUploader = wp.media({
				title: '<?php _e( 'Select Author Image', 'bw-ai-schema-pro' ); ?>',
				button: { text: '<?php _e( 'Use this image', 'bw-ai-schema-pro' ); ?>' },
				multiple: false
			});

			mediaUploader.on('select', function() {
				var attachment = mediaUploader.state().get('selection').first().toJSON();
				$('#external_image').val(attachment.url);
				$('#external_image_preview').html('<img src="' + attachment.url + '" style="max-width: 100px; height: auto; border-radius: 50%;">');
			});

			mediaUploader.open();
		});
	} else {
		$('#upload_external_image').hide();
	}

	// Update preview on URL change
	$('#external_image').on('change blur', function() {
		var url = $(this).val();
		if (url) {
			$('#external_image_preview').html('<img src="' + url + '" style="max-width: 100px; height: auto; border-radius: 50%;" onerror="this.style.display=\'none\'">');
		} else {
			$('#external_image_preview').html('');
		}
	});
});
</script>

<style>
.bw-schema-authors-section {
	background: #fff;
	border: 1px solid #ccd0d4;
	padding: 20px;
	box-shadow: 0 1px 1px rgba(0,0,0,.04);
}

.bw-schema-authors-section h2 {
	font-size: 1.3em;
	font-weight: 600;
}

.required {
	color: #dc3232;
}

.form-table th {
	width: 180px;
}
</style>
