/**
 * BW AI Schema Pro Admin JavaScript
 */

(function($) {
    'use strict';

    // Initialize when DOM is ready
    $(document).ready(function() {
        
        // Fix grid layout issues
        fixGridLayout();
        
        // Initialize tabs
        initializeTabs();
        
        // Initialize repeater fields
        initializeRepeaters();
        
        // Initialize schema type selector
        initializeSchemaTypeSelector();
        
        // Initialize preview functionality
        initializePreview();
        
        // Initialize AI tools
        initializeAITools();
        
        // Initialize post meta box mode switcher
        initializeModeSwitcher();
        
    });
    
    /**
     * Fix grid layout issues
     */
    function fixGridLayout() {
        // Force a reflow to ensure proper layout calculation
        $('.bw-schema-dashboard-grid, .bw-schema-stats-grid').each(function() {
            var $container = $(this);
            
            // Trigger reflow
            $container.hide().show(0);
            
            // Ensure cards wrap properly
            $container.find('.bw-schema-card, .bw-schema-stat-card').each(function() {
                var $card = $(this);
                // Remove any inline styles that might interfere
                $card.css({
                    'width': '',
                    'max-width': ''
                });
            });
        });
        
        // Add a class to indicate layout is ready
        $('.bw-schema-admin-wrap').addClass('layout-ready');
    }
    
    /**
     * Initialize tab functionality
     */
    function initializeTabs() {
        $('.bw-schema-tab-nav a').on('click', function(e) {
            e.preventDefault();
            
            var $this = $(this);
            var target = $this.attr('href');
            
            // Update active tab
            $('.bw-schema-tab-nav a').removeClass('active');
            $this.addClass('active');
            
            // Show corresponding content
            $('.bw-schema-tab-content').removeClass('active');
            $(target).addClass('active');
        });
    }
    
    /**
     * Initialize repeater fields
     */
    function initializeRepeaters() {
        // Add new field
        $('.bw-schema-add-field').on('click', function() {
            var $repeater = $(this).closest('.bw-schema-repeater');
            var $template = $repeater.find('.bw-schema-repeater-item').first().clone();
            
            // Clear values
            $template.find('input').val('');
            
            // Insert before add button
            $template.insertBefore($(this));
            
            // Update field names
            updateRepeaterFieldNames($repeater);
        });
        
        // Remove field
        $(document).on('click', '.bw-schema-remove-field', function() {
            var $repeater = $(this).closest('.bw-schema-repeater');
            var itemCount = $repeater.find('.bw-schema-repeater-item').length;
            
            if (itemCount > 1) {
                $(this).closest('.bw-schema-repeater-item').remove();
                updateRepeaterFieldNames($repeater);
            }
        });
    }
    
    /**
     * Update repeater field names
     */
    function updateRepeaterFieldNames($repeater) {
        $repeater.find('.bw-schema-repeater-item').each(function(index) {
            $(this).find('input').each(function() {
                var name = $(this).attr('name');
                if (name) {
                    // Update array index
                    name = name.replace(/\[\d+\]/, '[' + index + ']');
                    $(this).attr('name', name);
                }
            });
        });
    }
    
    /**
     * Initialize schema type selector
     */
    function initializeSchemaTypeSelector() {
        $('.bw-schema-type-card').on('click', function() {
            var $this = $(this);
            var type = $this.data('type');
            
            // Update selection
            $('.bw-schema-type-card').removeClass('selected');
            $this.addClass('selected');
            
            // Update hidden input
            $('#bw_schema_type').val(type);
            
            // Show type-specific fields
            showTypeSpecificFields(type);
        });
    }
    
    /**
     * Show type-specific fields
     */
    function showTypeSpecificFields(type) {
        // Hide all type-specific fields
        $('.bw-schema-type-fields').hide();
        
        // Show fields for selected type
        $('.bw-schema-type-fields[data-type="' + type + '"]').show();
    }
    
    /**
     * Initialize preview functionality
     */
    function initializePreview() {
        $('#bw-schema-preview-btn').on('click', function() {
            var $button = $(this);
            var $output = $('#bw-schema-preview-output');
            
            // Show loading
            $button.prop('disabled', true).text(bwSchema.i18n.loading || 'Loading...');
            
            // Get form data
            var formData = {
                action: 'bw_schema_preview',
                nonce: bwSchema.nonce,
                post_id: $('#post_ID').val(),
                schema_type: $('#bw_schema_type').val(),
                custom_schema: $('#bw_schema_custom').val()
            };
            
            // Add additional fields
            $('.bw-schema-ai-fields input, .bw-schema-ai-fields textarea').each(function() {
                formData[$(this).attr('name')] = $(this).val();
            });
            
            // Make AJAX request
            $.post(bwSchema.ajaxurl, formData, function(response) {
                if (response.success) {
                    $output.html('<pre>' + JSON.stringify(response.data, null, 2) + '</pre>').slideDown();
                } else {
                    $output.html('<div class="bw-schema-notice error">' + response.data + '</div>').slideDown();
                }
            }).always(function() {
                $button.prop('disabled', false).text(bwSchema.i18n.preview || 'Preview Schema Markup');
            });
        });
    }
    
    /**
     * Initialize AI tools
     */
    function initializeAITools() {
        // Auto-detect schema type
        $('#bw-schema-auto-detect').on('click', function() {
            var $button = $(this);
            var $result = $('#bw-schema-auto-detect-result');
            
            // Show loading
            $button.prop('disabled', true);
            $result.html('<span class="bw-schema-spinner"></span>');
            
            // Get post content
            var content = '';
            if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor) {
                content = tinyMCE.activeEditor.getContent();
            } else {
                content = $('#content').val();
            }
            
            var formData = {
                action: 'bw_schema_auto_detect',
                nonce: bwSchema.nonce,
                content: content,
                title: $('#title').val()
            };
            
            $.post(bwSchema.ajaxurl, formData, function(response) {
                if (response.success) {
                    $result.html('<div class="bw-schema-notice success">Suggested type: <strong>' + response.data.type + '</strong></div>');
                    
                    // Auto-select the type
                    $('.bw-schema-type-card[data-type="' + response.data.type + '"]').click();
                } else {
                    $result.html('<div class="bw-schema-notice error">' + response.data + '</div>');
                }
            }).always(function() {
                $button.prop('disabled', false);
            });
        });
        
        // Extract entities
        $('#bw-schema-extract-entities').on('click', function() {
            var $button = $(this);
            var $result = $('#bw-schema-entities-result');
            
            // Show loading
            $button.prop('disabled', true);
            $result.html('<span class="bw-schema-spinner"></span>');
            
            // Get post content
            var content = '';
            if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor) {
                content = tinyMCE.activeEditor.getContent();
            } else {
                content = $('#content').val();
            }
            
            var formData = {
                action: 'bw_schema_extract_entities',
                nonce: bwSchema.nonce,
                content: content
            };
            
            $.post(bwSchema.ajaxurl, formData, function(response) {
                if (response.success) {
                    // Update entity fields
                    if (response.data.entities) {
                        $('#bw_schema_about_entities').val(response.data.entities.join(', '));
                    }
                    
                    $result.html('<div class="bw-schema-notice success">Entities extracted successfully!</div>');
                } else {
                    $result.html('<div class="bw-schema-notice error">' + response.data + '</div>');
                }
            }).always(function() {
                $button.prop('disabled', false);
            });
        });
        
        // Validate schema
        $('#bw-schema-validate').on('click', function() {
            var $button = $(this);
            var $result = $('#bw-schema-validation-result');
            
            // Show loading
            $button.prop('disabled', true);
            $result.html('<span class="bw-schema-spinner"></span>');
            
            var schemaData = $('#bw_schema_custom').val();
            
            if (!schemaData) {
                $result.html('<div class="bw-schema-notice error">Please enter schema JSON-LD to validate.</div>');
                $button.prop('disabled', false);
                return;
            }
            
            try {
                // Basic JSON validation
                var parsed = JSON.parse(schemaData);
                
                // Make AJAX request for advanced validation
                var formData = {
                    action: 'bw_schema_validate',
                    nonce: bwSchema.nonce,
                    schema: schemaData
                };
                
                $.post(bwSchema.ajaxurl, formData, function(response) {
                    if (response.success) {
                        $result.html('<div class="bw-schema-notice success">Schema is valid!</div>');
                    } else {
                        $result.html('<div class="bw-schema-notice error">Validation errors: ' + response.data + '</div>');
                    }
                }).always(function() {
                    $button.prop('disabled', false);
                });
                
            } catch (e) {
                $result.html('<div class="bw-schema-notice error">Invalid JSON: ' + e.message + '</div>');
                $button.prop('disabled', false);
            }
        });
    }
    
    /**
     * Helper function to show notifications
     */
    function showNotification(message, type) {
        var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
        $('.wrap h1').after($notice);
        
        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            $notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    /**
     * Initialize mode switcher in post meta box
     */
    function initializeModeSwitcher() {
        // Handle mode switching
        $('input[name="bw_schema_simple_mode"]').on('change', function() {
            var mode = $(this).val();
            
            if (mode === 'yes') {
                $('#bw-schema-simple-mode').slideDown();
                $('#bw-schema-advanced-mode').slideUp();
            } else {
                $('#bw-schema-simple-mode').slideUp();
                $('#bw-schema-advanced-mode').slideDown();
            }
        });
        
        // Handle schema type change to load simple fields
        $('#bw_schema_type').on('change', function() {
            var type = $(this).val();
            var postId = $('#post_ID').val();
            var $fieldsContainer = $('#bw-schema-simple-fields');
            
            if (!type) {
                $fieldsContainer.html('');
                return;
            }
            
            // Show loading
            $fieldsContainer.html('<p>Loading fields...</p>');
            
            // Load fields via AJAX
            $.post(bwSchema.ajaxurl, {
                action: 'bw_schema_get_simple_fields',
                nonce: bwSchema.nonce,
                type: type,
                post_id: postId
            }, function(response) {
                if (response.success) {
                    $fieldsContainer.html(response.data);
                } else {
                    $fieldsContainer.html('<p class="error">Error loading fields.</p>');
                }
            });
        });
    }

    /**
     * Initialize completion widget toggle
     */
    function initializeCompletionWidget() {
        // Toggle completed items section
        $('.bw-schema-completion-widget .checklist-section.complete .toggle-section').on('click', function() {
            var $section = $(this).closest('.checklist-section');
            var $list = $section.find('ul');
            var $icon = $(this).find('.dashicons');

            if ($list.is(':visible')) {
                $list.slideUp(200);
                $section.removeClass('expanded').addClass('collapsed');
            } else {
                $list.slideDown(200);
                $section.removeClass('collapsed').addClass('expanded');
            }
        });
    }

    // Initialize completion widget on document ready
    $(document).ready(function() {
        initializeCompletionWidget();
    });

})(jQuery);