<?php
/**
 * Team Page Detection for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Team_Detector {
	
	/**
	 * Check if current page is a team page
	 * 
	 * @return bool
	 */
	public static function is_team_page() {
		// Get team post type setting
		$team_post_type = get_option( 'bw_schema_team_post_type', 'sw_teammember' );
		
		// Check if it's a team post type archive
		if ( is_post_type_archive( $team_post_type ) ) {
			return true;
		}
		
		// Check if it's a page with team-related URL or slug
		if ( is_page() ) {
			global $post;
			
			// Get the current page URL
			$current_url = get_permalink();
			
			// Get configured team page detection patterns
			$team_url_patterns = self::get_team_url_patterns();
			
			// Check URL patterns
			foreach ( $team_url_patterns as $pattern ) {
				if ( strpos( $current_url, $pattern ) !== false ) {
					return true;
				}
			}
			
			// Check page slug
			if ( $post && isset( $post->post_name ) ) {
				$team_slug_patterns = self::get_team_slug_patterns();
				foreach ( $team_slug_patterns as $pattern ) {
					if ( strpos( $post->post_name, $pattern ) !== false ) {
						return true;
					}
				}
			}
			
			// Check if page has a specific meta flag
			if ( $post && get_post_meta( $post->ID, '_bw_schema_is_team_page', true ) === 'yes' ) {
				return true;
			}
			
			// Check if page contains team member query blocks or shortcodes
			if ( $post && self::page_contains_team_content( $post ) ) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Get team URL patterns
	 * 
	 * @return array
	 */
	private static function get_team_url_patterns() {
		$default_patterns = array(
			'/team',
			'/about/team',
			'/our-team',
			'/meet-the-team',
			'/staff',
			'/about-us/team',
			'/company/team'
		);
		
		return apply_filters( 'bw_schema_team_url_patterns', $default_patterns );
	}
	
	/**
	 * Get team slug patterns
	 * 
	 * @return array
	 */
	private static function get_team_slug_patterns() {
		$default_patterns = array(
			'team',
			'our-team',
			'meet-the-team',
			'staff',
			'employees',
			'team-members'
		);
		
		return apply_filters( 'bw_schema_team_slug_patterns', $default_patterns );
	}
	
	/**
	 * Check if page contains team content
	 * 
	 * @param WP_Post $post
	 * @return bool
	 */
	private static function page_contains_team_content( $post ) {
		if ( ! $post || empty( $post->post_content ) ) {
			return false;
		}
		
		$content = $post->post_content;
		$team_post_type = get_option( 'bw_schema_team_post_type', 'sw_teammember' );
		
		// Check for query loop blocks with team post type
		if ( has_blocks( $content ) ) {
			$blocks = parse_blocks( $content );
			if ( self::contains_team_query_block( $blocks, $team_post_type ) ) {
				return true;
			}
		}
		
		// Check for team-related shortcodes
		$team_shortcodes = apply_filters( 'bw_schema_team_shortcodes', array(
			'team',
			'team_members',
			'staff_grid',
			'employee_list',
			$team_post_type
		) );
		
		foreach ( $team_shortcodes as $shortcode ) {
			if ( has_shortcode( $content, $shortcode ) ) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Recursively check blocks for team query
	 * 
	 * @param array $blocks
	 * @param string $team_post_type
	 * @return bool
	 */
	private static function contains_team_query_block( $blocks, $team_post_type ) {
		foreach ( $blocks as $block ) {
			// Check if it's a query loop block
			if ( $block['blockName'] === 'core/query' ) {
				if ( isset( $block['attrs']['query']['postType'] ) && 
				     $block['attrs']['query']['postType'] === $team_post_type ) {
					return true;
				}
			}
			
			// Check inner blocks
			if ( ! empty( $block['innerBlocks'] ) ) {
				if ( self::contains_team_query_block( $block['innerBlocks'], $team_post_type ) ) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * Add meta box for team page designation
	 */
	public static function add_team_page_meta_box() {
		add_meta_box(
			'bw_schema_team_page',
			__( 'Team Page Settings', 'bw-ai-schema-pro' ),
			array( __CLASS__, 'render_team_page_meta_box' ),
			'page',
			'side',
			'default'
		);
	}
	
	/**
	 * Render team page meta box
	 */
	public static function render_team_page_meta_box( $post ) {
		wp_nonce_field( 'bw_schema_team_page', 'bw_schema_team_page_nonce' );
		
		$is_team_page = get_post_meta( $post->ID, '_bw_schema_is_team_page', true );
		?>
		<p>
			<label>
				<input type="checkbox" name="bw_schema_is_team_page" value="yes" <?php checked( $is_team_page, 'yes' ); ?> />
				<?php _e( 'This is a team page', 'bw-ai-schema-pro' ); ?>
			</label>
		</p>
		<p class="description">
			<?php _e( 'Check this box if this page displays your team members and should use team schema markup.', 'bw-ai-schema-pro' ); ?>
		</p>
		<?php
	}
	
	/**
	 * Save team page meta
	 */
	public static function save_team_page_meta( $post_id ) {
		if ( ! isset( $_POST['bw_schema_team_page_nonce'] ) || 
		     ! wp_verify_nonce( $_POST['bw_schema_team_page_nonce'], 'bw_schema_team_page' ) ) {
			return;
		}
		
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}
		
		if ( isset( $_POST['bw_schema_is_team_page'] ) ) {
			update_post_meta( $post_id, '_bw_schema_is_team_page', 'yes' );
		} else {
			delete_post_meta( $post_id, '_bw_schema_is_team_page' );
		}
	}
	
	/**
	 * Initialize team detector
	 */
	public static function init() {
		// Add meta box for pages
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_team_page_meta_box' ) );
		
		// Save meta box data
		add_action( 'save_post_page', array( __CLASS__, 'save_team_page_meta' ) );
	}
}

// Initialize
BW_Schema_Team_Detector::init();