<?php
/**
 * Helper Functions for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Helpers {
	
	/**
	 * Create a tooltip
	 */
	public static function tooltip( $text, $icon = 'dashicons-editor-help' ) {
		return sprintf(
			'<span class="bw-tooltip">
				<span class="dashicons %s"></span>
				<span class="bw-tooltip-content">%s</span>
			</span>',
			esc_attr( $icon ),
			esc_html( $text )
		);
	}
	
	/**
	 * Create help text
	 */
	public static function help_text( $text ) {
		return sprintf(
			'<p class="bw-help-text">%s</p>',
			wp_kses_post( $text )
		);
	}
	
	/**
	 * Get schema type label
	 */
	public static function get_schema_type_label( $type ) {
		$types = BW_Schema_Core::get_schema_types();
		list( $main_type, $sub_type ) = explode( ':', $type );
		
		if ( isset( $types[$main_type]['subtypes'][$sub_type] ) ) {
			return $types[$main_type]['subtypes'][$sub_type];
		}
		
		return ucfirst( $sub_type );
	}
	
	/**
	 * Get schema type icon
	 */
	public static function get_schema_type_icon( $type ) {
		$types = BW_Schema_Core::get_schema_types();
		$main_type = explode( ':', $type )[0];
		
		if ( isset( $types[$main_type]['icon'] ) ) {
			return $types[$main_type]['icon'];
		}
		
		return 'dashicons-media-default';
	}
	
	/**
	 * Format schema JSON for display
	 */
	public static function format_schema_json( $schema ) {
		return wp_json_encode( $schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );
	}
	
	/**
	 * Check if post type supports schema
	 */
	public static function post_type_supports_schema( $post_type ) {
		$supported = get_option( 'bw_schema_supported_post_types', array( 'post', 'page' ) );
		return in_array( $post_type, $supported );
	}
	
	/**
	 * Get available social platforms
	 */
	public static function get_social_platforms() {
		return array(
			'facebook' => array(
				'label' => 'Facebook',
				'icon' => 'dashicons-facebook',
				'placeholder' => 'https://facebook.com/yourpage'
			),
			'twitter' => array(
				'label' => 'X (Twitter)',
				'icon' => 'dashicons-twitter',
				'placeholder' => 'https://x.com/yourprofile'
			),
			'instagram' => array(
				'label' => 'Instagram',
				'icon' => 'dashicons-instagram',
				'placeholder' => 'https://instagram.com/yourprofile'
			),
			'linkedin' => array(
				'label' => 'LinkedIn',
				'icon' => 'dashicons-linkedin',
				'placeholder' => 'https://linkedin.com/in/yourprofile'
			),
			'youtube' => array(
				'label' => 'YouTube',
				'icon' => 'dashicons-youtube',
				'placeholder' => 'https://youtube.com/c/yourchannel'
			),
			'pinterest' => array(
				'label' => 'Pinterest',
				'icon' => 'dashicons-pinterest',
				'placeholder' => 'https://pinterest.com/yourprofile'
			),
			'github' => array(
				'label' => 'GitHub',
				'icon' => 'dashicons-editor-code',
				'placeholder' => 'https://github.com/yourprofile'
			),
			'tiktok' => array(
				'label' => 'TikTok',
				'icon' => 'dashicons-video-alt3',
				'placeholder' => 'https://tiktok.com/@yourprofile'
			),
			'snapchat' => array(
				'label' => 'Snapchat',
				'icon' => 'dashicons-camera',
				'placeholder' => 'https://snapchat.com/add/yourprofile'
			),
			'reddit' => array(
				'label' => 'Reddit',
				'icon' => 'dashicons-reddit',
				'placeholder' => 'https://reddit.com/user/yourprofile'
			),
			'tumblr' => array(
				'label' => 'Tumblr',
				'icon' => 'dashicons-tumblr',
				'placeholder' => 'https://yourprofile.tumblr.com'
			),
			'vimeo' => array(
				'label' => 'Vimeo',
				'icon' => 'dashicons-video-alt2',
				'placeholder' => 'https://vimeo.com/yourprofile'
			),
			'medium' => array(
				'label' => 'Medium',
				'icon' => 'dashicons-edit-page',
				'placeholder' => 'https://medium.com/@yourprofile'
			),
			'quora' => array(
				'label' => 'Quora',
				'icon' => 'dashicons-editor-help',
				'placeholder' => 'https://quora.com/profile/yourprofile'
			),
			'whatsapp' => array(
				'label' => 'WhatsApp',
				'icon' => 'dashicons-whatsapp',
				'placeholder' => 'https://wa.me/yourphonenumber'
			),
			'telegram' => array(
				'label' => 'Telegram',
				'icon' => 'dashicons-email-alt',
				'placeholder' => 'https://t.me/yourprofile'
			),
			'discord' => array(
				'label' => 'Discord',
				'icon' => 'dashicons-microphone',
				'placeholder' => 'https://discord.gg/yourserver'
			),
			'twitch' => array(
				'label' => 'Twitch',
				'icon' => 'dashicons-video-alt',
				'placeholder' => 'https://twitch.tv/yourprofile'
			),
			'spotify' => array(
				'label' => 'Spotify',
				'icon' => 'dashicons-playlist-audio',
				'placeholder' => 'https://open.spotify.com/user/yourprofile'
			),
			'soundcloud' => array(
				'label' => 'SoundCloud',
				'icon' => 'dashicons-format-audio',
				'placeholder' => 'https://soundcloud.com/yourprofile'
			),
			'behance' => array(
				'label' => 'Behance',
				'icon' => 'dashicons-art',
				'placeholder' => 'https://behance.net/yourprofile'
			),
			'dribbble' => array(
				'label' => 'Dribbble',
				'icon' => 'dashicons-art',
				'placeholder' => 'https://dribbble.com/yourprofile'
			),
			'flickr' => array(
				'label' => 'Flickr',
				'icon' => 'dashicons-camera-alt',
				'placeholder' => 'https://flickr.com/people/yourprofile'
			),
			'500px' => array(
				'label' => '500px',
				'icon' => 'dashicons-camera',
				'placeholder' => 'https://500px.com/p/yourprofile'
			),
			'vk' => array(
				'label' => 'VK',
				'icon' => 'dashicons-share',
				'placeholder' => 'https://vk.com/yourprofile'
			),
			'wechat' => array(
				'label' => 'WeChat',
				'icon' => 'dashicons-format-chat',
				'placeholder' => 'Your WeChat ID'
			),
			'xing' => array(
				'label' => 'XING',
				'icon' => 'dashicons-networking',
				'placeholder' => 'https://xing.com/profile/yourprofile'
			),
			'yelp' => array(
				'label' => 'Yelp',
				'icon' => 'dashicons-star-filled',
				'placeholder' => 'https://yelp.com/biz/yourbusiness'
			),
			'tripadvisor' => array(
				'label' => 'TripAdvisor',
				'icon' => 'dashicons-location',
				'placeholder' => 'https://tripadvisor.com/yourbusiness'
			),
			'foursquare' => array(
				'label' => 'Foursquare',
				'icon' => 'dashicons-location-alt',
				'placeholder' => 'https://foursquare.com/v/yourvenue'
			),
			'angellist' => array(
				'label' => 'AngelList',
				'icon' => 'dashicons-businessman',
				'placeholder' => 'https://angel.co/yourprofile'
			),
			'productHunt' => array(
				'label' => 'Product Hunt',
				'icon' => 'dashicons-megaphone',
				'placeholder' => 'https://producthunt.com/@yourprofile'
			),
			'gitlab' => array(
				'label' => 'GitLab',
				'icon' => 'dashicons-editor-code',
				'placeholder' => 'https://gitlab.com/yourprofile'
			),
			'bitbucket' => array(
				'label' => 'Bitbucket',
				'icon' => 'dashicons-editor-code',
				'placeholder' => 'https://bitbucket.org/yourprofile'
			),
			'stackoverflow' => array(
				'label' => 'Stack Overflow',
				'icon' => 'dashicons-editor-code',
				'placeholder' => 'https://stackoverflow.com/users/yourid'
			),
			'researchgate' => array(
				'label' => 'ResearchGate',
				'icon' => 'dashicons-analytics',
				'placeholder' => 'https://researchgate.net/profile/yourprofile'
			),
			'academia' => array(
				'label' => 'Academia.edu',
				'icon' => 'dashicons-welcome-learn-more',
				'placeholder' => 'https://youruni.academia.edu/yourname'
			),
			'orcid' => array(
				'label' => 'ORCID',
				'icon' => 'dashicons-id',
				'placeholder' => 'https://orcid.org/0000-0000-0000-0000'
			),
			'googlescholar' => array(
				'label' => 'Google Scholar',
				'icon' => 'dashicons-welcome-learn-more',
				'placeholder' => 'https://scholar.google.com/citations?user=yourid'
			),
			'custom' => array(
				'label' => 'Custom URL',
				'icon' => 'dashicons-admin-links',
				'placeholder' => 'https://yourwebsite.com'
			)
		);
	}
	
	/**
	 * Get business types
	 */
	public static function get_business_types() {
		return array(
			'general' => __( 'General Business', 'bw-ai-schema-pro' ),
			'consulting' => __( 'Consulting / Professional Services', 'bw-ai-schema-pro' ),
			'technology' => __( 'Technology / Software', 'bw-ai-schema-pro' ),
			'retail' => __( 'Retail / E-commerce', 'bw-ai-schema-pro' ),
			'hospitality' => __( 'Hospitality / Tourism', 'bw-ai-schema-pro' ),
			'medical' => __( 'Medical / Healthcare', 'bw-ai-schema-pro' ),
			'legal' => __( 'Legal Services', 'bw-ai-schema-pro' ),
			'financial' => __( 'Financial Services', 'bw-ai-schema-pro' ),
			'realestate' => __( 'Real Estate', 'bw-ai-schema-pro' ),
			'education' => __( 'Education / Training', 'bw-ai-schema-pro' ),
			'nonprofit' => __( 'Non-Profit Organization', 'bw-ai-schema-pro' ),
			'entertainment' => __( 'Entertainment / Media', 'bw-ai-schema-pro' )
		);
	}
	
	/**
	 * Sanitize schema data
	 */
	public static function sanitize_schema_data( $data ) {
		if ( is_array( $data ) ) {
			foreach ( $data as $key => $value ) {
				$data[$key] = self::sanitize_schema_data( $value );
			}
		} else {
			$data = sanitize_text_field( $data );
		}
		
		return $data;
	}
	
	/**
	 * Get schema validation URL
	 */
	public static function get_validation_url( $url = '' ) {
		if ( empty( $url ) ) {
			$url = home_url();
		}
		
		return 'https://validator.schema.org/#url=' . urlencode( $url );
	}
	
	/**
	 * Get Rich Results Test URL
	 */
	public static function get_rich_results_url( $url = '' ) {
		if ( empty( $url ) ) {
			$url = home_url();
		}
		
		return 'https://search.google.com/test/rich-results?url=' . urlencode( $url );
	}
	
	/**
	 * Get all posts organized by post type for select dropdowns
	 */
	public static function get_posts_for_select() {
		$results = array();
		
		// Get all public post types
		$args = array(
			'public' => true,
			'show_ui' => true // Ensure it has a UI
		);
		
		$post_types = get_post_types( $args, 'objects' );
		
		// Sort post types alphabetically
		uasort( $post_types, function( $a, $b ) {
			return strcmp( $a->labels->name, $b->labels->name );
		} );
		
		foreach ( $post_types as $post_type ) {
			// Skip attachments
			if ( $post_type->name === 'attachment' ) {
				continue;
			}
			
			// Use direct database query for better performance with custom post types
			global $wpdb;
			
			$posts = $wpdb->get_results( $wpdb->prepare(
				"SELECT ID, post_title 
				FROM {$wpdb->posts} 
				WHERE post_type = %s 
				AND post_status = 'publish' 
				ORDER BY post_title ASC 
				LIMIT 100",
				$post_type->name
			) );
			
			if ( ! empty( $posts ) ) {
				$results[$post_type->name] = array(
					'label' => $post_type->labels->name,
					'singular' => $post_type->labels->singular_name,
					'posts' => $posts
				);
			}
		}
		
		return $results;
	}
	
	/**
	 * Render post select options grouped by post type
	 */
	public static function render_post_select_options( $selected = '', $show_post_type = true ) {
		$posts_by_type = self::get_posts_for_select();
		
		foreach ( $posts_by_type as $post_type => $data ) {
			$label = $show_post_type ? $data['label'] . ' (' . $post_type . ')' : $data['label'];
			echo '<optgroup label="' . esc_attr( $label ) . '">';
			
			foreach ( $data['posts'] as $post ) {
				echo '<option value="' . esc_attr( $post->ID ) . '" ' . selected( $selected, $post->ID, false ) . '>';
				echo esc_html( $post->post_title );
				echo '</option>';
			}
			
			echo '</optgroup>';
		}
	}
}