<?php
/**
 * Schema Cache Handler for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Cache {
	
	/**
	 * Cache group
	 */
	const CACHE_GROUP = 'bw_schema_cache';
	
	/**
	 * Cache expiration (24 hours)
	 */
	const CACHE_EXPIRATION = DAY_IN_SECONDS;
	
	/**
	 * Get cached schema
	 */
	public static function get( $key ) {
		// Check if caching is enabled
		if ( ! self::is_enabled() ) {
			return false;
		}
		
		return wp_cache_get( $key, self::CACHE_GROUP );
	}
	
	/**
	 * Set cached schema
	 */
	public static function set( $key, $data ) {
		// Check if caching is enabled
		if ( ! self::is_enabled() ) {
			return false;
		}
		
		return wp_cache_set( $key, $data, self::CACHE_GROUP, self::CACHE_EXPIRATION );
	}
	
	/**
	 * Delete cached schema
	 */
	public static function delete( $key ) {
		return wp_cache_delete( $key, self::CACHE_GROUP );
	}
	
	/**
	 * Clear all schema cache
	 */
	public static function clear_all() {
		wp_cache_flush();
	}
	
	/**
	 * Clear post cache
	 */
	public static function clear_post_cache( $post_id ) {
		// Clear specific post cache
		self::delete( 'post_' . $post_id );
		
		// Clear author cache
		$post = get_post( $post_id );
		if ( $post ) {
			self::delete( 'author_' . $post->post_author );
		}
		
		// Clear organization cache
		self::delete( 'organization' );
		self::delete( 'website' );
	}
	
	/**
	 * Generate cache key
	 */
	public static function get_cache_key( $type, $id = null ) {
		if ( $id ) {
			return $type . '_' . $id;
		}
		return $type;
	}
	
	/**
	 * Check if caching is enabled
	 */
	public static function is_enabled() {
		// Check if caching is disabled in settings
		if ( get_option( 'bw_schema_enable_cache', 'yes' ) === 'no' ) {
			return false;
		}
		
		// Don't cache in development environments
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			return false;
		}
		
		// Don't cache for logged-in users (admins)
		if ( is_user_logged_in() && current_user_can( 'manage_options' ) ) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Hook into WordPress actions
	 */
	public static function init() {
		// Clear cache when posts are saved
		add_action( 'save_post', array( __CLASS__, 'clear_post_cache' ) );
		add_action( 'delete_post', array( __CLASS__, 'clear_post_cache' ) );
		
		// Clear cache when plugin settings are updated
		add_action( 'update_option_bw_schema_organization', array( __CLASS__, 'clear_all' ) );
		add_action( 'update_option_bw_schema_enable_schema', array( __CLASS__, 'clear_all' ) );
		add_action( 'update_option_bw_schema_enable_breadcrumbs', array( __CLASS__, 'clear_all' ) );
		add_action( 'update_option_bw_schema_post_type_defaults', array( __CLASS__, 'clear_all' ) );
		
		// Clear cache when user meta is updated
		add_action( 'updated_user_meta', array( __CLASS__, 'clear_author_cache' ), 10, 3 );
	}
	
	/**
	 * Clear author cache when user meta is updated
	 */
	public static function clear_author_cache( $meta_id, $user_id, $meta_key ) {
		if ( $meta_key === '_bw_schema_author_data' ) {
			self::delete( 'author_' . $user_id );
		}
	}
}