<?php
/**
 * Test script for BW Schema Pro author display issue
 * 
 * Usage: Navigate to this file in your browser with ?post_id=XXX
 */

// Load WordPress
require_once( dirname( __FILE__ ) . '/../../../wp-load.php' );

// Check if user is admin
if ( ! current_user_can( 'manage_options' ) ) {
    wp_die( 'Access denied.' );
}

$post_id = isset( $_GET['post_id'] ) ? intval( $_GET['post_id'] ) : 0;

if ( ! $post_id ) {
    wp_die( 'Please provide a post_id parameter.' );
}

// Get the post
$post = get_post( $post_id );
if ( ! $post ) {
    wp_die( 'Post not found.' );
}

// Include necessary classes
require_once dirname( __FILE__ ) . '/includes/class-bw-schema-author-override.php';

// Get multiple authors data
$multiple_authors = get_post_meta( $post_id, '_bw_schema_multiple_authors', true );
$custom_authors_list = get_option( 'bw_schema_custom_authors', array() );

// Build author names array manually
$author_names = array();
$missing_authors = array();

if ( ! empty( $multiple_authors ) && is_array( $multiple_authors ) ) {
    foreach ( $multiple_authors as $index => $author_data ) {
        $name = null;
        $found = false;
        
        switch ( $author_data['type'] ) {
            case 'wordpress':
                if ( ! empty( $author_data['wordpress_user_id'] ) ) {
                    $user = get_userdata( $author_data['wordpress_user_id'] );
                    if ( $user ) {
                        $name = $user->display_name;
                        $found = true;
                    }
                }
                break;
                
            case 'custom':
                if ( ! empty( $author_data['custom_author_id'] ) ) {
                    foreach ( $custom_authors_list as $custom_author ) {
                        if ( $custom_author['id'] === $author_data['custom_author_id'] ) {
                            $name = $custom_author['name'];
                            $found = true;
                            break;
                        }
                    }
                }
                break;
                
            case 'external':
                if ( ! empty( $author_data['external']['name'] ) ) {
                    $name = $author_data['external']['name'];
                    $found = true;
                }
                break;
        }
        
        if ( $name ) {
            $author_names[] = $name;
        } else {
            $missing_authors[] = array(
                'index' => $index + 1,
                'type' => $author_data['type'],
                'data' => $author_data
            );
        }
    }
}

// Test the format_author_names function
$reflection = new ReflectionClass( 'BW_Schema_Author_Override' );
$method = $reflection->getMethod( 'format_author_names' );
$method->setAccessible( true );

$formatted_simple = $method->invoke( null, $author_names );

?>
<!DOCTYPE html>
<html>
<head>
    <title>Author Display Test</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
        table { border-collapse: collapse; width: 100%; margin: 20px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        .code { background: #f0f0f0; padding: 10px; font-family: monospace; }
    </style>
</head>
<body>
    <h1>Author Display Test</h1>
    <h2>Post: <?php echo esc_html( $post->post_title ); ?></h2>
    
    <h3>Summary</h3>
    <table>
        <tr>
            <th>Total Authors Configured</th>
            <td><?php echo count( $multiple_authors ); ?></td>
        </tr>
        <tr>
            <th>Authors Successfully Retrieved</th>
            <td class="<?php echo count( $author_names ) === count( $multiple_authors ) ? 'success' : 'error'; ?>">
                <?php echo count( $author_names ); ?>
            </td>
        </tr>
        <tr>
            <th>Missing Authors</th>
            <td class="<?php echo count( $missing_authors ) > 0 ? 'error' : 'success'; ?>">
                <?php echo count( $missing_authors ); ?>
            </td>
        </tr>
        <tr>
            <th>Expected Display</th>
            <td>
                <?php if ( count( $multiple_authors ) <= 5 ) : ?>
                    All <?php echo count( $multiple_authors ); ?> authors should be shown
                <?php else : ?>
                    First 3 authors + "and <?php echo count( $multiple_authors ) - 3; ?> others"
                <?php endif; ?>
            </td>
        </tr>
        <tr>
            <th>Actual Display</th>
            <td class="<?php echo strpos( $formatted_simple, 'others' ) !== false && count( $multiple_authors ) <= 5 ? 'error' : 'success'; ?>">
                <?php echo esc_html( $formatted_simple ); ?>
            </td>
        </tr>
    </table>
    
    <h3>Author Details</h3>
    <table>
        <tr>
            <th>#</th>
            <th>Type</th>
            <th>Name</th>
            <th>Status</th>
            <th>Details</th>
        </tr>
        <?php foreach ( $multiple_authors as $index => $author_data ) : ?>
            <?php
            $name = isset( $author_names[$index] ) ? $author_names[$index] : null;
            $is_missing = false;
            foreach ( $missing_authors as $missing ) {
                if ( $missing['index'] - 1 === $index ) {
                    $is_missing = true;
                    break;
                }
            }
            ?>
            <tr>
                <td><?php echo $index + 1; ?></td>
                <td><?php echo esc_html( $author_data['type'] ); ?></td>
                <td class="<?php echo $is_missing ? 'error' : 'success'; ?>">
                    <?php echo $name ? esc_html( $name ) : 'NOT FOUND'; ?>
                </td>
                <td><?php echo $is_missing ? '❌ Missing' : '✅ Found'; ?></td>
                <td>
                    <?php
                    switch ( $author_data['type'] ) {
                        case 'wordpress':
                            echo 'User ID: ' . ( $author_data['wordpress_user_id'] ?? 'Not set' );
                            break;
                        case 'custom':
                            echo 'Custom ID: ' . ( $author_data['custom_author_id'] ?? 'Not set' );
                            break;
                        case 'external':
                            echo 'External Name: ' . ( $author_data['external']['name'] ?? 'Not set' );
                            break;
                    }
                    ?>
                </td>
            </tr>
        <?php endforeach; ?>
    </table>
    
    <?php if ( ! empty( $missing_authors ) ) : ?>
        <h3 class="error">Missing Authors Details</h3>
        <?php foreach ( $missing_authors as $missing ) : ?>
            <div style="border: 1px solid red; padding: 10px; margin: 10px 0;">
                <h4>Author <?php echo $missing['index']; ?> (<?php echo $missing['type']; ?>)</h4>
                <pre><?php print_r( $missing['data'] ); ?></pre>
                
                <?php if ( $missing['type'] === 'custom' && ! empty( $missing['data']['custom_author_id'] ) ) : ?>
                    <p><strong>Looking for Custom Author ID:</strong> <?php echo $missing['data']['custom_author_id']; ?></p>
                    <p><strong>Available Custom Author IDs:</strong></p>
                    <ul>
                        <?php foreach ( $custom_authors_list as $ca ) : ?>
                            <li><?php echo $ca['id']; ?> - <?php echo $ca['name']; ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
    
    <h3>Diagnostic Information</h3>
    <ul>
        <li><strong>Custom Authors Enabled:</strong> <?php echo get_option( 'bw_schema_use_custom_authors', 'no' ); ?></li>
        <li><strong>Total Custom Authors in Database:</strong> <?php echo count( $custom_authors_list ); ?></li>
        <li><strong>LiteSpeed Cache Active:</strong> <?php echo is_plugin_active( 'litespeed-cache/litespeed-cache.php' ) ? 'Yes' : 'No'; ?></li>
    </ul>
    
    <h3>Recommendations</h3>
    <ol>
        <?php if ( is_plugin_active( 'litespeed-cache/litespeed-cache.php' ) ) : ?>
            <li class="warning">LiteSpeed Cache is active. Try clearing the cache.</li>
        <?php endif; ?>
        <?php if ( ! empty( $missing_authors ) ) : ?>
            <li class="error">Some authors could not be found. Check the missing authors section above.</li>
        <?php endif; ?>
        <li>Check the browser console for JavaScript errors.</li>
        <li>View page source and search for the author names to see if they're in the HTML but hidden by CSS.</li>
    </ol>
</body>
</html>