<?php
/**
 * Review Schema Implementation
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Review extends BW_Schema_Base {
	
	/**
	 * Generate review schema
	 */
	public function generate( $post, $type = 'Review' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema = array_merge( $schema, $this->get_common_properties( $post ) );
		
		// Add review-specific properties
		$schema = array_merge( $schema, $this->get_review_properties( $post ) );
		
		// Add author
		$schema['author'] = $this->get_author_properties( $post->post_author, $post );
		
		// Add publisher
		$schema['publisher'] = $this->get_organization_properties();
		
		return apply_filters( 'bw_schema_review', $schema, $post, $this );
	}
	
	/**
	 * Get review-specific properties
	 */
	private function get_review_properties( $post ) {
		$properties = array(
			'reviewBody' => wp_strip_all_tags( apply_filters( 'the_content', $post->post_content ) )
		);
		
		// Add review rating if available
		$rating = get_post_meta( $post->ID, '_bw_schema_review_rating', true );
		if ( $rating ) {
			$properties['reviewRating'] = array(
				'@type' => 'Rating',
				'ratingValue' => floatval( $rating ),
				'bestRating' => 5,
				'worstRating' => 1
			);
		}
		
		// Add item reviewed
		$item_type = get_post_meta( $post->ID, '_bw_schema_review_item_type', true );
		$item_name = get_post_meta( $post->ID, '_bw_schema_review_item_name', true );
		
		if ( ! $item_name ) {
			// Extract from title if possible (e.g., "Review of Product Name")
			$title = get_the_title( $post->ID );
			if ( preg_match( '/review of (.+)/i', $title, $matches ) ) {
				$item_name = $matches[1];
			} else {
				$item_name = $title;
			}
		}
		
		if ( $item_name ) {
			$properties['itemReviewed'] = array(
				'@type' => $item_type ?: 'Thing',
				'name' => $item_name
			);
			
			// Add additional properties based on item type
			if ( $item_type === 'Product' ) {
				$brand = get_post_meta( $post->ID, '_bw_schema_review_item_brand', true );
				if ( $brand ) {
					$properties['itemReviewed']['brand'] = array(
						'@type' => 'Brand',
						'name' => $brand
					);
				}
			} elseif ( $item_type === 'LocalBusiness' ) {
				$address = get_post_meta( $post->ID, '_bw_schema_review_item_address', true );
				if ( $address ) {
					$properties['itemReviewed']['address'] = $address;
				}
			}
		}
		
		// Add aggregate rating if multiple reviews exist
		$aggregate = $this->get_aggregate_rating( $post );
		if ( $aggregate ) {
			$properties['aggregateRating'] = $aggregate;
		}
		
		return $properties;
	}
	
	/**
	 * Get aggregate rating
	 */
	private function get_aggregate_rating( $post ) {
		// This would typically pull from a review system or comments
		// For now, return null unless we have data
		$review_count = get_post_meta( $post->ID, '_bw_schema_review_count', true );
		$average_rating = get_post_meta( $post->ID, '_bw_schema_average_rating', true );
		
		if ( $review_count && $average_rating ) {
			return array(
				'@type' => 'AggregateRating',
				'ratingValue' => floatval( $average_rating ),
				'reviewCount' => intval( $review_count ),
				'bestRating' => 5,
				'worstRating' => 1
			);
		}
		
		return null;
	}
	
	/**
	 * Get author properties
	 */
	protected function get_author_properties( $author_id, $post = null ) {
		// Check for custom author
		if ( $post ) {
			$custom_author_id = get_post_meta( $post->ID, '_bw_schema_custom_author', true );
			if ( $custom_author_id && get_option( 'bw_schema_use_custom_authors' ) ) {
				$custom_authors = get_option( 'bw_schema_custom_authors', array() );
				foreach ( $custom_authors as $author ) {
					if ( $author['id'] === $custom_author_id ) {
						return $this->format_custom_author( $author );
					}
				}
			}
		}
		
		// Fall back to WordPress user
		$user = get_userdata( $author_id );
		if ( ! $user ) {
			return array(
				'@type' => 'Person',
				'name' => 'Unknown Author'
			);
		}
		
		$author_schema = array(
			'@type' => 'Person',
			'name' => $user->display_name,
			'url' => get_author_posts_url( $user->ID )
		);
		
		if ( $user->description ) {
			$author_schema['description'] = $user->description;
		}
		
		return $author_schema;
	}
	
	/**
	 * Format custom author
	 */
	private function format_custom_author( $author ) {
		$schema = array(
			'@type' => 'Person',
			'name' => $author['name']
		);
		
		if ( ! empty( $author['job_title'] ) ) {
			$schema['jobTitle'] = $author['job_title'];
		}
		
		if ( ! empty( $author['bio'] ) ) {
			$schema['description'] = $author['bio'];
		}
		
		if ( ! empty( $author['image'] ) ) {
			$schema['image'] = $author['image'];
		}
		
		// Add sameAs for social profiles
		$sameAs = array();
		if ( ! empty( $author['linkedin'] ) ) {
			$sameAs[] = $author['linkedin'];
		}
		if ( ! empty( $author['twitter'] ) ) {
			$sameAs[] = $author['twitter'];
		}
		if ( ! empty( $author['website'] ) ) {
			$sameAs[] = $author['website'];
		}
		
		if ( ! empty( $sameAs ) ) {
			$schema['sameAs'] = $sameAs;
		}
		
		return $schema;
	}
	
	/**
	 * Get organization properties
	 */
	protected function get_organization_properties() {
		$org_data = get_option( 'bw_schema_organization', array() );
		
		$organization = array(
			'@type' => 'Organization',
			'name' => ! empty( $org_data['name'] ) ? $org_data['name'] : get_bloginfo( 'name' ),
			'url' => ! empty( $org_data['url'] ) ? $org_data['url'] : home_url()
		);
		
		if ( ! empty( $org_data['logo'] ) ) {
			$organization['logo'] = array(
				'@type' => 'ImageObject',
				'url' => $org_data['logo']
			);
		}
		
		return $organization;
	}
	
	/**
	 * Get subtypes for this schema
	 */
	public static function get_subtypes() {
		return array(
			'Review' => __( 'General Review', 'bw-ai-schema-pro' ),
			'ProductReview' => __( 'Product Review', 'bw-ai-schema-pro' ),
			'ServiceReview' => __( 'Service Review', 'bw-ai-schema-pro' ),
			'RestaurantReview' => __( 'Restaurant Review', 'bw-ai-schema-pro' ),
			'MovieReview' => __( 'Movie Review', 'bw-ai-schema-pro' ),
			'BookReview' => __( 'Book Review', 'bw-ai-schema-pro' )
		);
	}
}