<?php
/**
 * Organization Schema Class for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Organization extends BW_Schema_Base {
	
	/**
	 * Generate organization schema
	 */
	public function generate( $post, $type = 'Organization' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema['name'] = get_the_title( $post );
		$schema['url'] = get_permalink( $post );
		
		// Add description
		if ( has_excerpt( $post ) ) {
			$schema['description'] = wp_strip_all_tags( get_the_excerpt( $post ) );
		}
		
		// Add logo/image
		if ( has_post_thumbnail( $post ) ) {
			$logo_url = get_the_post_thumbnail_url( $post, 'full' );
			$schema['logo'] = array(
				'@type' => 'ImageObject',
				'url' => $logo_url,
			);
			$schema['image'] = $logo_url;
		}
		
		// Add founding date
		$founding_date = get_post_meta( $post->ID, '_bw_schema_founding_date', true );
		if ( ! empty( $founding_date ) ) {
			$schema['foundingDate'] = $founding_date;
		}
		
		// Add founders
		$founders = get_post_meta( $post->ID, '_bw_schema_founders', true );
		if ( ! empty( $founders ) && is_array( $founders ) ) {
			$founder_list = array();
			foreach ( $founders as $founder ) {
				if ( ! empty( $founder ) ) {
					$founder_list[] = array(
						'@type' => 'Person',
						'name' => $founder,
					);
				}
			}
			if ( ! empty( $founder_list ) ) {
				$schema['founder'] = $founder_list;
			}
		}
		
		// Add address
		$address = get_post_meta( $post->ID, '_bw_schema_address', true );
		if ( ! empty( $address ) && is_array( $address ) ) {
			$schema['address'] = array(
				'@type' => 'PostalAddress',
				'streetAddress' => $address['street'] ?? '',
				'addressLocality' => $address['city'] ?? '',
				'addressRegion' => $address['state'] ?? '',
				'postalCode' => $address['zip'] ?? '',
				'addressCountry' => $address['country'] ?? '',
			);
		}
		
		// Add contact info
		$telephone = get_post_meta( $post->ID, '_bw_schema_telephone', true );
		if ( ! empty( $telephone ) ) {
			$schema['telephone'] = $telephone;
		}
		
		$email = get_post_meta( $post->ID, '_bw_schema_email', true );
		if ( ! empty( $email ) ) {
			$schema['email'] = $email;
		}
		
		// Add contact points
		$contact_points = get_post_meta( $post->ID, '_bw_schema_contact_points', true );
		if ( ! empty( $contact_points ) && is_array( $contact_points ) ) {
			$contacts = array();
			foreach ( $contact_points as $contact ) {
				if ( ! empty( $contact['type'] ) && ! empty( $contact['telephone'] ) ) {
					$contact_point = array(
						'@type' => 'ContactPoint',
						'contactType' => $contact['type'],
						'telephone' => $contact['telephone'],
					);
					if ( ! empty( $contact['languages'] ) ) {
						$contact_point['availableLanguage'] = $contact['languages'];
					}
					$contacts[] = $contact_point;
				}
			}
			if ( ! empty( $contacts ) ) {
				$schema['contactPoint'] = $contacts;
			}
		}
		
		// Add social profiles (sameAs)
		$social_profiles = get_post_meta( $post->ID, '_bw_schema_social_profiles', true );
		if ( ! empty( $social_profiles ) && is_array( $social_profiles ) ) {
			$schema['sameAs'] = array_filter( $social_profiles );
		}
		
		// Add awards and certifications (AI-optimized for credibility)
		$awards = get_post_meta( $post->ID, '_bw_schema_awards', true );
		if ( ! empty( $awards ) && is_array( $awards ) ) {
			$schema['award'] = array_filter( $awards );
		}
		
		// Add certifications
		$certifications = get_post_meta( $post->ID, '_bw_schema_certifications', true );
		if ( ! empty( $certifications ) && is_array( $certifications ) ) {
			$cert_list = array();
			foreach ( $certifications as $cert ) {
				if ( ! empty( $cert['name'] ) ) {
					$certification = array(
						'@type' => 'Certification',
						'name' => $cert['name'],
					);
					if ( ! empty( $cert['issuedBy'] ) ) {
						$certification['issuedBy'] = array(
							'@type' => 'Organization',
							'name' => $cert['issuedBy'],
						);
					}
					$cert_list[] = $certification;
				}
			}
			if ( ! empty( $cert_list ) ) {
				$schema['hasCertification'] = $cert_list;
			}
		}
		
		// Add member organizations
		$member_of = get_post_meta( $post->ID, '_bw_schema_member_of', true );
		if ( ! empty( $member_of ) && is_array( $member_of ) ) {
			$members = array();
			foreach ( $member_of as $member ) {
				if ( ! empty( $member ) ) {
					$members[] = array(
						'@type' => 'Organization',
						'name' => $member,
					);
				}
			}
			if ( ! empty( $members ) ) {
				$schema['memberOf'] = $members;
			}
		}
		
		// Add type-specific properties
		switch ( $type ) {
			case 'Corporation':
				$schema = $this->add_corporation_properties( $schema, $post );
				break;
				
			case 'EducationalOrganization':
				$schema = $this->add_educational_properties( $schema, $post );
				break;
				
			case 'GovernmentOrganization':
				$schema = $this->add_government_properties( $schema, $post );
				break;
				
			case 'NGO':
				$schema = $this->add_ngo_properties( $schema, $post );
				break;
		}
		
		// Add AI-optimized properties
		$schema = $this->add_ai_properties( $schema, $post );
		
		// Add aggregate rating if available
		$rating_value = get_post_meta( $post->ID, '_bw_schema_rating_value', true );
		$rating_count = get_post_meta( $post->ID, '_bw_schema_rating_count', true );
		if ( ! empty( $rating_value ) && ! empty( $rating_count ) ) {
			$schema['aggregateRating'] = array(
				'@type' => 'AggregateRating',
				'ratingValue' => floatval( $rating_value ),
				'ratingCount' => intval( $rating_count ),
			);
		}
		
		return apply_filters( 'bw_schema_organization', $schema, $post, $type );
	}
	
	/**
	 * Add corporation-specific properties
	 */
	private function add_corporation_properties( $schema, $post ) {
		// Add ticker symbol
		$ticker = get_post_meta( $post->ID, '_bw_schema_ticker_symbol', true );
		if ( ! empty( $ticker ) ) {
			$schema['tickerSymbol'] = $ticker;
		}
		
		// Add number of employees
		$employees = get_post_meta( $post->ID, '_bw_schema_number_of_employees', true );
		if ( ! empty( $employees ) ) {
			$schema['numberOfEmployees'] = array(
				'@type' => 'QuantitativeValue',
				'value' => intval( $employees ),
			);
		}
		
		return $schema;
	}
	
	/**
	 * Add educational organization properties
	 */
	private function add_educational_properties( $schema, $post ) {
		// Add alumni
		$notable_alumni = get_post_meta( $post->ID, '_bw_schema_notable_alumni', true );
		if ( ! empty( $notable_alumni ) && is_array( $notable_alumni ) ) {
			$alumni = array();
			foreach ( $notable_alumni as $alum ) {
				if ( ! empty( $alum ) ) {
					$alumni[] = array(
						'@type' => 'Person',
						'name' => $alum,
					);
				}
			}
			if ( ! empty( $alumni ) ) {
				$schema['alumni'] = $alumni;
			}
		}
		
		// Add offered courses
		$courses = get_post_meta( $post->ID, '_bw_schema_courses_offered', true );
		if ( ! empty( $courses ) && is_array( $courses ) ) {
			$schema['hasOfferCatalog'] = array(
				'@type' => 'OfferCatalog',
				'name' => 'Course Catalog',
				'itemListElement' => array_map( function( $course ) {
					return array(
						'@type' => 'Course',
						'name' => $course,
					);
				}, array_filter( $courses ) ),
			);
		}
		
		return $schema;
	}
	
	/**
	 * Add government organization properties
	 */
	private function add_government_properties( $schema, $post ) {
		// Add jurisdiction
		$jurisdiction = get_post_meta( $post->ID, '_bw_schema_jurisdiction', true );
		if ( ! empty( $jurisdiction ) ) {
			$schema['areaServed'] = array(
				'@type' => 'AdministrativeArea',
				'name' => $jurisdiction,
			);
		}
		
		// Add government services
		$services = get_post_meta( $post->ID, '_bw_schema_government_services', true );
		if ( ! empty( $services ) && is_array( $services ) ) {
			$schema['hasOfferCatalog'] = array(
				'@type' => 'OfferCatalog',
				'name' => 'Government Services',
				'itemListElement' => array_map( function( $service ) {
					return array(
						'@type' => 'GovernmentService',
						'name' => $service,
					);
				}, array_filter( $services ) ),
			);
		}
		
		return $schema;
	}
	
	/**
	 * Add NGO properties
	 */
	private function add_ngo_properties( $schema, $post ) {
		// Add mission
		$mission = get_post_meta( $post->ID, '_bw_schema_mission', true );
		if ( ! empty( $mission ) ) {
			$schema['mission'] = $mission;
		}
		
		// Add areas of focus
		$focus_areas = get_post_meta( $post->ID, '_bw_schema_focus_areas', true );
		if ( ! empty( $focus_areas ) && is_array( $focus_areas ) ) {
			$schema['knowsAbout'] = array_filter( $focus_areas );
		}
		
		// Add donation information
		$donation_url = get_post_meta( $post->ID, '_bw_schema_donation_url', true );
		if ( ! empty( $donation_url ) ) {
			$schema['potentialAction'] = array(
				'@type' => 'DonateAction',
				'target' => $donation_url,
			);
		}
		
		return $schema;
	}
}