<?php
/**
 * JobPosting Schema Class for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_JobPosting extends BW_Schema_Base {
	
	/**
	 * Generate job posting schema
	 */
	public function generate( $post, $type = 'JobPosting' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema = array_merge( $schema, $this->get_common_properties( $post ) );
		
		// Add title (required)
		$schema['title'] = get_the_title( $post );
		
		// Add hiring organization (required)
		$hiring_org = $this->get_hiring_organization( $post );
		if ( ! empty( $hiring_org ) ) {
			$schema['hiringOrganization'] = $hiring_org;
		}
		
		// Add job location (required)
		$job_location = $this->get_job_location( $post );
		if ( ! empty( $job_location ) ) {
			$schema['jobLocation'] = $job_location;
		}
		
		// Add date posted (required)
		$schema['datePosted'] = get_the_date( 'c', $post );
		
		// Add valid through
		$valid_through = get_post_meta( $post->ID, '_bw_schema_valid_through', true );
		if ( ! empty( $valid_through ) ) {
			$schema['validThrough'] = $valid_through;
		}
		
		// Add employment type
		$employment_type = get_post_meta( $post->ID, '_bw_schema_employment_type', true );
		if ( ! empty( $employment_type ) ) {
			if ( is_array( $employment_type ) ) {
				$schema['employmentType'] = $employment_type;
			} else {
				$schema['employmentType'] = array( $employment_type );
			}
		}
		
		// Add job location type
		$job_location_type = get_post_meta( $post->ID, '_bw_schema_job_location_type', true );
		if ( ! empty( $job_location_type ) && is_array( $job_location_type ) ) {
			$schema['jobLocationType'] = $job_location_type;
		}
		
		// Add base salary
		$base_salary = $this->get_base_salary( $post );
		if ( ! empty( $base_salary ) ) {
			$schema['baseSalary'] = $base_salary;
		}
		
		// Add identifier
		$job_id = get_post_meta( $post->ID, '_bw_schema_job_id', true );
		if ( ! empty( $job_id ) ) {
			$schema['identifier'] = array(
				'@type' => 'PropertyValue',
				'name' => 'Job ID',
				'value' => $job_id,
			);
		}
		
		// Add seniority level
		$seniority_level = get_post_meta( $post->ID, '_bw_schema_seniority_level', true );
		if ( ! empty( $seniority_level ) ) {
			$schema['seniorityLevel'] = $seniority_level;
		}
		
		// Add experience requirements
		$experience_requirements = get_post_meta( $post->ID, '_bw_schema_experience_requirements', true );
		if ( ! empty( $experience_requirements ) ) {
			$schema['experienceRequirements'] = $experience_requirements;
		}
		
		// Add educational requirements
		$education_requirements = get_post_meta( $post->ID, '_bw_schema_education_requirements', true );
		if ( ! empty( $education_requirements ) ) {
			$schema['educationRequirements'] = array(
				'@type' => 'EducationalOccupationalCredential',
				'credentialCategory' => $education_requirements,
			);
		}
		
		// Add qualifications
		$qualifications = get_post_meta( $post->ID, '_bw_schema_qualifications', true );
		if ( ! empty( $qualifications ) ) {
			$schema['qualifications'] = $qualifications;
		}
		
		// Add responsibilities
		$responsibilities = get_post_meta( $post->ID, '_bw_schema_responsibilities', true );
		if ( ! empty( $responsibilities ) ) {
			$schema['responsibilities'] = $responsibilities;
		}
		
		// Add skills
		$skills = get_post_meta( $post->ID, '_bw_schema_skills', true );
		if ( ! empty( $skills ) && is_array( $skills ) ) {
			$skill_list = array();
			foreach ( $skills as $skill ) {
				$skill_list[] = array(
					'@type' => 'DefinedTerm',
					'name' => $skill,
				);
			}
			$schema['skills'] = $skill_list;
		}
		
		// Add incentive compensation
		$incentive_compensation = get_post_meta( $post->ID, '_bw_schema_incentive_compensation', true );
		if ( ! empty( $incentive_compensation ) ) {
			$schema['incentiveCompensation'] = $incentive_compensation;
		}
		
		// Add job benefits
		$job_benefits = get_post_meta( $post->ID, '_bw_schema_job_benefits', true );
		if ( ! empty( $job_benefits ) ) {
			$schema['jobBenefits'] = $job_benefits;
		}
		
		// Add industry
		$industry = get_post_meta( $post->ID, '_bw_schema_industry', true );
		if ( ! empty( $industry ) ) {
			$schema['industry'] = $industry;
		}
		
		// Add occupational category
		$occupational_category = get_post_meta( $post->ID, '_bw_schema_occupational_category', true );
		if ( ! empty( $occupational_category ) ) {
			$schema['occupationalCategory'] = $occupational_category;
		}
		
		// Add work hours
		$work_hours = get_post_meta( $post->ID, '_bw_schema_work_hours', true );
		if ( ! empty( $work_hours ) ) {
			$schema['workHours'] = $work_hours;
		}
		
		// Add application deadline
		$application_deadline = get_post_meta( $post->ID, '_bw_schema_application_deadline', true );
		if ( ! empty( $application_deadline ) ) {
			$schema['applicationDeadline'] = $application_deadline;
		}
		
		// Add job start date
		$job_start_date = get_post_meta( $post->ID, '_bw_schema_job_start_date', true );
		if ( ! empty( $job_start_date ) ) {
			$schema['jobStartDate'] = $job_start_date;
		}
		
		// Add application instructions
		$application_instructions = get_post_meta( $post->ID, '_bw_schema_application_instructions', true );
		if ( ! empty( $application_instructions ) ) {
			$schema['applicationInstructions'] = $application_instructions;
		}
		
		// Add direct apply
		$direct_apply = get_post_meta( $post->ID, '_bw_schema_direct_apply', true );
		if ( $direct_apply === 'yes' ) {
			$schema['directApply'] = true;
		}
		
		// Add in language
		$schema['inLanguage'] = get_locale();
		
		// Add AI-optimized properties
		$schema = $this->add_ai_properties( $schema, $post );
		
		return apply_filters( 'bw_schema_jobposting', $schema, $post, $type );
	}
	
	/**
	 * Get hiring organization
	 */
	private function get_hiring_organization( $post ) {
		$org_name = get_post_meta( $post->ID, '_bw_schema_hiring_organization', true );
		
		if ( empty( $org_name ) ) {
			return $this->get_organization_properties();
		}
		
		$org = array(
			'@type' => 'Organization',
			'name' => $org_name,
		);
		
		$org_url = get_post_meta( $post->ID, '_bw_schema_hiring_organization_url', true );
		if ( ! empty( $org_url ) ) {
			$org['url'] = $org_url;
		}
		
		$org_logo = get_post_meta( $post->ID, '_bw_schema_hiring_organization_logo', true );
		if ( ! empty( $org_logo ) ) {
			$org['logo'] = $org_logo;
		}
		
		$org_same_as = get_post_meta( $post->ID, '_bw_schema_hiring_organization_same_as', true );
		if ( ! empty( $org_same_as ) && is_array( $org_same_as ) ) {
			$org['sameAs'] = array_filter( $org_same_as );
		}
		
		return $org;
	}
	
	/**
	 * Get job location
	 */
	private function get_job_location( $post ) {
		$location = array(
			'@type' => 'Place',
		);
		
		// Check if remote
		$is_remote = get_post_meta( $post->ID, '_bw_schema_is_remote', true );
		if ( $is_remote === 'yes' ) {
			$location['name'] = 'Remote';
			return $location;
		}
		
		// Add address
		$address = array(
			'@type' => 'PostalAddress',
		);
		
		$street = get_post_meta( $post->ID, '_bw_schema_street_address', true );
		if ( ! empty( $street ) ) {
			$address['streetAddress'] = $street;
		}
		
		$city = get_post_meta( $post->ID, '_bw_schema_address_locality', true );
		if ( ! empty( $city ) ) {
			$address['addressLocality'] = $city;
		}
		
		$region = get_post_meta( $post->ID, '_bw_schema_address_region', true );
		if ( ! empty( $region ) ) {
			$address['addressRegion'] = $region;
		}
		
		$postal_code = get_post_meta( $post->ID, '_bw_schema_postal_code', true );
		if ( ! empty( $postal_code ) ) {
			$address['postalCode'] = $postal_code;
		}
		
		$country = get_post_meta( $post->ID, '_bw_schema_address_country', true );
		if ( ! empty( $country ) ) {
			$address['addressCountry'] = $country;
		}
		
		if ( count( $address ) > 1 ) {
			$location['address'] = $address;
		}
		
		// Add geo coordinates if available
		$latitude = get_post_meta( $post->ID, '_bw_schema_latitude', true );
		$longitude = get_post_meta( $post->ID, '_bw_schema_longitude', true );
		
		if ( ! empty( $latitude ) && ! empty( $longitude ) ) {
			$location['geo'] = array(
				'@type' => 'GeoCoordinates',
				'latitude' => floatval( $latitude ),
				'longitude' => floatval( $longitude ),
			);
		}
		
		return $location;
	}
	
	/**
	 * Get base salary
	 */
	private function get_base_salary( $post ) {
		$salary_min = get_post_meta( $post->ID, '_bw_schema_salary_min', true );
		$salary_max = get_post_meta( $post->ID, '_bw_schema_salary_max', true );
		$salary_currency = get_post_meta( $post->ID, '_bw_schema_salary_currency', true );
		$salary_unit = get_post_meta( $post->ID, '_bw_schema_salary_unit', true );
		
		if ( empty( $salary_min ) && empty( $salary_max ) ) {
			return null;
		}
		
		$salary = array(
			'@type' => 'MonetaryAmount',
		);
		
		if ( ! empty( $salary_currency ) ) {
			$salary['currency'] = $salary_currency;
		} else {
			$salary['currency'] = 'USD';
		}
		
		$value = array(
			'@type' => 'QuantitativeValue',
		);
		
		if ( ! empty( $salary_min ) && ! empty( $salary_max ) ) {
			$value['minValue'] = floatval( $salary_min );
			$value['maxValue'] = floatval( $salary_max );
		} elseif ( ! empty( $salary_min ) ) {
			$value['value'] = floatval( $salary_min );
		} elseif ( ! empty( $salary_max ) ) {
			$value['value'] = floatval( $salary_max );
		}
		
		if ( ! empty( $salary_unit ) ) {
			$value['unitText'] = $salary_unit;
		} else {
			$value['unitText'] = 'YEAR';
		}
		
		$salary['value'] = $value;
		
		return $salary;
	}
}