<?php
/**
 * Event Schema Class for BW AI Schema Pro
 *
 * @package BW_AI_Schema_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class BW_Schema_Event extends BW_Schema_Base {
	
	/**
	 * Generate event schema
	 */
	public function generate( $post, $type = 'Event' ) {
		// Base schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
		);
		
		// Add common properties
		$schema = array_merge( $schema, $this->get_common_properties( $post ) );
		
		// Add event status
		$event_status = get_post_meta( $post->ID, '_bw_schema_event_status', true );
		$schema['eventStatus'] = ! empty( $event_status ) ? $event_status : 'https://schema.org/EventScheduled';
		
		// Add event attendance mode
		$attendance_mode = get_post_meta( $post->ID, '_bw_schema_attendance_mode', true );
		$schema['eventAttendanceMode'] = ! empty( $attendance_mode ) ? $attendance_mode : 'https://schema.org/OfflineEventAttendanceMode';
		
		// Add start date
		$start_date = get_post_meta( $post->ID, '_bw_schema_start_date', true );
		if ( ! empty( $start_date ) ) {
			$schema['startDate'] = $start_date;
		}
		
		// Add end date
		$end_date = get_post_meta( $post->ID, '_bw_schema_end_date', true );
		if ( ! empty( $end_date ) ) {
			$schema['endDate'] = $end_date;
		}
		
		// Add location
		$location = $this->get_location_properties( $post );
		if ( ! empty( $location ) ) {
			$schema['location'] = $location;
		}
		
		// Add organizer
		$organizer = $this->get_organizer_properties( $post );
		if ( ! empty( $organizer ) ) {
			$schema['organizer'] = $organizer;
		}
		
		// Add performer
		$performers = get_post_meta( $post->ID, '_bw_schema_performers', true );
		if ( ! empty( $performers ) && is_array( $performers ) ) {
			$performer_list = array();
			foreach ( $performers as $performer ) {
				$performer_list[] = array(
					'@type' => 'Person',
					'name' => $performer,
				);
			}
			$schema['performer'] = $performer_list;
		}
		
		// Add offers
		$offers = $this->get_offer_properties( $post );
		if ( ! empty( $offers ) ) {
			$schema['offers'] = $offers;
		}
		
		// Add audience
		$audience = get_post_meta( $post->ID, '_bw_schema_audience', true );
		if ( ! empty( $audience ) ) {
			$schema['audience'] = array(
				'@type' => 'Audience',
				'audienceType' => $audience,
			);
		}
		
		// Add maximum attendee capacity
		$max_capacity = get_post_meta( $post->ID, '_bw_schema_max_capacity', true );
		if ( ! empty( $max_capacity ) ) {
			$schema['maximumAttendeeCapacity'] = intval( $max_capacity );
		}
		
		// Add duration
		$duration = get_post_meta( $post->ID, '_bw_schema_duration', true );
		if ( ! empty( $duration ) ) {
			$schema['duration'] = $duration;
		}
		
		// Add in language
		$schema['inLanguage'] = get_locale();
		
		// Add AI-optimized properties
		$schema = $this->add_ai_properties( $schema, $post );
		
		// Add specific properties based on event type
		switch ( $type ) {
			case 'BusinessEvent':
				$schema = $this->add_business_event_properties( $schema, $post );
				break;
				
			case 'MusicEvent':
				$schema = $this->add_music_event_properties( $schema, $post );
				break;
				
			case 'SportsEvent':
				$schema = $this->add_sports_event_properties( $schema, $post );
				break;
				
			case 'EducationEvent':
				$schema = $this->add_education_event_properties( $schema, $post );
				break;
		}
		
		return apply_filters( 'bw_schema_event', $schema, $post, $type );
	}
	
	/**
	 * Get location properties
	 */
	private function get_location_properties( $post ) {
		$location_type = get_post_meta( $post->ID, '_bw_schema_location_type', true );
		
		if ( $location_type === 'virtual' ) {
			$virtual_url = get_post_meta( $post->ID, '_bw_schema_virtual_location_url', true );
			return array(
				'@type' => 'VirtualLocation',
				'url' => ! empty( $virtual_url ) ? $virtual_url : get_permalink( $post ),
			);
		}
		
		// Physical location
		$location = array(
			'@type' => 'Place',
		);
		
		$venue_name = get_post_meta( $post->ID, '_bw_schema_venue_name', true );
		if ( ! empty( $venue_name ) ) {
			$location['name'] = $venue_name;
		}
		
		// Add address
		$address = array(
			'@type' => 'PostalAddress',
		);
		
		$street = get_post_meta( $post->ID, '_bw_schema_street_address', true );
		if ( ! empty( $street ) ) {
			$address['streetAddress'] = $street;
		}
		
		$city = get_post_meta( $post->ID, '_bw_schema_address_city', true );
		if ( ! empty( $city ) ) {
			$address['addressLocality'] = $city;
		}
		
		$region = get_post_meta( $post->ID, '_bw_schema_address_region', true );
		if ( ! empty( $region ) ) {
			$address['addressRegion'] = $region;
		}
		
		$postal_code = get_post_meta( $post->ID, '_bw_schema_postal_code', true );
		if ( ! empty( $postal_code ) ) {
			$address['postalCode'] = $postal_code;
		}
		
		$country = get_post_meta( $post->ID, '_bw_schema_address_country', true );
		if ( ! empty( $country ) ) {
			$address['addressCountry'] = $country;
		}
		
		if ( count( $address ) > 1 ) {
			$location['address'] = $address;
		}
		
		return $location;
	}
	
	/**
	 * Get organizer properties
	 */
	private function get_organizer_properties( $post ) {
		$organizer_type = get_post_meta( $post->ID, '_bw_schema_organizer_type', true );
		$organizer_name = get_post_meta( $post->ID, '_bw_schema_organizer_name', true );
		
		if ( empty( $organizer_name ) ) {
			return $this->get_organization_properties();
		}
		
		$organizer = array(
			'@type' => $organizer_type === 'person' ? 'Person' : 'Organization',
			'name' => $organizer_name,
		);
		
		$organizer_url = get_post_meta( $post->ID, '_bw_schema_organizer_url', true );
		if ( ! empty( $organizer_url ) ) {
			$organizer['url'] = $organizer_url;
		}
		
		return $organizer;
	}
	
	/**
	 * Get offer properties
	 */
	private function get_offer_properties( $post ) {
		$offers = array();
		
		// Basic offer
		$offer = array(
			'@type' => 'Offer',
		);
		
		$price = get_post_meta( $post->ID, '_bw_schema_price', true );
		if ( ! empty( $price ) ) {
			$offer['price'] = $price;
		}
		
		$currency = get_post_meta( $post->ID, '_bw_schema_price_currency', true );
		if ( ! empty( $currency ) ) {
			$offer['priceCurrency'] = $currency;
		}
		
		$ticket_url = get_post_meta( $post->ID, '_bw_schema_ticket_url', true );
		if ( ! empty( $ticket_url ) ) {
			$offer['url'] = $ticket_url;
		}
		
		$availability = get_post_meta( $post->ID, '_bw_schema_availability', true );
		if ( ! empty( $availability ) ) {
			$offer['availability'] = $availability;
		}
		
		$valid_from = get_post_meta( $post->ID, '_bw_schema_valid_from', true );
		if ( ! empty( $valid_from ) ) {
			$offer['validFrom'] = $valid_from;
		}
		
		if ( count( $offer ) > 1 ) {
			$offers[] = $offer;
		}
		
		return ! empty( $offers ) ? $offers : null;
	}
	
	/**
	 * Add business event specific properties
	 */
	private function add_business_event_properties( $schema, $post ) {
		// Add industry
		$industry = get_post_meta( $post->ID, '_bw_schema_industry', true );
		if ( ! empty( $industry ) ) {
			$schema['industry'] = $industry;
		}
		
		// Add target audience
		$target_audience = get_post_meta( $post->ID, '_bw_schema_target_audience', true );
		if ( ! empty( $target_audience ) ) {
			$schema['typicalAgeRange'] = $target_audience;
		}
		
		return $schema;
	}
	
	/**
	 * Add music event specific properties
	 */
	private function add_music_event_properties( $schema, $post ) {
		// Add music group
		$music_group = get_post_meta( $post->ID, '_bw_schema_music_group', true );
		if ( ! empty( $music_group ) ) {
			$schema['performer'] = array(
				'@type' => 'MusicGroup',
				'name' => $music_group,
			);
		}
		
		// Add genre
		$genre = get_post_meta( $post->ID, '_bw_schema_music_genre', true );
		if ( ! empty( $genre ) ) {
			$schema['genre'] = $genre;
		}
		
		return $schema;
	}
	
	/**
	 * Add sports event specific properties
	 */
	private function add_sports_event_properties( $schema, $post ) {
		// Add sport
		$sport = get_post_meta( $post->ID, '_bw_schema_sport', true );
		if ( ! empty( $sport ) ) {
			$schema['sport'] = $sport;
		}
		
		// Add home team
		$home_team = get_post_meta( $post->ID, '_bw_schema_home_team', true );
		if ( ! empty( $home_team ) ) {
			$schema['homeTeam'] = array(
				'@type' => 'SportsTeam',
				'name' => $home_team,
			);
		}
		
		// Add away team
		$away_team = get_post_meta( $post->ID, '_bw_schema_away_team', true );
		if ( ! empty( $away_team ) ) {
			$schema['awayTeam'] = array(
				'@type' => 'SportsTeam',
				'name' => $away_team,
			);
		}
		
		return $schema;
	}
	
	/**
	 * Add education event specific properties
	 */
	private function add_education_event_properties( $schema, $post ) {
		// Add educational level
		$edu_level = get_post_meta( $post->ID, '_bw_schema_educational_level', true );
		if ( ! empty( $edu_level ) ) {
			$schema['educationalLevel'] = $edu_level;
		}
		
		// Add instructor
		$instructor = get_post_meta( $post->ID, '_bw_schema_instructor', true );
		if ( ! empty( $instructor ) ) {
			$schema['instructor'] = array(
				'@type' => 'Person',
				'name' => $instructor,
			);
		}
		
		// Add course instance
		$course_code = get_post_meta( $post->ID, '_bw_schema_course_code', true );
		if ( ! empty( $course_code ) ) {
			$schema['courseCode'] = $course_code;
		}
		
		return $schema;
	}
}