<?php
/**
 * Plugin Name: BW Blocks
 * Plugin URI: https://bowdenworks.com/plugins/bw-blocks
 * Description: Custom Gutenberg blocks including Badge, Hero H1, and Buttons by Bowden Works
 * Version: 1.0.0
 * Author: Bowden Works
 * Author URI: https://bowdenworks.com
 * License: GPL v2 or later
 * Text Domain: bw-blocks
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BW_BLOCKS_VERSION', '1.0.0');
define('BW_BLOCKS_PATH', plugin_dir_path(__FILE__));
define('BW_BLOCKS_URL', plugin_dir_url(__FILE__));

class BW_Blocks {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'bw_register_blocks'));
        add_action('enqueue_block_editor_assets', array($this, 'bw_enqueue_block_editor_assets'));
        add_action('enqueue_block_assets', array($this, 'bw_enqueue_block_assets'));
    }
    
    public function bw_register_blocks() {
        // Register BW Badge Block
        register_block_type('bw-blocks/badge', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_badge_block'),
            'attributes' => array(
                'icon' => array(
                    'type' => 'string',
                    'default' => '✓'
                ),
                'text' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet'
                ),
                'animation' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Hero H1 Block
        register_block_type('bw-blocks/hero-h1', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_hero_h1_block'),
            'attributes' => array(
                'text' => array(
                    'type' => 'string',
                    'default' => 'Lorem Ipsum<br>Dolor Sit Amet'
                ),
                'gradient' => array(
                    'type' => 'boolean',
                    'default' => true
                ),
                'glitchHover' => array(
                    'type' => 'boolean',
                    'default' => true
                ),
                'alignment' => array(
                    'type' => 'string',
                    'default' => 'center'
                )
            )
        ));
        
        // Register BW Program Block
        register_block_type('bw-blocks/program', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_program_block'),
            'attributes' => array(
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'flagship'
                ),
                'badge' => array(
                    'type' => 'string',
                    'default' => 'FLAGSHIP PROGRAM'
                ),
                'title' => array(
                    'type' => 'string',
                    'default' => 'Program Title'
                ),
                'description' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.'
                ),
                'features' => array(
                    'type' => 'array',
                    'default' => array(
                        'Feature one goes here',
                        'Feature two goes here',
                        'Feature three goes here',
                        'Feature four goes here'
                    )
                ),
                'price' => array(
                    'type' => 'string',
                    'default' => '$999'
                ),
                'priceNote' => array(
                    'type' => 'string',
                    'default' => 'Early bird available'
                ),
                'buttonText' => array(
                    'type' => 'string',
                    'default' => 'Enroll Now →'
                ),
                'buttonUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                )
            )
        ));
        
        // Register BW Buttons Block
        register_block_type('bw-blocks/buttons', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_buttons_block'),
            'attributes' => array(
                'primaryText' => array(
                    'type' => 'string',
                    'default' => 'Button Text →'
                ),
                'primaryUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                ),
                'secondaryText' => array(
                    'type' => 'string',
                    'default' => 'Button Text'
                ),
                'secondaryUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                ),
                'showSecondary' => array(
                    'type' => 'boolean',
                    'default' => true
                ),
                'alignment' => array(
                    'type' => 'string',
                    'default' => 'center'
                )
            )
        ));
        
        // Register BW Value Card Block
        register_block_type('bw-blocks/value-card', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_value_card_block'),
            'attributes' => array(
                'icon' => array(
                    'type' => 'string',
                    'default' => '🎯'
                ),
                'title' => array(
                    'type' => 'string',
                    'default' => 'Lorem Ipsum Dolor'
                ),
                'description' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
                ),
                'iconColor' => array(
                    'type' => 'string',
                    'default' => 'purple'
                )
            )
        ));
        
        // Register BW Course Header Block
        register_block_type('bw-blocks/course-header', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor', 
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_course_header_block'),
            'attributes' => array(
                'title' => array(
                    'type' => 'string',
                    'default' => 'AI Fundamentals Masterclass'
                ),
                'subtitle' => array(
                    'type' => 'string',
                    'default' => 'Master the foundations of artificial intelligence'
                ),
                'duration' => array(
                    'type' => 'string',
                    'default' => '12 Weeks'
                ),
                'format' => array(
                    'type' => 'string',
                    'default' => 'Online + Live Sessions'
                ),
                'level' => array(
                    'type' => 'string',
                    'default' => 'Beginner to Intermediate'
                ),
                'certification' => array(
                    'type' => 'string',
                    'default' => 'Professional Certificate'
                ),
                'price' => array(
                    'type' => 'string',
                    'default' => '$2,499'
                ),
                'enrollmentText' => array(
                    'type' => 'string',
                    'default' => 'Enroll Now'
                ),
                'enrollmentUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                ),
                'showEnrollmentCard' => array(
                    'type' => 'boolean',
                    'default' => true
                ),
                'enrollmentFeatures' => array(
                    'type' => 'array',
                    'default' => array(
                        'Lorem ipsum dolor sit amet',
                        'Consectetur adipiscing elit sed',
                        'Do eiusmod tempor incididunt',
                        'Ut labore et dolore magna',
                        'Aliqua ut enim ad minim',
                        'Veniam quis nostrud exercitation',
                        'Ullamco laboris nisi ut aliquip'
                    )
                ),
                'priceSubtext' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet'
                ),
                'secondaryButtonText' => array(
                    'type' => 'string',
                    'default' => 'Button Text'
                ),
                'secondaryButtonUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                ),
                'guarantee' => array(
                    'type' => 'string',
                    'default' => '30-day money-back guarantee'
                ),
                'nextCohortDate' => array(
                    'type' => 'string',
                    'default' => 'Tuesday, September 16th, 2025'
                ),
                'sessionSchedule' => array(
                    'type' => 'string',
                    'default' => 'Weekly sessions on Tuesdays, 2:00-3:00 PM PST'
                )
            )
        ));
        
        // Register BW Course Curriculum Block
        register_block_type('bw-blocks/course-curriculum', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_course_curriculum_block'),
            'attributes' => array(
                'title' => array(
                    'type' => 'string',
                    'default' => 'Course Curriculum'
                ),
                'modules' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'title' => 'Module 1: Introduction to AI',
                            'duration' => '2 weeks',
                            'lessons' => array(
                                'What is Artificial Intelligence?',
                                'History and Evolution of AI',
                                'Types of AI Systems',
                                'Real-world Applications'
                            )
                        ),
                        array(
                            'title' => 'Module 2: Machine Learning Basics',
                            'duration' => '3 weeks',
                            'lessons' => array(
                                'Introduction to Machine Learning',
                                'Supervised vs Unsupervised Learning',
                                'Common ML Algorithms',
                                'Hands-on Projects'
                            )
                        )
                    )
                ),
                'showDuration' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Course Instructor Block
        register_block_type('bw-blocks/course-instructor', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_course_instructor_block'),
            'attributes' => array(
                'name' => array(
                    'type' => 'string',
                    'default' => 'Dr. Sarah Johnson'
                ),
                'title' => array(
                    'type' => 'string',
                    'default' => 'AI Research Lead'
                ),
                'bio' => array(
                    'type' => 'string',
                    'default' => 'With over 15 years of experience in artificial intelligence and machine learning, Dr. Johnson has led groundbreaking research at top tech companies and universities.'
                ),
                'imageUrl' => array(
                    'type' => 'string',
                    'default' => 'https://via.placeholder.com/300x300/667eea/ffffff?text=Instructor'
                ),
                'credentials' => array(
                    'type' => 'array',
                    'default' => array(
                        'PhD in Computer Science, MIT',
                        'Former Google AI Research Lead',
                        'Published 50+ AI Papers',
                        '10+ Years Teaching Experience'
                    )
                )
            )
        ));
        
        // Register BW Course FAQ Block
        register_block_type('bw-blocks/course-faq', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_course_faq_block'),
            'attributes' => array(
                'title' => array(
                    'type' => 'string',
                    'default' => 'Frequently Asked Questions'
                ),
                'faqs' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'question' => 'Who is this course for?',
                            'answer' => 'This course is designed for beginners and intermediate learners who want to understand AI fundamentals and apply them in real-world scenarios.'
                        ),
                        array(
                            'question' => 'What are the prerequisites?',
                            'answer' => 'Basic programming knowledge is helpful but not required. We cover all necessary foundations during the course.'
                        ),
                        array(
                            'question' => 'How long do I have access to the course?',
                            'answer' => 'You get lifetime access to all course materials, including future updates and additions.'
                        )
                    )
                )
            )
        ));
        
        // Register BW Icon Card Block (with SVG icon support)
        register_block_type('bw-blocks/icon-card', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_icon_card_block'),
            'attributes' => array(
                'iconType' => array(
                    'type' => 'string',
                    'default' => 'target' // target, lightning, shield, rocket, chart, users
                ),
                'title' => array(
                    'type' => 'string',
                    'default' => 'Lorem Ipsum Dolor'
                ),
                'description' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
                ),
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'primary' // primary, pink, teal
                )
            )
        ));
        
        // Register BW Value Grid Item Block
        register_block_type('bw-blocks/value-grid-item', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_value_grid_item_block'),
            'attributes' => array(
                'number' => array(
                    'type' => 'string',
                    'default' => '100+'
                ),
                'label' => array(
                    'type' => 'string',
                    'default' => 'Projects Completed'
                ),
                'icon' => array(
                    'type' => 'string',
                    'default' => '🚀'
                ),
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'purple'
                ),
                'animateNumber' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Service Block
        register_block_type('bw-blocks/service', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_service_block'),
            'attributes' => array(
                'icon' => array(
                    'type' => 'string',
                    'default' => '🚀'
                ),
                'title' => array(
                    'type' => 'string',
                    'default' => 'Service Title'
                ),
                'description' => array(
                    'type' => 'string',
                    'default' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
                ),
                'metadata' => array(
                    'type' => 'string',
                    'default' => 'Professional Service'
                ),
                'features' => array(
                    'type' => 'array',
                    'default' => array(
                        'Feature one goes here',
                        'Feature two goes here',
                        'Feature three goes here'
                    )
                ),
                'buttonText' => array(
                    'type' => 'string',
                    'default' => 'Learn More →'
                ),
                'buttonUrl' => array(
                    'type' => 'string',
                    'default' => '#'
                ),
                'images' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'url' => 'https://via.placeholder.com/400x300/667eea/ffffff?text=Service+1',
                            'alt' => 'Service Image 1'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/400x300/764ba2/ffffff?text=Service+2',
                            'alt' => 'Service Image 2'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/400x300/00d4aa/ffffff?text=Service+3',
                            'alt' => 'Service Image 3'
                        )
                    )
                ),
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'primary'
                ),
                'showImages' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Horizontal Timeline Block
        register_block_type('bw-blocks/horizontal-timeline', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_horizontal_timeline_block'),
            'attributes' => array(
                'steps' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'number' => '1',
                            'title' => 'Real Experience',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing'
                        ),
                        array(
                            'number' => '2',
                            'title' => 'Proven Results',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing'
                        ),
                        array(
                            'number' => '3',
                            'title' => 'Comprehensive Approach',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing'
                        ),
                        array(
                            'number' => '4',
                            'title' => 'Practical Focus',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing'
                        )
                    )
                ),
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'primary'
                ),
                'showConnector' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Comparison Table Block
        register_block_type('bw-blocks/comparison-table', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_comparison_table_block'),
            'attributes' => array(
                'headers' => array(
                    'type' => 'array',
                    'default' => array('Path', 'Description', 'Investment', 'Next Step')
                ),
                'rows' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'path' => 'Individual Training',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing',
                            'investment' => '$197 - $4,999',
                            'buttonText' => 'Learn More',
                            'buttonUrl' => '#'
                        ),
                        array(
                            'path' => 'Corporate Team Training',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing',
                            'investment' => 'Custom pricing',
                            'buttonText' => 'Get Quote',
                            'buttonUrl' => '#'
                        ),
                        array(
                            'path' => 'Consulting Engagement',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing',
                            'investment' => 'Project-based',
                            'buttonText' => 'Contact Us',
                            'buttonUrl' => '#'
                        ),
                        array(
                            'path' => 'Legal Compliance Workshop',
                            'description' => 'Lorem ipsum dolor sit amet consectetur adipiscing',
                            'investment' => 'Contact for pricing',
                            'buttonText' => 'Inquire Now',
                            'buttonUrl' => '#'
                        )
                    )
                ),
                'colorScheme' => array(
                    'type' => 'string',
                    'default' => 'primary'
                ),
                'mobileCards' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
        
        // Register BW Gallery Grid Block
        register_block_type('bw-blocks/gallery-grid', array(
            'editor_script' => 'bw-blocks-editor',
            'editor_style' => 'bw-blocks-editor',
            'style' => 'bw-blocks-style',
            'render_callback' => array($this, 'bw_render_gallery_grid_block'),
            'attributes' => array(
                'images' => array(
                    'type' => 'array',
                    'default' => array(
                        array(
                            'url' => 'https://via.placeholder.com/600x520/667eea/ffffff?text=Featured',
                            'title' => 'Featured Project',
                            'description' => 'Our flagship design showcasing innovation'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/300x250/764ba2/ffffff?text=Project+2',
                            'title' => 'Web Development',
                            'description' => 'Modern responsive web applications'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/300x250/00d4aa/ffffff?text=Project+3',
                            'title' => 'Mobile Design',
                            'description' => 'User-centric mobile experiences'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/300x250/f093fb/ffffff?text=Project+4',
                            'title' => 'Brand Identity',
                            'description' => 'Complete brand transformation'
                        ),
                        array(
                            'url' => 'https://via.placeholder.com/300x250/f5576c/ffffff?text=Project+5',
                            'title' => 'UI/UX Design',
                            'description' => 'Intuitive interface design'
                        )
                    )
                ),
                'columns' => array(
                    'type' => 'number',
                    'default' => 3
                ),
                'gap' => array(
                    'type' => 'number',
                    'default' => 20
                ),
                'borderRadius' => array(
                    'type' => 'number',
                    'default' => 16
                ),
                'enableLightbox' => array(
                    'type' => 'boolean',
                    'default' => true
                )
            )
        ));
    }
    
    public function bw_enqueue_block_editor_assets() {
        wp_enqueue_script(
            'bw-blocks-editor',
            BW_BLOCKS_URL . 'assets/bw-blocks-editor.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
            BW_BLOCKS_VERSION,
            true
        );
        
        wp_enqueue_style(
            'bw-blocks-editor',
            BW_BLOCKS_URL . 'assets/bw-blocks-editor.css',
            array('wp-edit-blocks'),
            BW_BLOCKS_VERSION
        );
    }
    
    public function bw_enqueue_block_assets() {
        wp_enqueue_style(
            'bw-blocks-style',
            BW_BLOCKS_URL . 'assets/bw-blocks-style.css',
            array(),
            BW_BLOCKS_VERSION
        );
        
        // Add gallery initialization script
        wp_add_inline_script('wp-blocks', '
            function bwInitGallery(galleryId, enableLightbox) {
                const gallery = document.getElementById(galleryId);
                if (!gallery) return;
                
                const controls = document.querySelector(\'.bw-gallery-controls[data-gallery="\' + galleryId + \'"]\');
                const prevBtn = controls ? controls.querySelector(\'.bw-gallery-prev\') : null;
                const nextBtn = controls ? controls.querySelector(\'.bw-gallery-next\') : null;
                const dotsContainer = controls ? controls.querySelector(\'.bw-gallery-dots\') : null;
                
                // Mobile carousel functionality
                if (window.innerWidth <= 768) {
                    const items = gallery.querySelectorAll(\'.bw-gallery-item\');
                    let currentIndex = 0;
                    
                    // Create dots
                    if (dotsContainer) {
                        items.forEach((item, index) => {
                            const dot = document.createElement(\'span\');
                            dot.className = \'bw-gallery-dot\';
                            if (index === 0) dot.classList.add(\'active\');
                            dot.addEventListener(\'click\', () => scrollToItem(index));
                            dotsContainer.appendChild(dot);
                        });
                    }
                    
                    function scrollToItem(index) {
                        if (index < 0 || index >= items.length) return;
                        currentIndex = index;
                        const item = items[index];
                        const scrollLeft = item.offsetLeft - (gallery.offsetWidth - item.offsetWidth) / 2;
                        gallery.scrollTo({ left: scrollLeft, behavior: \'smooth\' });
                        updateDots();
                    }
                    
                    function updateDots() {
                        const dots = dotsContainer ? dotsContainer.querySelectorAll(\'.bw-gallery-dot\') : [];
                        dots.forEach((dot, index) => {
                            dot.classList.toggle(\'active\', index === currentIndex);
                        });
                    }
                    
                    // Navigation buttons
                    if (prevBtn) {
                        prevBtn.addEventListener(\'click\', () => {
                            scrollToItem(currentIndex - 1);
                        });
                    }
                    
                    if (nextBtn) {
                        nextBtn.addEventListener(\'click\', () => {
                            scrollToItem(currentIndex + 1);
                        });
                    }
                    
                    // Update on scroll
                    gallery.addEventListener(\'scroll\', () => {
                        const scrollLeft = gallery.scrollLeft;
                        const itemWidth = items[0].offsetWidth;
                        const newIndex = Math.round(scrollLeft / itemWidth);
                        if (newIndex !== currentIndex) {
                            currentIndex = newIndex;
                            updateDots();
                        }
                    });
                }
                
                // Lightbox functionality
                if (enableLightbox) {
                    const items = gallery.querySelectorAll(\'.bw-gallery-item img\');
                    items.forEach(img => {
                        img.style.cursor = \'pointer\';
                        img.addEventListener(\'click\', function() {
                            // Simple lightbox - you can enhance this
                            const overlay = document.createElement(\'div\');
                            overlay.style.cssText = \'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.9);z-index:99999;display:flex;align-items:center;justify-content:center;cursor:pointer;\';
                            const largeImg = document.createElement(\'img\');
                            largeImg.src = this.src;
                            largeImg.style.cssText = \'max-width:90%;max-height:90%;object-fit:contain;\';
                            overlay.appendChild(largeImg);
                            overlay.addEventListener(\'click\', () => overlay.remove());
                            document.body.appendChild(overlay);
                        });
                    });
                }
            }
        ');
    }
    
    public function bw_render_badge_block($attributes) {
        $icon = esc_html($attributes['icon']);
        $text = esc_html($attributes['text']);
        $animation_class = $attributes['animation'] ? 'bw-animated' : '';
        
        return sprintf(
            '<div class="bw-badge-wrapper">
                <div class="bw-badge %s">
                    <span class="bw-badge-icon">%s</span>
                    <span>%s</span>
                </div>
            </div>',
            esc_attr($animation_class),
            $icon,
            $text
        );
    }
    
    public function bw_render_hero_h1_block($attributes) {
        $text = wp_kses_post($attributes['text']);
        $gradient_class = $attributes['gradient'] ? 'bw-gradient-text' : '';
        $glitch_class = $attributes['glitchHover'] ? 'bw-glitch-hover' : '';
        $alignment = esc_attr($attributes['alignment']);
        
        // Always add white color style for non-gradient text
        $color_style = !$attributes['gradient'] ? 'color: white;' : '';
        
        return sprintf(
            '<h1 class="bw-hero-h1 %s %s" style="text-align: %s; %s">%s</h1>',
            esc_attr($gradient_class),
            esc_attr($glitch_class),
            $alignment,
            $color_style,
            $text
        );
    }
    
    public function bw_render_program_block($attributes) {
        $color_scheme = esc_attr($attributes['colorScheme']);
        $badge = esc_html($attributes['badge']);
        $title = esc_html($attributes['title']);
        $description = esc_html($attributes['description']);
        $features = $attributes['features'];
        $price = esc_html($attributes['price']);
        $price_note = esc_html($attributes['priceNote']);
        $button_text = esc_html($attributes['buttonText']);
        $button_url = esc_url($attributes['buttonUrl']);
        
        $output = sprintf('<div class="bw-program-card bw-program-%s">', $color_scheme);
        
        // Header
        $output .= '<div class="bw-program-header">';
        $output .= sprintf('<span class="bw-program-badge">%s</span>', $badge);
        $output .= sprintf('<h3>%s</h3>', $title);
        $output .= '</div>';
        
        // Body
        $output .= '<div class="bw-program-body">';
        $output .= sprintf('<p class="bw-program-description">%s</p>', $description);
        
        // Features
        if (!empty($features)) {
            $output .= '<div class="bw-program-features">';
            foreach ($features as $feature) {
                $output .= '<div class="bw-feature-item">';
                $output .= '<span class="bw-feature-icon">✓</span>';
                $output .= sprintf('<span class="bw-feature-text">%s</span>', esc_html($feature));
                $output .= '</div>';
            }
            $output .= '</div>';
        }
        
        // Footer
        $output .= '<div class="bw-program-footer">';
        $output .= '<div class="bw-price-section">';
        $output .= sprintf('<span class="bw-price">%s</span>', $price);
        if (!empty($price_note)) {
            $output .= sprintf('<span class="bw-price-note">%s</span>', $price_note);
        }
        $output .= '</div>';
        $output .= sprintf('<a href="%s" class="bw-btn bw-btn-primary">%s</a>', $button_url, $button_text);
        $output .= '</div>';
        
        $output .= '</div>'; // program-body
        $output .= '</div>'; // program-card
        
        return $output;
    }
    
    public function bw_render_buttons_block($attributes) {
        $primary_text = esc_html($attributes['primaryText']);
        $primary_url = esc_url($attributes['primaryUrl']);
        $secondary_text = esc_html($attributes['secondaryText']);
        $secondary_url = esc_url($attributes['secondaryUrl']);
        $show_secondary = $attributes['showSecondary'];
        $alignment = esc_attr($attributes['alignment']);
        
        $output = sprintf('<div class="bw-buttons" style="text-align: %s;">', $alignment);
        $output .= sprintf('<a href="%s" class="bw-btn bw-btn-primary">%s</a>', $primary_url, $primary_text);
        
        if ($show_secondary) {
            $output .= sprintf('<a href="%s" class="bw-btn bw-btn-secondary">%s</a>', $secondary_url, $secondary_text);
        }
        
        $output .= '</div>';
        
        return $output;
    }
    
    public function bw_render_gallery_grid_block($attributes) {
        $images = $attributes['images'];
        $columns = intval($attributes['columns']);
        $gap = intval($attributes['gap']);
        $border_radius = intval($attributes['borderRadius']);
        $enable_lightbox = $attributes['enableLightbox'];
        
        if (empty($images)) {
            return '<div class="bw-gallery-grid-placeholder">Please add images to the gallery</div>';
        }
        
        $unique_id = 'bw-gallery-' . uniqid();
        
        $output = sprintf('<div id="%s" class="bw-gallery-grid" data-columns="%d">', esc_attr($unique_id), $columns);
        
        foreach ($images as $index => $image) {
            $image_url = isset($image['url']) ? esc_url($image['url']) : '';
            $image_alt = isset($image['alt']) ? esc_attr($image['alt']) : '';
            $image_title = isset($image['title']) ? esc_html($image['title']) : '';
            $image_description = isset($image['description']) ? esc_html($image['description']) : '';
            
            $output .= sprintf(
                '<div class="bw-gallery-item">
                    <img src="%s" alt="%s" loading="lazy" />
                    <div class="bw-gallery-overlay">
                        <h3 class="bw-gallery-title">%s</h3>
                        <p class="bw-gallery-description">%s</p>
                    </div>
                </div>',
                $image_url,
                $image_alt,
                $image_title,
                $image_description
            );
        }
        
        $output .= '</div>';
        
        // Add mobile carousel controls
        $output .= sprintf(
            '<div class="bw-gallery-controls" data-gallery="%s">
                <button class="bw-gallery-prev" aria-label="Previous">‹</button>
                <button class="bw-gallery-next" aria-label="Next">›</button>
                <div class="bw-gallery-dots"></div>
            </div>',
            esc_attr($unique_id)
        );
        
        // Add inline styles for this specific gallery
        $output .= sprintf(
            '<style>
                #%s { gap: %dpx; }
                #%s .bw-gallery-item img { border-radius: %dpx; }
            </style>',
            esc_attr($unique_id),
            $gap,
            esc_attr($unique_id),
            $border_radius
        );
        
        // Add initialization script
        $output .= sprintf(
            '<script>
            document.addEventListener("DOMContentLoaded", function() {
                bwInitGallery("%s", %s);
            });
            </script>',
            esc_attr($unique_id),
            $enable_lightbox ? 'true' : 'false'
        );
        
        return $output;
    }
    
    public function bw_render_value_card_block($attributes) {
        $icon = esc_html($attributes['icon']);
        $title = esc_html($attributes['title']);
        $description = esc_html($attributes['description']);
        $icon_color = esc_attr($attributes['iconColor']);
        
        $output = sprintf(
            '<div class="bw-value-card scroll-scale">
                <div class="bw-value-card-icon bw-icon-%s">
                    <span>%s</span>
                </div>
                <h3 class="bw-value-card-title">%s</h3>
                <p class="bw-value-card-description">%s</p>
            </div>',
            $icon_color,
            $icon,
            $title,
            $description
        );
        
        // Add scroll animation script
        static $script_added = false;
        if (!$script_added) {
            $output .= '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const observerOptions = {
                    threshold: 0.1,
                    rootMargin: "0px 0px -50px 0px"
                };
                
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach((entry, index) => {
                        if (entry.isIntersecting) {
                            setTimeout(() => {
                                entry.target.classList.add("visible");
                            }, index * 200);
                            observer.unobserve(entry.target);
                        }
                    });
                }, observerOptions);
                
                document.querySelectorAll(".bw-value-card.scroll-scale").forEach(card => {
                    observer.observe(card);
                });
            });
            </script>';
            $script_added = true;
        }
        
        return $output;
    }
    
    public function bw_render_service_block($attributes) {
        $icon = esc_html($attributes['icon']);
        $title = esc_html($attributes['title']);
        $description = esc_html($attributes['description']);
        $metadata = esc_html($attributes['metadata']);
        $features = $attributes['features'];
        $button_text = esc_html($attributes['buttonText']);
        $button_url = esc_url($attributes['buttonUrl']);
        $images = $attributes['images'];
        $color_scheme = esc_attr($attributes['colorScheme']);
        $show_images = $attributes['showImages'];
        
        $output = sprintf('<div class="bw-service-detail bw-service-%s">', $color_scheme);
        
        // Service header with icon and content
        $output .= '<div class="bw-service-header">';
        
        // Icon area
        $output .= '<div class="bw-service-icon-area">';
        $output .= sprintf('<div class="bw-service-icon">%s</div>', $icon);
        $output .= '</div>';
        
        // Content area
        $output .= '<div class="bw-service-content">';
        $output .= sprintf('<h3 class="bw-service-title">%s</h3>', $title);
        $output .= sprintf('<p class="bw-service-description">%s</p>', $description);
        
        // Service meta with styled accent
        if (!empty($metadata)) {
            $output .= '<div class="bw-service-meta">';
            $output .= sprintf('<span class="bw-service-meta-label">Service Type:</span> %s', $metadata);
            $output .= '</div>';
        }
        
        // Features list
        if (!empty($features)) {
            $output .= '<ul class="bw-service-features">';
            foreach ($features as $feature) {
                $output .= sprintf('<li>%s</li>', esc_html($feature));
            }
            $output .= '</ul>';
        }
        
        // Button
        if (!empty($button_text)) {
            $output .= sprintf('<a href="%s" class="bw-btn bw-btn-primary">%s</a>', $button_url, $button_text);
        }
        
        $output .= '</div>'; // service-content
        $output .= '</div>'; // service-header
        
        // Service images section with separator
        if ($show_images && !empty($images)) {
            $output .= '<div class="bw-service-separator"></div>';
            $output .= '<div class="bw-service-images">';
            $output .= '<div class="bw-service-images-grid">';
            
            foreach ($images as $index => $image) {
                if ($index < 3) { // Limit to 3 images
                    $image_url = isset($image['url']) ? esc_url($image['url']) : '';
                    $image_alt = isset($image['alt']) ? esc_attr($image['alt']) : '';
                    
                    $output .= '<div class="bw-service-image-item">';
                    if (!empty($image_url)) {
                        $output .= sprintf('<img src="%s" alt="%s" loading="lazy" />', $image_url, $image_alt);
                    }
                    $output .= '<div class="bw-service-image-overlay"></div>';
                    $output .= '</div>';
                }
            }
            
            $output .= '</div>'; // service-images-grid
            $output .= '</div>'; // service-images
        }
        
        $output .= '</div>'; // service-detail
        
        return $output;
    }
    
    public function bw_render_horizontal_timeline_block($attributes) {
        $steps = $attributes['steps'];
        $color_scheme = esc_attr($attributes['colorScheme']);
        $show_connector = $attributes['showConnector'];
        
        $output = sprintf('<div class="bw-process-timeline bw-timeline-%s%s">', 
            $color_scheme,
            $show_connector ? '' : ' no-connector'
        );
        
        foreach ($steps as $index => $step) {
            $number = esc_html($step['number']);
            $title = esc_html($step['title']);
            $description = esc_html($step['description']);
            
            $output .= '<div class="bw-process-step">';
            $output .= sprintf('<div class="bw-step-number">%s</div>', $number);
            $output .= sprintf('<h3 class="bw-step-title">%s</h3>', $title);
            $output .= sprintf('<p class="bw-step-description">%s</p>', $description);
            $output .= '</div>';
        }
        
        $output .= '</div>'; // process-timeline
        
        // Add scroll animation script
        static $script_added = false;
        if (!$script_added) {
            $output .= '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const observerOptions = {
                    threshold: 0.3,
                    rootMargin: "0px 0px -50px 0px"
                };
                
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            entry.target.classList.add("visible");
                            observer.unobserve(entry.target);
                        }
                    });
                }, observerOptions);
                
                document.querySelectorAll(".bw-process-timeline").forEach(timeline => {
                    observer.observe(timeline);
                });
            });
            </script>';
            $script_added = true;
        }
        
        return $output;
    }
    
    public function bw_render_comparison_table_block($attributes) {
        $headers = $attributes['headers'];
        $rows = $attributes['rows'];
        $color_scheme = esc_attr($attributes['colorScheme']);
        $mobile_cards = $attributes['mobileCards'];
        
        $output = sprintf('<div class="bw-comparison-table-wrapper bw-table-%s%s">', 
            $color_scheme,
            $mobile_cards ? ' mobile-cards' : ''
        );
        
        // Desktop table
        $output .= '<table class="bw-comparison-table">';
        
        // Table header
        $output .= '<thead><tr>';
        foreach ($headers as $header) {
            $output .= sprintf('<th>%s</th>', esc_html($header));
        }
        $output .= '</tr></thead>';
        
        // Table body
        $output .= '<tbody>';
        foreach ($rows as $row) {
            $output .= '<tr>';
            $output .= sprintf('<td><strong>%s</strong></td>', esc_html($row['path']));
            $output .= sprintf('<td>%s</td>', esc_html($row['description']));
            $output .= sprintf('<td>%s</td>', esc_html($row['investment']));
            $output .= sprintf('<td><a href="%s" class="bw-table-link">%s</a></td>', 
                esc_url($row['buttonUrl']), 
                esc_html($row['buttonText'])
            );
            $output .= '</tr>';
        }
        $output .= '</tbody>';
        $output .= '</table>';
        
        // Mobile cards view
        if ($mobile_cards) {
            $output .= '<div class="bw-comparison-cards">';
            foreach ($rows as $row) {
                $output .= sprintf('<div class="bw-comparison-card bw-card-%s">', $color_scheme);
                $output .= sprintf('<h3 class="bw-card-title">%s</h3>', esc_html($row['path']));
                $output .= '<div class="bw-card-content">';
                $output .= sprintf('<div class="bw-card-row"><span class="bw-card-label">%s:</span> <span>%s</span></div>', 
                    esc_html($headers[1]), esc_html($row['description']));
                $output .= sprintf('<div class="bw-card-row"><span class="bw-card-label">%s:</span> <span>%s</span></div>', 
                    esc_html($headers[2]), esc_html($row['investment']));
                $output .= '</div>';
                $output .= sprintf('<a href="%s" class="bw-card-button">%s</a>', 
                    esc_url($row['buttonUrl']), 
                    esc_html($row['buttonText'])
                );
                $output .= '</div>';
            }
            $output .= '</div>';
        }
        
        $output .= '</div>'; // wrapper
        
        return $output;
    }
    
    public function bw_render_icon_card_block($attributes) {
        $icon_type = esc_attr($attributes['iconType']);
        $title = esc_html($attributes['title']);
        $description = esc_html($attributes['description']);
        $color_scheme = esc_attr($attributes['colorScheme']);
        
        // Define SVG icons for each type
        $icons = array(
            'target' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="6"/><circle cx="12" cy="12" r="2"/></svg>',
            'lightning' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg>',
            'shield' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>',
            'rocket' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M4.5 16.5c-1.5 1.26-2 5-2 5s3.74-.5 5-2c.71-.84.7-2.13-.09-2.91a2.18 2.18 0 0 0-2.91-.09z"/><path d="m12 15-3-3a22 22 0 0 1 2-3.95A12.88 12.88 0 0 1 22 2c0 2.72-.78 7.5-6 11a22.35 22.35 0 0 1-4 2z"/><path d="M9 12H4s.55-3.03 2-4c1.62-1.08 5 0 5 0"/><path d="M12 15v5s3.03-.55 4-2c1.08-1.62 0-5 0-5"/></svg>',
            'chart' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="20" x2="12" y2="10"/><line x1="18" y1="20" x2="18" y2="4"/><line x1="6" y1="20" x2="6" y2="16"/></svg>',
            'users' => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>'
        );
        
        $icon_svg = isset($icons[$icon_type]) ? $icons[$icon_type] : $icons['target'];
        
        $output = sprintf(
            '<div class="bw-icon-card bw-icon-card-%s scroll-scale">
                <div class="bw-icon-card-icon">
                    %s
                </div>
                <h3 class="bw-icon-card-title">%s</h3>
                <p class="bw-icon-card-description">%s</p>
            </div>',
            $color_scheme,
            $icon_svg,
            $title,
            $description
        );
        
        // Add scroll animation script (only once)
        static $script_added = false;
        if (!$script_added) {
            $output .= '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const observerOptions = {
                    threshold: 0.1,
                    rootMargin: "0px 0px -50px 0px"
                };
                
                const observer = new IntersectionObserver(function(entries) {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            entry.target.classList.add("visible");
                            observer.unobserve(entry.target);
                        }
                    });
                }, observerOptions);
                
                document.querySelectorAll(".bw-icon-card.scroll-scale").forEach(card => {
                    observer.observe(card);
                });
            });
            </script>';
            $script_added = true;
        }
        
        return $output;
    }
    
    public function bw_render_course_header_block($attributes) {
        $title = esc_html($attributes['title']);
        $subtitle = esc_html($attributes['subtitle']);
        $duration = esc_html($attributes['duration']);
        $format = esc_html($attributes['format']);
        $level = esc_html($attributes['level']);
        $certification = esc_html($attributes['certification']);
        $price = esc_html($attributes['price']);
        $enrollment_text = esc_html($attributes['enrollmentText']);
        $enrollment_url = esc_url($attributes['enrollmentUrl']);
        $show_enrollment_card = isset($attributes['showEnrollmentCard']) ? $attributes['showEnrollmentCard'] : true;
        $enrollment_features = isset($attributes['enrollmentFeatures']) ? $attributes['enrollmentFeatures'] : array();
        $price_subtext = isset($attributes['priceSubtext']) ? esc_html($attributes['priceSubtext']) : '';
        $secondary_button_text = isset($attributes['secondaryButtonText']) ? esc_html($attributes['secondaryButtonText']) : '';
        $secondary_button_url = isset($attributes['secondaryButtonUrl']) ? esc_url($attributes['secondaryButtonUrl']) : '#';
        $guarantee = isset($attributes['guarantee']) ? esc_html($attributes['guarantee']) : '';
        $next_cohort_date = isset($attributes['nextCohortDate']) ? esc_html($attributes['nextCohortDate']) : '';
        $session_schedule = isset($attributes['sessionSchedule']) ? esc_html($attributes['sessionSchedule']) : '';
        
        $output = '<div class="bw-course-header' . ($show_enrollment_card ? ' has-enrollment-card' : '') . '">';
        $output .= '<div class="bw-course-header-wrapper">';
        
        // Main content section
        $output .= '<div class="bw-course-header-content">';
        $output .= sprintf('<h1 class="bw-course-title">%s</h1>', $title);
        $output .= sprintf('<p class="bw-course-subtitle">%s</p>', $subtitle);
        
        $output .= '<div class="bw-course-meta">';
        $output .= sprintf('<div class="bw-meta-item"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/></svg><span>%s</span></div>', $duration);
        $output .= sprintf('<div class="bw-meta-item"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 10v6M2 10l10-5 10 5-10 5z"/><path d="M6 12v5c3 3 9 3 12 0v-5"/></svg><span>%s</span></div>', $certification);
        $output .= sprintf('<div class="bw-meta-item"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="2" y="3" width="20" height="14" rx="2" ry="2"/><line x1="8" y1="21" x2="16" y2="21"/><line x1="12" y1="17" x2="12" y2="21"/></svg><span>%s</span></div>', $format);
        $output .= sprintf('<div class="bw-meta-item"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="4" width="18" height="18" rx="2" ry="2"/><line x1="16" y1="2" x2="16" y2="6"/><line x1="8" y1="2" x2="8" y2="6"/><line x1="3" y1="10" x2="21" y2="10"/></svg><span>Tuesdays 2-3PM PST</span></div>');
        $output .= '</div>';
        
        // Since 2015 badge
        $output .= '<div class="bw-course-badge"><span class="bw-badge-star">⭐</span> Since 2015</div>';
        
        // Next cohort info
        if (!empty($next_cohort_date)) {
            $output .= '<div class="bw-course-cohort-info">';
            $output .= sprintf('<p class="bw-cohort-starts"><strong>Next cohort starts:</strong> %s</p>', $next_cohort_date);
            if (!empty($session_schedule)) {
                $output .= sprintf('<p class="bw-cohort-schedule">%s</p>', $session_schedule);
            }
            $output .= '</div>';
        }
        
        // Desktop only CTA (hidden on mobile when card is shown)
        if (!$show_enrollment_card) {
            $output .= '<div class="bw-course-cta desktop-only">';
            $output .= sprintf('<div class="bw-course-price">%s</div>', $price);
            $output .= sprintf('<a href="%s" class="bw-btn bw-btn-primary bw-btn-large">%s</a>', $enrollment_url, $enrollment_text);
            $output .= '</div>';
        }
        
        $output .= '</div>'; // course-header-content
        
        // Enrollment card section
        if ($show_enrollment_card) {
            $output .= '<div class="bw-enrollment-card">';
            $output .= '<div class="bw-enrollment-card-inner">';
            $output .= '<h3 class="bw-enrollment-title">Enroll Today</h3>';
            $output .= sprintf('<div class="bw-enrollment-price">%s</div>', $price);
            if (!empty($price_subtext)) {
                $output .= sprintf('<p class="bw-enrollment-subtext">%s</p>', $price_subtext);
            }
            
            // Highlight box
            $output .= '<div class="bw-enrollment-highlight">';
            $output .= '<p>Lorem ipsum dolor sit amet + consectetur adipiscing</p>';
            $output .= '</div>';
            
            // Features list
            if (!empty($enrollment_features)) {
                $output .= '<ul class="bw-enrollment-features">';
                foreach ($enrollment_features as $feature) {
                    $output .= sprintf('<li>%s</li>', esc_html($feature));
                }
                $output .= '</ul>';
            }
            
            // Buttons
            $output .= '<div class="bw-enrollment-buttons">';
            $output .= sprintf('<a href="%s" class="bw-btn bw-btn-primary bw-btn-block">%s</a>', $enrollment_url, $enrollment_text);
            if (!empty($secondary_button_text)) {
                $output .= sprintf('<a href="%s" class="bw-btn bw-btn-outline bw-btn-block">%s</a>', $secondary_button_url, $secondary_button_text);
            }
            $output .= '</div>';
            
            if (!empty($guarantee)) {
                $output .= sprintf('<p class="bw-enrollment-guarantee">%s</p>', $guarantee);
            }
            
            $output .= '</div>'; // enrollment-card-inner
            $output .= '</div>'; // enrollment-card
        }
        
        $output .= '</div>'; // course-header-wrapper
        $output .= '</div>'; // course-header
        
        return $output;
    }
    
    public function bw_render_course_curriculum_block($attributes) {
        $title = esc_html($attributes['title']);
        $modules = $attributes['modules'];
        $show_duration = isset($attributes['showDuration']) ? $attributes['showDuration'] : true;
        
        $output = '<div class="bw-course-curriculum">';
        $output .= sprintf('<h2 class="bw-curriculum-title">%s</h2>', $title);
        $output .= '<div class="bw-curriculum-modules">';
        
        foreach ($modules as $index => $module) {
            $module_id = 'module-' . ($index + 1);
            $output .= sprintf('<div class="bw-curriculum-module" data-module="%s">', $module_id);
            $output .= '<div class="bw-module-header" onclick="bwToggleModule(this)">';
            
            if ($show_duration && !empty($module['duration'])) {
                $output .= sprintf('<div class="bw-module-info"><h3>%s</h3><span class="bw-module-duration">%s</span></div>', 
                    esc_html($module['title']), 
                    esc_html($module['duration'])
                );
            } else {
                $output .= sprintf('<div class="bw-module-info"><h3>%s</h3></div>', 
                    esc_html($module['title'])
                );
            }
            
            $output .= '<svg class="bw-module-toggle" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="6 9 12 15 18 9"/></svg>';
            $output .= '</div>';
            
            $output .= '<div class="bw-module-content">';
            $output .= '<ul class="bw-module-lessons">';
            foreach ($module['lessons'] as $lesson) {
                $output .= sprintf('<li>%s</li>', esc_html($lesson));
            }
            $output .= '</ul>';
            $output .= '</div>';
            $output .= '</div>';
        }
        
        $output .= '</div>';
        $output .= '</div>';
        
        // Add JavaScript for accordion functionality
        static $script_added = false;
        if (!$script_added) {
            $output .= '<script>
            function bwToggleModule(header) {
                const module = header.parentElement;
                const content = module.querySelector(".bw-module-content");
                const toggle = header.querySelector(".bw-module-toggle");
                
                module.classList.toggle("active");
                
                if (module.classList.contains("active")) {
                    content.style.maxHeight = content.scrollHeight + "px";
                } else {
                    content.style.maxHeight = "0";
                }
            }
            </script>';
            $script_added = true;
        }
        
        return $output;
    }
    
    public function bw_render_course_instructor_block($attributes) {
        $name = esc_html($attributes['name']);
        $title = esc_html($attributes['title']);
        $bio = esc_html($attributes['bio']);
        $image_url = esc_url($attributes['imageUrl']);
        $credentials = $attributes['credentials'];
        
        $output = '<div class="bw-course-instructor">';
        $output .= '<div class="bw-instructor-card">';
        $output .= sprintf('<img src="%s" alt="%s" class="bw-instructor-image">', $image_url, $name);
        $output .= '<div class="bw-instructor-info">';
        $output .= sprintf('<h3 class="bw-instructor-name">%s</h3>', $name);
        $output .= sprintf('<p class="bw-instructor-title">%s</p>', $title);
        $output .= sprintf('<p class="bw-instructor-bio">%s</p>', $bio);
        
        if (!empty($credentials)) {
            $output .= '<ul class="bw-instructor-credentials">';
            foreach ($credentials as $credential) {
                $output .= sprintf('<li>%s</li>', esc_html($credential));
            }
            $output .= '</ul>';
        }
        
        $output .= '</div>';
        $output .= '</div>';
        $output .= '</div>';
        
        return $output;
    }
    
    public function bw_render_course_faq_block($attributes) {
        $title = esc_html($attributes['title']);
        $faqs = $attributes['faqs'];
        
        $output = '<div class="bw-course-faq">';
        $output .= sprintf('<h2 class="bw-faq-title">%s</h2>', $title);
        $output .= '<div class="bw-faq-list">';
        
        foreach ($faqs as $index => $faq) {
            $faq_id = 'faq-' . ($index + 1);
            $output .= sprintf('<div class="bw-faq-item" data-faq="%s">', $faq_id);
            $output .= '<div class="bw-faq-question" onclick="bwToggleFAQ(this)">';
            $output .= sprintf('<h3>%s</h3>', esc_html($faq['question']));
            $output .= '<svg class="bw-faq-toggle" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg>';
            $output .= '</div>';
            $output .= sprintf('<div class="bw-faq-answer"><p>%s</p></div>', esc_html($faq['answer']));
            $output .= '</div>';
        }
        
        $output .= '</div>';
        $output .= '</div>';
        
        // Add JavaScript for FAQ accordion
        static $script_added = false;
        if (!$script_added) {
            $output .= '<script>
            function bwToggleFAQ(question) {
                const item = question.parentElement;
                const answer = item.querySelector(".bw-faq-answer");
                const toggle = question.querySelector(".bw-faq-toggle");
                
                item.classList.toggle("active");
                
                if (item.classList.contains("active")) {
                    answer.style.maxHeight = answer.scrollHeight + "px";
                } else {
                    answer.style.maxHeight = "0";
                }
            }
            </script>';
            $script_added = true;
        }
        
        return $output;
    }
    
    public function bw_render_value_grid_item_block($attributes) {
        $number = esc_html($attributes['number']);
        $label = esc_html($attributes['label']);
        $icon = esc_html($attributes['icon']);
        $color_scheme = esc_attr($attributes['colorScheme']);
        $animate_number = $attributes['animateNumber'];
        
        // Extract numeric value for animation
        $numeric_value = preg_replace('/[^0-9]/', '', $number);
        $suffix = preg_replace('/[0-9]/', '', $number);
        
        $output = sprintf(
            '<div class="bw-value-grid-item bw-value-%s" data-animate="%s">
                <div class="bw-value-icon">%s</div>
                <div class="bw-value-content">
                    <div class="bw-value-number" data-value="%s" data-suffix="%s">%s</div>
                    <div class="bw-value-label">%s</div>
                </div>
            </div>',
            $color_scheme,
            $animate_number ? 'true' : 'false',
            $icon,
            $numeric_value,
            $suffix,
            $number,
            $label
        );
        
        // Add initialization script for number animation
        if ($animate_number) {
            $output .= '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting && !entry.target.classList.contains("animated")) {
                            const element = entry.target.querySelector(".bw-value-number");
                            if (element) {
                                const finalValue = parseInt(element.dataset.value);
                                const suffix = element.dataset.suffix;
                                let currentValue = 0;
                                const increment = Math.ceil(finalValue / 50);
                                const timer = setInterval(() => {
                                    currentValue += increment;
                                    if (currentValue >= finalValue) {
                                        currentValue = finalValue;
                                        clearInterval(timer);
                                    }
                                    element.textContent = currentValue + suffix;
                                }, 30);
                                entry.target.classList.add("animated");
                            }
                        }
                    });
                }, { threshold: 0.5 });
                
                document.querySelectorAll(".bw-value-grid-item[data-animate=\"true\"]").forEach(item => {
                    observer.observe(item);
                });
            });
            </script>';
        }
        
        return $output;
    }
}

// Initialize the plugin
BW_Blocks::get_instance();