/**
 * BW Blocks Editor JavaScript
 * Registers custom Gutenberg blocks
 */

(function(wp) {
    const { registerBlockType } = wp.blocks;
    const { __ } = wp.i18n;
    const { InspectorControls, RichText, AlignmentToolbar, BlockControls } = wp.blockEditor || wp.editor;
    const { PanelBody, TextControl, ToggleControl, Toolbar } = wp.components;
    const { Fragment, createElement: el } = wp.element;

    // BW Badge Block
    registerBlockType('bw-blocks/badge', {
        title: __('BW Badge', 'bw-blocks'),
        description: __('Hero badge with icon and text', 'bw-blocks'),
        category: 'design',
        icon: 'award',
        keywords: ['badge', 'hero', 'bw'],
        attributes: {
            icon: {
                type: 'string',
                default: '✓'
            },
            text: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet'
            },
            animation: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { icon, text, animation } = attributes;

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Badge Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Icon', 'bw-blocks'),
                            value: icon,
                            onChange: (value) => setAttributes({ icon: value }),
                            help: __('Enter an emoji or character for the icon', 'bw-blocks')
                        }),
                        el(ToggleControl, {
                            label: __('Enable Animation', 'bw-blocks'),
                            checked: animation,
                            onChange: (value) => setAttributes({ animation: value })
                        })
                    )
                ),
                el('div', {
                    style: {
                        textAlign: 'center',
                        display: 'block'
                    }
                },
                    el('div', { 
                        className: 'bw-badge' + (animation ? ' bw-animated' : ''),
                        style: { 
                            display: 'inline-flex',
                            alignItems: 'center',
                            gap: '10px',
                            background: 'rgba(102, 126, 234, 0.9)',
                            border: '1px solid rgba(255,255,255,0.2)',
                            padding: '10px 24px',
                            borderRadius: '50px',
                            color: 'white'
                        }
                    },
                    el('span', { 
                        className: 'bw-badge-icon',
                        style: {
                            width: '20px',
                            height: '20px',
                            background: 'linear-gradient(135deg, #00d4aa 0%, #00b894 100%)',
                            borderRadius: '50%',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            fontSize: '12px',
                            fontWeight: 'bold',
                            color: 'white'
                        }
                    }, icon),
                        el(RichText, {
                            tagName: 'span',
                            value: text,
                            onChange: (value) => setAttributes({ text: value }),
                            placeholder: __('Badge text...', 'bw-blocks'),
                            style: {
                                color: 'white'
                            }
                        })
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Hero H1 Block
    registerBlockType('bw-blocks/hero-h1', {
        title: __('BW Hero H1', 'bw-blocks'),
        description: __('Large hero heading with gradient effect', 'bw-blocks'),
        category: 'design',
        icon: 'heading',
        keywords: ['heading', 'h1', 'hero', 'bw'],
        attributes: {
            text: {
                type: 'string',
                default: 'Lorem Ipsum<br>Dolor Sit Amet'
            },
            gradient: {
                type: 'boolean',
                default: true
            },
            glitchHover: {
                type: 'boolean',
                default: true
            },
            alignment: {
                type: 'string',
                default: 'center'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { text, gradient, glitchHover, alignment } = attributes;

            return el(Fragment, {},
                el(BlockControls, {},
                    el(AlignmentToolbar, {
                        value: alignment,
                        onChange: (value) => setAttributes({ alignment: value })
                    })
                ),
                el(InspectorControls, {},
                    el(PanelBody, { title: __('H1 Settings', 'bw-blocks') },
                        el(ToggleControl, {
                            label: __('Gradient Text Effect', 'bw-blocks'),
                            checked: gradient,
                            onChange: (value) => setAttributes({ gradient: value })
                        }),
                        el(ToggleControl, {
                            label: __('Glitch Hover Effect', 'bw-blocks'),
                            checked: glitchHover,
                            onChange: (value) => setAttributes({ glitchHover: value })
                        })
                    )
                ),
                el(RichText, {
                    tagName: 'h1',
                    value: text,
                    onChange: (value) => setAttributes({ text: value }),
                    placeholder: __('Hero heading...', 'bw-blocks'),
                    className: 'bw-hero-h1' + (gradient ? ' bw-gradient-text' : ''),
                    style: {
                        fontSize: '60px',
                        fontWeight: '900',
                        lineHeight: '1.1',
                        textAlign: alignment,
                        color: gradient ? 'white' : 'white',
                        background: gradient ? 'linear-gradient(to right, #ffffff 0%, rgba(255,255,255,0.9) 100%)' : 'none',
                        WebkitBackgroundClip: gradient ? 'text' : 'initial',
                        WebkitTextFillColor: gradient ? 'transparent' : 'white'
                    }
                })
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Program Block
    registerBlockType('bw-blocks/program', {
        title: __('BW Program', 'bw-blocks'),
        description: __('Program card with features and pricing', 'bw-blocks'),
        category: 'design',
        icon: 'welcome-learn-more',
        keywords: ['program', 'course', 'card', 'bw'],
        attributes: {
            colorScheme: {
                type: 'string',
                default: 'flagship'
            },
            badge: {
                type: 'string',
                default: 'FLAGSHIP PROGRAM'
            },
            title: {
                type: 'string',
                default: 'Program Title'
            },
            description: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.'
            },
            features: {
                type: 'array',
                default: [
                    'Feature one goes here',
                    'Feature two goes here',
                    'Feature three goes here',
                    'Feature four goes here'
                ]
            },
            price: {
                type: 'string',
                default: '$999'
            },
            priceNote: {
                type: 'string',
                default: 'Early bird available'
            },
            buttonText: {
                type: 'string',
                default: 'Enroll Now →'
            },
            buttonUrl: {
                type: 'string',
                default: '#'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { colorScheme, badge, title, description, features, price, priceNote, buttonText, buttonUrl } = attributes;

            const colorOptions = [
                { label: 'Flagship (Purple)', value: 'flagship' },
                { label: 'Agency (Green)', value: 'agency' },
                { label: 'Self-Paced (Pink)', value: 'self' },
                { label: 'Legal (Yellow)', value: 'legal' }
            ];

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Program Settings', 'bw-blocks') },
                        el('div', { className: 'components-base-control' },
                            el('label', { className: 'components-base-control__label' }, __('Color Scheme', 'bw-blocks')),
                            el('select', {
                                value: colorScheme,
                                onChange: (e) => setAttributes({ colorScheme: e.target.value }),
                                style: { width: '100%' }
                            },
                                colorOptions.map(option => 
                                    el('option', { value: option.value, key: option.value }, option.label)
                                )
                            )
                        ),
                        el(TextControl, {
                            label: __('Badge Text', 'bw-blocks'),
                            value: badge,
                            onChange: (value) => setAttributes({ badge: value })
                        }),
                        el(TextControl, {
                            label: __('Title', 'bw-blocks'),
                            value: title,
                            onChange: (value) => setAttributes({ title: value })
                        }),
                        el('textarea', {
                            placeholder: __('Description', 'bw-blocks'),
                            value: description,
                            onChange: (e) => setAttributes({ description: e.target.value }),
                            style: { width: '100%', minHeight: '80px' }
                        })
                    ),
                    el(PanelBody, { title: __('Features', 'bw-blocks'), initialOpen: false },
                        features.map((feature, index) => 
                            el(TextControl, {
                                key: index,
                                label: __('Feature ', 'bw-blocks') + (index + 1),
                                value: feature,
                                onChange: (value) => {
                                    const newFeatures = [...features];
                                    newFeatures[index] = value;
                                    setAttributes({ features: newFeatures });
                                }
                            })
                        )
                    ),
                    el(PanelBody, { title: __('Pricing & Button', 'bw-blocks'), initialOpen: false },
                        el(TextControl, {
                            label: __('Price', 'bw-blocks'),
                            value: price,
                            onChange: (value) => setAttributes({ price: value })
                        }),
                        el(TextControl, {
                            label: __('Price Note', 'bw-blocks'),
                            value: priceNote,
                            onChange: (value) => setAttributes({ priceNote: value })
                        }),
                        el(TextControl, {
                            label: __('Button Text', 'bw-blocks'),
                            value: buttonText,
                            onChange: (value) => setAttributes({ buttonText: value })
                        }),
                        el(TextControl, {
                            label: __('Button URL', 'bw-blocks'),
                            value: buttonUrl,
                            onChange: (value) => setAttributes({ buttonUrl: value })
                        })
                    )
                ),
                el('div', { 
                    className: 'bw-program-card bw-program-' + colorScheme,
                    style: {
                        maxWidth: '400px',
                        margin: '0 auto',
                        borderRadius: '24px',
                        overflow: 'hidden',
                        boxShadow: '0 10px 30px rgba(0,0,0,0.08)'
                    }
                },
                    el('div', {
                        className: 'bw-program-header',
                        style: {
                            padding: '30px',
                            color: 'white',
                            background: colorScheme === 'flagship' ? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)' :
                                       colorScheme === 'agency' ? 'linear-gradient(135deg, #00d4aa 0%, #00b894 100%)' :
                                       colorScheme === 'self' ? 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)' :
                                       'linear-gradient(135deg, #fa709a 0%, #fee140 100%)'
                        }
                    },
                        el('span', {
                            style: {
                                display: 'inline-block',
                                fontSize: '11px',
                                fontWeight: '700',
                                letterSpacing: '1px',
                                textTransform: 'uppercase',
                                marginBottom: '10px',
                                opacity: '0.9'
                            }
                        }, badge),
                        el('h3', {
                            style: {
                                fontSize: '24px',
                                fontWeight: '700',
                                margin: '0'
                            }
                        }, title)
                    ),
                    el('div', {
                        className: 'bw-program-body',
                        style: {
                            padding: '30px',
                            background: 'white'
                        }
                    },
                        el('p', {
                            style: {
                                fontSize: '15px',
                                lineHeight: '1.6',
                                color: '#666',
                                marginBottom: '20px'
                            }
                        }, description),
                        el('div', { className: 'bw-program-features' },
                            features.map((feature, index) => 
                                el('div', {
                                    key: index,
                                    style: {
                                        display: 'flex',
                                        alignItems: 'flex-start',
                                        gap: '10px',
                                        marginBottom: '12px'
                                    }
                                },
                                    el('span', {
                                        style: {
                                            color: '#00d4aa',
                                            fontWeight: 'bold'
                                        }
                                    }, '✓'),
                                    el('span', {
                                        style: {
                                            fontSize: '14px',
                                            color: '#333'
                                        }
                                    }, feature)
                                )
                            )
                        ),
                        el('div', {
                            style: {
                                marginTop: '20px',
                                paddingTop: '20px',
                                borderTop: '1px solid #eee',
                                display: 'flex',
                                justifyContent: 'space-between',
                                alignItems: 'center'
                            }
                        },
                            el('div', {},
                                el('div', {
                                    style: {
                                        fontSize: '28px',
                                        fontWeight: '700',
                                        color: '#333'
                                    }
                                }, price),
                                priceNote && el('div', {
                                    style: {
                                        fontSize: '12px',
                                        color: '#999',
                                        marginTop: '2px'
                                    }
                                }, priceNote)
                            ),
                            el('a', {
                                href: buttonUrl,
                                style: {
                                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                                    color: 'white',
                                    padding: '12px 24px',
                                    borderRadius: '50px',
                                    fontSize: '14px',
                                    fontWeight: '600',
                                    textDecoration: 'none'
                                }
                            }, buttonText)
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Buttons Block
    registerBlockType('bw-blocks/buttons', {
        title: __('BW Buttons', 'bw-blocks'),
        description: __('Hero call-to-action buttons', 'bw-blocks'),
        category: 'design',
        icon: 'button',
        keywords: ['button', 'cta', 'hero', 'bw'],
        attributes: {
            primaryText: {
                type: 'string',
                default: 'Button Text →'
            },
            primaryUrl: {
                type: 'string',
                default: '#'
            },
            secondaryText: {
                type: 'string',
                default: 'Button Text'
            },
            secondaryUrl: {
                type: 'string',
                default: '#'
            },
            showSecondary: {
                type: 'boolean',
                default: true
            },
            alignment: {
                type: 'string',
                default: 'center'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { primaryText, primaryUrl, secondaryText, secondaryUrl, showSecondary, alignment } = attributes;

            return el(Fragment, {},
                el(BlockControls, {},
                    el(AlignmentToolbar, {
                        value: alignment,
                        onChange: (value) => setAttributes({ alignment: value })
                    })
                ),
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Primary Button', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Text', 'bw-blocks'),
                            value: primaryText,
                            onChange: (value) => setAttributes({ primaryText: value })
                        }),
                        el(TextControl, {
                            label: __('URL', 'bw-blocks'),
                            value: primaryUrl,
                            onChange: (value) => setAttributes({ primaryUrl: value })
                        })
                    ),
                    el(PanelBody, { title: __('Secondary Button', 'bw-blocks') },
                        el(ToggleControl, {
                            label: __('Show Secondary Button', 'bw-blocks'),
                            checked: showSecondary,
                            onChange: (value) => setAttributes({ showSecondary: value })
                        }),
                        showSecondary && el(TextControl, {
                            label: __('Text', 'bw-blocks'),
                            value: secondaryText,
                            onChange: (value) => setAttributes({ secondaryText: value })
                        }),
                        showSecondary && el(TextControl, {
                            label: __('URL', 'bw-blocks'),
                            value: secondaryUrl,
                            onChange: (value) => setAttributes({ secondaryUrl: value })
                        })
                    )
                ),
                el('div', { 
                    className: 'bw-buttons',
                    style: {
                        display: 'flex',
                        gap: '20px',
                        justifyContent: alignment,
                        flexWrap: 'wrap'
                    }
                },
                    el('a', {
                        href: primaryUrl,
                        className: 'bw-btn bw-btn-primary',
                        style: {
                            background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                            color: 'white',
                            padding: '16px 32px',
                            borderRadius: '50px',
                            fontSize: '16px',
                            fontWeight: '600',
                            textDecoration: 'none',
                            display: 'inline-block',
                            transition: 'all 0.3s',
                            textTransform: 'uppercase',
                            letterSpacing: '0.5px'
                        }
                    }, primaryText),
                    showSecondary && el('a', {
                        href: secondaryUrl,
                        className: 'bw-btn bw-btn-secondary',
                        style: {
                            background: 'transparent',
                            color: '#667eea',
                            border: '2px solid #667eea',
                            padding: '14px 30px',
                            borderRadius: '50px',
                            fontSize: '16px',
                            fontWeight: '600',
                            textDecoration: 'none',
                            display: 'inline-block',
                            transition: 'all 0.3s',
                            textTransform: 'uppercase',
                            letterSpacing: '0.5px'
                        }
                    }, secondaryText)
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Gallery Grid Block
    registerBlockType('bw-blocks/gallery-grid', {
        title: __('BW Gallery Grid', 'bw-blocks'),
        description: __('Masonry gallery grid with mobile carousel', 'bw-blocks'),
        category: 'design',
        icon: 'images-alt2',
        keywords: ['gallery', 'grid', 'images', 'bw'],
        attributes: {
            images: {
                type: 'array',
                default: [
                    {
                        url: 'https://via.placeholder.com/600x520/667eea/ffffff?text=Featured',
                        title: 'Featured Project',
                        description: 'Our flagship design showcasing innovation'
                    },
                    {
                        url: 'https://via.placeholder.com/300x250/764ba2/ffffff?text=Project+2',
                        title: 'Web Development',
                        description: 'Modern responsive web applications'
                    },
                    {
                        url: 'https://via.placeholder.com/300x250/00d4aa/ffffff?text=Project+3',
                        title: 'Mobile Design',
                        description: 'User-centric mobile experiences'
                    },
                    {
                        url: 'https://via.placeholder.com/300x250/f093fb/ffffff?text=Project+4',
                        title: 'Brand Identity',
                        description: 'Complete brand transformation'
                    },
                    {
                        url: 'https://via.placeholder.com/300x250/f5576c/ffffff?text=Project+5',
                        title: 'UI/UX Design',
                        description: 'Intuitive interface design'
                    }
                ]
            },
            gap: {
                type: 'number',
                default: 20
            },
            borderRadius: {
                type: 'number',
                default: 16
            },
            enableLightbox: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { images, gap, borderRadius, enableLightbox } = attributes;
            const { MediaUpload, MediaUploadCheck } = wp.blockEditor || wp.editor;
            const { Button, RangeControl } = wp.components;
            
            const onSelectImages = (media) => {
                const newImages = media.map((item, index) => ({
                    id: item.id,
                    url: item.url,
                    alt: item.alt,
                    title: item.title || `Gallery Image ${index + 1}`,
                    description: item.caption || item.description || 'Click to view details'
                }));
                setAttributes({ images: newImages });
            };

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Gallery Settings', 'bw-blocks') },
                        el('div', { className: 'bw-blocks-help' },
                            el('strong', {}, __('Layout Info:', 'bw-blocks')),
                            el('p', {}, __('This gallery displays 5 images: 1 large featured image on the left, and 4 smaller images in a 2x2 grid on the right.', 'bw-blocks'))
                        ),
                        el(RangeControl, {
                            label: __('Gap (px)', 'bw-blocks'),
                            value: gap,
                            onChange: (value) => setAttributes({ gap: value }),
                            min: 0,
                            max: 50
                        }),
                        el(RangeControl, {
                            label: __('Border Radius (px)', 'bw-blocks'),
                            value: borderRadius,
                            onChange: (value) => setAttributes({ borderRadius: value }),
                            min: 0,
                            max: 30
                        }),
                        el(ToggleControl, {
                            label: __('Enable Lightbox', 'bw-blocks'),
                            checked: enableLightbox,
                            onChange: (value) => setAttributes({ enableLightbox: value })
                        })
                    )
                ),
                el('div', { className: 'bw-gallery-grid-editor' },
                    el('div', { className: 'bw-gallery-info' },
                        el('span', { className: 'bw-gallery-info-icon' }, '🖼️'),
                        el('div', { className: 'bw-gallery-info-text' },
                            el('strong', {}, 'Gallery Layout: '),
                            'Displays exactly 5 images - 1 large featured image + 4 grid images'
                        )
                    ),
                    el(MediaUploadCheck, {},
                        el(MediaUpload, {
                            onSelect: onSelectImages,
                            allowedTypes: ['image'],
                            multiple: true,
                            value: images.map(img => img.id),
                            render: function(obj) {
                                return el('div', {},
                                    images.length === 0 ?
                                        el('div', { className: 'bw-gallery-empty-state' },
                                            el('h3', {}, __('BW Gallery Grid', 'bw-blocks')),
                                            el('p', {}, __('Add exactly 5 images to create your gallery', 'bw-blocks')),
                                            el('p', { style: { fontSize: '12px', color: '#999' } }, 
                                                __('Layout: 1 large image (left) + 4 small images (right grid)', 'bw-blocks')
                                            ),
                                            el(Button, {
                                                className: 'bw-gallery-add-button',
                                                onClick: obj.open,
                                                isPrimary: true
                                            }, __('Select 5 Images', 'bw-blocks'))
                                        )
                                    :
                                        el(Fragment, {},
                                            images.length !== 5 && el('div', {
                                                style: {
                                                    background: '#fff3cd',
                                                    border: '1px solid #ffc107',
                                                    borderRadius: '4px',
                                                    padding: '10px',
                                                    marginBottom: '15px',
                                                    fontSize: '14px',
                                                    color: '#856404'
                                                }
                                            }, `⚠️ Gallery requires exactly 5 images. Currently: ${images.length} image(s)`),
                                            el('div', {
                                                className: 'bw-gallery-preview',
                                                style: {
                                                    gap: gap + 'px'
                                                }
                                            },
                                                images.slice(0, 5).map((image, index) =>
                                                    el('div', {
                                                        key: index,
                                                        className: 'bw-gallery-preview-item',
                                                        style: {
                                                            borderRadius: borderRadius + 'px'
                                                        }
                                                    },
                                                        el('img', {
                                                            src: image.url,
                                                            alt: image.alt || ''
                                                        })
                                                    )
                                                )
                                            ),
                                            el(Button, {
                                                onClick: obj.open,
                                                isSecondary: true,
                                                style: { marginTop: '10px' }
                                            }, images.length === 5 ? __('Change Images', 'bw-blocks') : __('Select 5 Images', 'bw-blocks'))
                                        )
                                );
                            }
                        })
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Value Card Block
    registerBlockType('bw-blocks/value-card', {
        title: __('BW Value Card', 'bw-blocks'),
        description: __('Value card with icon, title and description', 'bw-blocks'),
        category: 'design',
        icon: 'id-alt',
        keywords: ['value', 'card', 'icon', 'bw'],
        attributes: {
            icon: {
                type: 'string',
                default: '🎯'
            },
            title: {
                type: 'string',
                default: 'Lorem Ipsum Dolor'
            },
            description: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
            },
            iconColor: {
                type: 'string',
                default: 'purple'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { icon, title, description, iconColor } = attributes;
            const { SelectControl, TextareaControl } = wp.components;
            
            const colorOptions = [
                { label: 'Purple/Orange Gradient', value: 'purple' },
                { label: 'Blue/Purple Gradient', value: 'blue' },
                { label: 'Green/Blue Gradient', value: 'green' },
                { label: 'Pink/Orange Gradient', value: 'pink' }
            ];

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Value Card Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Icon (Emoji)', 'bw-blocks'),
                            value: icon,
                            onChange: (value) => setAttributes({ icon: value }),
                            help: __('Enter an emoji for the icon', 'bw-blocks')
                        }),
                        el(TextControl, {
                            label: __('Title', 'bw-blocks'),
                            value: title,
                            onChange: (value) => setAttributes({ title: value })
                        }),
                        el(TextareaControl, {
                            label: __('Description', 'bw-blocks'),
                            value: description,
                            onChange: (value) => setAttributes({ description: value }),
                            rows: 4
                        }),
                        el(SelectControl, {
                            label: __('Icon Color', 'bw-blocks'),
                            value: iconColor,
                            options: colorOptions,
                            onChange: (value) => setAttributes({ iconColor: value })
                        })
                    )
                ),
                el('div', { 
                    className: 'bw-value-card-editor',
                    style: {
                        padding: '30px',
                        background: 'white',
                        borderRadius: '16px',
                        boxShadow: '0 4px 20px rgba(0,0,0,0.08)',
                        textAlign: 'left'
                    }
                },
                    el('div', {
                        style: {
                            width: '60px',
                            height: '60px',
                            borderRadius: '12px',
                            background: iconColor === 'purple' ? 'linear-gradient(135deg, #8b5cf6 0%, #ec4899 100%)' :
                                       iconColor === 'blue' ? 'linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%)' :
                                       iconColor === 'green' ? 'linear-gradient(135deg, #10b981 0%, #3b82f6 100%)' :
                                       'linear-gradient(135deg, #ec4899 0%, #f97316 100%)',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            fontSize: '28px',
                            marginBottom: '20px'
                        }
                    }, icon),
                    el('h3', {
                        style: {
                            fontSize: '20px',
                            fontWeight: '700',
                            color: '#1a1a2e',
                            marginBottom: '12px',
                            fontFamily: 'Inter, sans-serif'
                        }
                    }, title),
                    el('p', {
                        style: {
                            fontSize: '15px',
                            color: '#666',
                            lineHeight: '1.6',
                            fontFamily: 'Inter, sans-serif',
                            margin: 0
                        }
                    }, description)
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Service Block
    registerBlockType('bw-blocks/service', {
        title: __('BW Service Block', 'bw-blocks'),
        description: __('Service detail block with icon, title, description, features, button and images', 'bw-blocks'),
        category: 'design',
        icon: 'admin-tools',
        keywords: ['service', 'detail', 'features', 'bw'],
        attributes: {
            icon: {
                type: 'string',
                default: '🚀'
            },
            title: {
                type: 'string',
                default: 'Service Title'
            },
            description: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
            },
            metadata: {
                type: 'string',
                default: 'Professional Service'
            },
            features: {
                type: 'array',
                default: [
                    'Feature one goes here',
                    'Feature two goes here',
                    'Feature three goes here'
                ]
            },
            buttonText: {
                type: 'string',
                default: 'Learn More →'
            },
            buttonUrl: {
                type: 'string',
                default: '#'
            },
            images: {
                type: 'array',
                default: [
                    { url: 'https://via.placeholder.com/400x300/667eea/ffffff?text=Service+1', alt: 'Service Image 1' },
                    { url: 'https://via.placeholder.com/400x300/764ba2/ffffff?text=Service+2', alt: 'Service Image 2' },
                    { url: 'https://via.placeholder.com/400x300/00d4aa/ffffff?text=Service+3', alt: 'Service Image 3' }
                ]
            },
            colorScheme: {
                type: 'string',
                default: 'primary'
            },
            showImages: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { icon, title, description, metadata, features, buttonText, buttonUrl, images, colorScheme, showImages } = attributes;
            const { SelectControl, TextareaControl, Button } = wp.components;
            const { MediaUpload, MediaUploadCheck } = wp.blockEditor;
            
            const colorOptions = [
                { label: 'Primary (Blue/Purple)', value: 'primary' },
                { label: 'Accent (Teal/Green)', value: 'accent' },
                { label: 'Warm (Pink/Orange)', value: 'warm' },
                { label: 'Dark (Dark Blue)', value: 'dark' }
            ];
            
            const updateFeature = (index, value) => {
                const newFeatures = [...features];
                newFeatures[index] = value;
                setAttributes({ features: newFeatures });
            };
            
            const addFeature = () => {
                const newFeatures = [...features, 'New feature'];
                setAttributes({ features: newFeatures });
            };
            
            const removeFeature = (index) => {
                const newFeatures = features.filter((_, i) => i !== index);
                setAttributes({ features: newFeatures });
            };

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Service Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Icon (Emoji)', 'bw-blocks'),
                            value: icon,
                            onChange: (value) => setAttributes({ icon: value }),
                            help: __('Enter an emoji for the service icon', 'bw-blocks')
                        }),
                        el(TextControl, {
                            label: __('Service Title', 'bw-blocks'),
                            value: title,
                            onChange: (value) => setAttributes({ title: value })
                        }),
                        el(TextControl, {
                            label: __('Metadata/Category', 'bw-blocks'),
                            value: metadata,
                            onChange: (value) => setAttributes({ metadata: value }),
                            help: __('e.g., Professional Service, Enterprise Solution', 'bw-blocks')
                        }),
                        el(TextareaControl, {
                            label: __('Description', 'bw-blocks'),
                            value: description,
                            onChange: (value) => setAttributes({ description: value }),
                            rows: 4
                        }),
                        el(SelectControl, {
                            label: __('Color Scheme', 'bw-blocks'),
                            value: colorScheme,
                            options: colorOptions,
                            onChange: (value) => setAttributes({ colorScheme: value })
                        })
                    ),
                    el(PanelBody, { title: __('Button Settings', 'bw-blocks'), initialOpen: false },
                        el(TextControl, {
                            label: __('Button Text', 'bw-blocks'),
                            value: buttonText,
                            onChange: (value) => setAttributes({ buttonText: value })
                        }),
                        el(TextControl, {
                            label: __('Button URL', 'bw-blocks'),
                            value: buttonUrl,
                            onChange: (value) => setAttributes({ buttonUrl: value }),
                            help: __('Enter the destination URL for the button', 'bw-blocks')
                        })
                    ),
                    el(PanelBody, { title: __('Features', 'bw-blocks'), initialOpen: false },
                        features.map((feature, index) => 
                            el('div', { 
                                key: index,
                                style: { 
                                    display: 'flex', 
                                    alignItems: 'center', 
                                    marginBottom: '10px',
                                    gap: '10px'
                                }
                            },
                                el(TextControl, {
                                    value: feature,
                                    onChange: (value) => updateFeature(index, value),
                                    style: { flex: 1 }
                                }),
                                el(Button, {
                                    isDestructive: true,
                                    isSmall: true,
                                    onClick: () => removeFeature(index)
                                }, '×')
                            )
                        ),
                        el(Button, {
                            isPrimary: true,
                            onClick: addFeature,
                            style: { marginTop: '10px' }
                        }, __('Add Feature', 'bw-blocks'))
                    ),
                    el(PanelBody, { title: __('Service Images', 'bw-blocks'), initialOpen: false },
                        el(ToggleControl, {
                            label: __('Show Images Section', 'bw-blocks'),
                            checked: showImages,
                            onChange: (value) => setAttributes({ showImages: value })
                        }),
                        showImages && el(MediaUploadCheck, {},
                            el(MediaUpload, {
                                onSelect: (media) => {
                                    const newImages = media.map(img => ({
                                        url: img.url,
                                        alt: img.alt || ''
                                    }));
                                    setAttributes({ images: newImages });
                                },
                                allowedTypes: ['image'],
                                multiple: true,
                                gallery: true,
                                value: images.map(img => img.id),
                                render: ({ open }) => (
                                    el(Button, { 
                                        onClick: open,
                                        isPrimary: true,
                                        style: { marginTop: '10px' }
                                    }, images.length > 0 ? __('Change Images', 'bw-blocks') : __('Select Images', 'bw-blocks'))
                                )
                            })
                        )
                    )
                ),
                el('div', { 
                    className: 'bw-service-editor',
                    style: {
                        padding: '40px',
                        background: 'white',
                        borderRadius: '16px',
                        border: '2px solid #e0e0e0'
                    }
                },
                    el('div', {
                        style: {
                            display: 'flex',
                            gap: '30px',
                            alignItems: 'flex-start'
                        }
                    },
                        el('div', {
                            style: {
                                fontSize: '56px',
                                width: '100px',
                                height: '100px',
                                background: colorScheme === 'primary' ? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)' :
                                           colorScheme === 'accent' ? 'linear-gradient(135deg, #00d4aa 0%, #00b894 100%)' :
                                           colorScheme === 'warm' ? 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)' :
                                           'linear-gradient(135deg, #1a1a2e 0%, #16213e 100%)',
                                borderRadius: '16px',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center',
                                color: 'white'
                            }
                        }, icon),
                        el('div', { style: { flex: 1 } },
                            el('h3', { 
                                style: { 
                                    fontSize: '24px',
                                    margin: '0 0 8px 0',
                                    color: '#333'
                                }
                            }, title),
                            metadata && el('div', {
                                style: {
                                    fontSize: '14px',
                                    color: '#666',
                                    marginBottom: '12px',
                                    padding: '8px 16px',
                                    background: 'rgba(102, 126, 234, 0.05)',
                                    borderRadius: '8px',
                                    borderLeft: '3px solid #667eea'
                                }
                            }, el('strong', {}, 'Service Type: '), metadata),
                            el('p', {
                                style: {
                                    fontSize: '15px',
                                    color: '#666',
                                    lineHeight: '1.6',
                                    margin: '0 0 20px 0'
                                }
                            }, description),
                            features.length > 0 && el('ul', {
                                style: {
                                    listStyle: 'none',
                                    padding: 0,
                                    margin: 0
                                }
                            },
                                features.map((feature, index) =>
                                    el('li', {
                                        key: index,
                                        style: {
                                            padding: '8px 0',
                                            paddingLeft: '24px',
                                            position: 'relative',
                                            color: '#555'
                                        }
                                    },
                                        el('span', {
                                            style: {
                                                position: 'absolute',
                                                left: 0,
                                                color: '#00d4aa'
                                            }
                                        }, '✓'),
                                        feature
                                    )
                                )
                            ),
                            buttonText && el('a', {
                                href: '#',
                                style: {
                                    display: 'inline-block',
                                    padding: '14px 28px',
                                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                                    color: 'white',
                                    borderRadius: '50px',
                                    textDecoration: 'none',
                                    fontSize: '14px',
                                    fontWeight: '600',
                                    marginTop: '20px'
                                },
                                onClick: (e) => e.preventDefault()
                            }, buttonText)
                        )
                    ),
                    showImages && images.length > 0 && el(Fragment, {},
                        el('hr', {
                            style: {
                                margin: '40px 0',
                                border: 'none',
                                borderTop: '2px solid #f0f0f0'
                            }
                        }),
                        el('div', {
                            style: {
                                display: 'grid',
                                gridTemplateColumns: 'repeat(3, 1fr)',
                                gap: '20px'
                            }
                        },
                            images.slice(0, 3).map((image, index) =>
                                el('div', {
                                    key: index,
                                    style: {
                                        borderRadius: '12px',
                                        overflow: 'hidden',
                                        background: '#f5f5f5',
                                        aspectRatio: '4/3'
                                    }
                                },
                                    el('img', {
                                        src: image.url,
                                        alt: image.alt || '',
                                        style: {
                                            width: '100%',
                                            height: '100%',
                                            objectFit: 'cover'
                                        }
                                    })
                                )
                            )
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Horizontal Timeline Block
    registerBlockType('bw-blocks/horizontal-timeline', {
        title: __('BW Horizontal Timeline', 'bw-blocks'),
        description: __('Process timeline with numbered steps', 'bw-blocks'),
        category: 'design',
        icon: 'editor-ol',
        keywords: ['timeline', 'process', 'steps', 'horizontal', 'bw'],
        attributes: {
            steps: {
                type: 'array',
                default: [
                    { number: '1', title: 'Real Experience', description: 'Lorem ipsum dolor sit amet consectetur adipiscing' },
                    { number: '2', title: 'Proven Results', description: 'Lorem ipsum dolor sit amet consectetur adipiscing' },
                    { number: '3', title: 'Comprehensive Approach', description: 'Lorem ipsum dolor sit amet consectetur adipiscing' },
                    { number: '4', title: 'Practical Focus', description: 'Lorem ipsum dolor sit amet consectetur adipiscing' }
                ]
            },
            colorScheme: {
                type: 'string',
                default: 'primary'
            },
            showConnector: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { steps, colorScheme, showConnector } = attributes;
            const { SelectControl, Button, TextareaControl } = wp.components;
            
            const colorOptions = [
                { label: 'Primary (Blue/Purple)', value: 'primary' },
                { label: 'Accent (Teal/Green)', value: 'accent' },
                { label: 'Warm (Pink/Orange)', value: 'warm' },
                { label: 'Dark (Dark Blue)', value: 'dark' }
            ];
            
            const updateStep = (index, field, value) => {
                const newSteps = [...steps];
                newSteps[index][field] = value;
                setAttributes({ steps: newSteps });
            };
            
            const addStep = () => {
                const newSteps = [...steps, {
                    number: String(steps.length + 1),
                    title: 'New Step',
                    description: 'Step description'
                }];
                setAttributes({ steps: newSteps });
            };
            
            const removeStep = (index) => {
                const newSteps = steps.filter((_, i) => i !== index);
                // Renumber steps
                newSteps.forEach((step, i) => {
                    step.number = String(i + 1);
                });
                setAttributes({ steps: newSteps });
            };

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Timeline Settings', 'bw-blocks') },
                        el(SelectControl, {
                            label: __('Color Scheme', 'bw-blocks'),
                            value: colorScheme,
                            options: colorOptions,
                            onChange: (value) => setAttributes({ colorScheme: value })
                        }),
                        el(ToggleControl, {
                            label: __('Show Connector Line', 'bw-blocks'),
                            checked: showConnector,
                            onChange: (value) => setAttributes({ showConnector: value })
                        })
                    ),
                    el(PanelBody, { title: __('Timeline Steps', 'bw-blocks'), initialOpen: true },
                        steps.map((step, index) => 
                            el('div', {
                                key: index,
                                style: {
                                    padding: '15px',
                                    marginBottom: '15px',
                                    background: '#f5f5f5',
                                    borderRadius: '8px',
                                    border: '1px solid #e0e0e0'
                                }
                            },
                                el('div', {
                                    style: {
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'space-between',
                                        marginBottom: '10px'
                                    }
                                },
                                    el('h4', { style: { margin: 0, fontSize: '14px', fontWeight: '600' } }, 
                                        __('Step ', 'bw-blocks') + step.number
                                    ),
                                    el(Button, {
                                        isDestructive: true,
                                        isSmall: true,
                                        onClick: () => removeStep(index),
                                        disabled: steps.length <= 2
                                    }, '×')
                                ),
                                el(TextControl, {
                                    label: __('Title', 'bw-blocks'),
                                    value: step.title,
                                    onChange: (value) => updateStep(index, 'title', value)
                                }),
                                el(TextareaControl, {
                                    label: __('Description', 'bw-blocks'),
                                    value: step.description,
                                    onChange: (value) => updateStep(index, 'description', value),
                                    rows: 2
                                })
                            )
                        ),
                        el(Button, {
                            isPrimary: true,
                            onClick: addStep,
                            style: { marginTop: '10px' },
                            disabled: steps.length >= 6
                        }, __('Add Step', 'bw-blocks')),
                        steps.length >= 6 && el('p', {
                            style: { color: '#999', fontSize: '12px', marginTop: '5px' }
                        }, __('Maximum 6 steps allowed', 'bw-blocks'))
                    )
                ),
                el('div', { 
                    className: 'bw-timeline-editor',
                    style: {
                        padding: '40px',
                        background: 'white',
                        borderRadius: '16px',
                        border: '2px solid #e0e0e0',
                        position: 'relative'
                    }
                },
                    showConnector && el('div', {
                        style: {
                            position: 'absolute',
                            top: '70px',
                            left: '10%',
                            right: '10%',
                            height: '3px',
                            background: colorScheme === 'primary' ? 'linear-gradient(90deg, #667eea, #764ba2)' :
                                       colorScheme === 'accent' ? 'linear-gradient(90deg, #00d4aa, #00b894)' :
                                       colorScheme === 'warm' ? 'linear-gradient(90deg, #f093fb, #f5576c)' :
                                       'linear-gradient(90deg, #1a1a2e, #16213e)',
                            opacity: '0.3'
                        }
                    }),
                    el('div', {
                        style: {
                            display: 'flex',
                            justifyContent: 'space-between',
                            position: 'relative'
                        }
                    },
                        steps.map((step, index) =>
                            el('div', {
                                key: index,
                                style: {
                                    textAlign: 'center',
                                    flex: 1,
                                    maxWidth: '200px',
                                    padding: '0 10px'
                                }
                            },
                                el('div', {
                                    style: {
                                        width: '60px',
                                        height: '60px',
                                        background: colorScheme === 'primary' ? 'linear-gradient(135deg, #667eea, #764ba2)' :
                                                   colorScheme === 'accent' ? 'linear-gradient(135deg, #00d4aa, #00b894)' :
                                                   colorScheme === 'warm' ? 'linear-gradient(135deg, #f093fb, #f5576c)' :
                                                   'linear-gradient(135deg, #1a1a2e, #16213e)',
                                        color: 'white',
                                        borderRadius: '50%',
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        margin: '0 auto 20px',
                                        fontWeight: '700',
                                        fontSize: '20px',
                                        boxShadow: '0 10px 20px rgba(102, 126, 234, 0.3)'
                                    }
                                }, step.number),
                                el('h3', {
                                    style: {
                                        fontSize: '16px',
                                        fontWeight: '600',
                                        margin: '0 0 8px 0',
                                        color: '#333'
                                    }
                                }, step.title),
                                el('p', {
                                    style: {
                                        fontSize: '13px',
                                        color: '#666',
                                        margin: 0,
                                        lineHeight: '1.4'
                                    }
                                }, step.description)
                            )
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Course Header Block
    registerBlockType('bw-blocks/course-header', {
        title: __('BW Course Header', 'bw-blocks'),
        description: __('Course header with title, meta info and CTA', 'bw-blocks'),
        category: 'design',
        icon: 'welcome-learn-more',
        keywords: ['course', 'header', 'education', 'bw'],
        attributes: {
            title: {
                type: 'string',
                default: 'AI Fundamentals Masterclass'
            },
            subtitle: {
                type: 'string',
                default: 'Master the foundations of artificial intelligence'
            },
            duration: {
                type: 'string',
                default: '12 Weeks'
            },
            format: {
                type: 'string',
                default: 'Online + Live Sessions'
            },
            level: {
                type: 'string',
                default: 'Beginner to Intermediate'
            },
            certification: {
                type: 'string',
                default: 'Professional Certificate'
            },
            price: {
                type: 'string',
                default: '$2,499'
            },
            enrollmentText: {
                type: 'string',
                default: 'Enroll Now'
            },
            enrollmentUrl: {
                type: 'string',
                default: '#'
            },
            showEnrollmentCard: {
                type: 'boolean',
                default: true
            },
            enrollmentFeatures: {
                type: 'array',
                default: [
                    'Lorem ipsum dolor sit amet',
                    'Consectetur adipiscing elit sed',
                    'Do eiusmod tempor incididunt',
                    'Ut labore et dolore magna',
                    'Aliqua ut enim ad minim',
                    'Veniam quis nostrud exercitation',
                    'Ullamco laboris nisi ut aliquip'
                ]
            },
            priceSubtext: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet'
            },
            secondaryButtonText: {
                type: 'string',
                default: 'Button Text'
            },
            secondaryButtonUrl: {
                type: 'string',
                default: '#'
            },
            guarantee: {
                type: 'string',
                default: '30-day money-back guarantee'
            },
            nextCohortDate: {
                type: 'string',
                default: 'Tuesday, September 16th, 2025'
            },
            sessionSchedule: {
                type: 'string',
                default: 'Weekly sessions on Tuesdays, 2:00-3:00 PM PST'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { 
                title, subtitle, duration, format, level, certification, price, 
                enrollmentText, enrollmentUrl, showEnrollmentCard, enrollmentFeatures,
                priceSubtext, secondaryButtonText, secondaryButtonUrl, guarantee,
                nextCohortDate, sessionSchedule 
            } = attributes;
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Course Header Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Title', 'bw-blocks'),
                            value: title,
                            onChange: (newValue) => setAttributes({ title: newValue })
                        }),
                        el(TextControl, {
                            label: __('Subtitle', 'bw-blocks'),
                            value: subtitle,
                            onChange: (newValue) => setAttributes({ subtitle: newValue })
                        }),
                        el(TextControl, {
                            label: __('Duration', 'bw-blocks'),
                            value: duration,
                            onChange: (newValue) => setAttributes({ duration: newValue })
                        }),
                        el(TextControl, {
                            label: __('Format', 'bw-blocks'),
                            value: format,
                            onChange: (newValue) => setAttributes({ format: newValue })
                        }),
                        el(TextControl, {
                            label: __('Certification', 'bw-blocks'),
                            value: certification,
                            onChange: (newValue) => setAttributes({ certification: newValue })
                        }),
                        el(TextControl, {
                            label: __('Price', 'bw-blocks'),
                            value: price,
                            onChange: (newValue) => setAttributes({ price: newValue })
                        })
                    ),
                    el(PanelBody, { title: __('Enrollment Card', 'bw-blocks'), initialOpen: false },
                        el(ToggleControl, {
                            label: __('Show Enrollment Card', 'bw-blocks'),
                            help: __('Display the enrollment card sidebar', 'bw-blocks'),
                            checked: showEnrollmentCard,
                            onChange: (newValue) => setAttributes({ showEnrollmentCard: newValue })
                        }),
                        showEnrollmentCard && el(Fragment, {},
                            el(TextControl, {
                                label: __('Enrollment Button Text', 'bw-blocks'),
                                value: enrollmentText,
                                onChange: (newValue) => setAttributes({ enrollmentText: newValue })
                            }),
                            el(TextControl, {
                                label: __('Enrollment URL', 'bw-blocks'),
                                value: enrollmentUrl,
                                onChange: (newValue) => setAttributes({ enrollmentUrl: newValue })
                            }),
                            el(TextControl, {
                                label: __('Price Subtext', 'bw-blocks'),
                                value: priceSubtext,
                                onChange: (newValue) => setAttributes({ priceSubtext: newValue })
                            }),
                            el(TextControl, {
                                label: __('Secondary Button Text', 'bw-blocks'),
                                value: secondaryButtonText,
                                onChange: (newValue) => setAttributes({ secondaryButtonText: newValue })
                            }),
                            el(TextControl, {
                                label: __('Secondary Button URL', 'bw-blocks'),
                                value: secondaryButtonUrl,
                                onChange: (newValue) => setAttributes({ secondaryButtonUrl: newValue })
                            }),
                            el(TextControl, {
                                label: __('Guarantee Text', 'bw-blocks'),
                                value: guarantee,
                                onChange: (newValue) => setAttributes({ guarantee: newValue })
                            })
                        )
                    ),
                    el(PanelBody, { title: __('Cohort Information', 'bw-blocks'), initialOpen: false },
                        el(TextControl, {
                            label: __('Next Cohort Date', 'bw-blocks'),
                            value: nextCohortDate,
                            onChange: (newValue) => setAttributes({ nextCohortDate: newValue })
                        }),
                        el(TextControl, {
                            label: __('Session Schedule', 'bw-blocks'),
                            value: sessionSchedule,
                            onChange: (newValue) => setAttributes({ sessionSchedule: newValue })
                        })
                    )
                ),
                el('div', {
                    style: {
                        padding: '20px',
                        background: '#f8f9fa',
                        borderRadius: '8px',
                        border: '2px dashed #667eea',
                        color: '#333'
                    }
                },
                    el('div', {
                        style: {
                            marginBottom: '15px',
                            padding: '10px',
                            background: '#fff3cd',
                            borderRadius: '6px',
                            fontSize: '13px',
                            color: '#856404'
                        }
                    }, '💡 Add class "bw-course-header-bg" to Kadence Row for purple gradient background'),
                    el('div', {
                        style: {
                            display: 'flex',
                            gap: '40px',
                            alignItems: 'flex-start'
                        }
                    },
                        // Main content
                        el('div', {
                            style: {
                                flex: 1
                            }
                        },
                            el('h1', { style: { fontSize: '32px', marginBottom: '10px', color: '#333' }}, title),
                            el('p', { style: { fontSize: '18px', marginBottom: '20px', color: '#666' }}, subtitle),
                            el('div', { style: { display: 'flex', flexWrap: 'wrap', gap: '10px', marginBottom: '20px' }},
                                el('span', { style: { padding: '5px 10px', background: 'rgba(102,126,234,0.1)', borderRadius: '5px', fontSize: '14px', color: '#667eea' }}, '🕐 ' + duration),
                                el('span', { style: { padding: '5px 10px', background: 'rgba(102,126,234,0.1)', borderRadius: '5px', fontSize: '14px', color: '#667eea' }}, '🎓 ' + certification),
                                el('span', { style: { padding: '5px 10px', background: 'rgba(102,126,234,0.1)', borderRadius: '5px', fontSize: '14px', color: '#667eea' }}, '💻 ' + format),
                                el('span', { style: { padding: '5px 10px', background: 'rgba(102,126,234,0.1)', borderRadius: '5px', fontSize: '14px', color: '#667eea' }}, '📅 Tuesdays 2-3PM')
                            ),
                            el('div', { style: { display: 'inline-flex', alignItems: 'center', gap: '8px', padding: '6px 12px', background: 'rgba(255,204,0,0.2)', borderRadius: '15px', marginBottom: '20px' }},
                                el('span', {}, '⭐'),
                                el('span', { style: { fontSize: '14px', fontWeight: '600', color: '#856404' }}, 'Since 2015')
                            ),
                            nextCohortDate && el('div', { style: { marginTop: '20px', padding: '15px', background: 'rgba(102,126,234,0.1)', borderRadius: '8px' }},
                                el('strong', {}, 'Next cohort starts: '),
                                el('span', {}, nextCohortDate),
                                sessionSchedule && el('div', { style: { marginTop: '5px', fontSize: '14px' }}, sessionSchedule)
                            )
                        ),
                        // Enrollment Card
                        showEnrollmentCard && el('div', {
                            style: {
                                width: '320px',
                                background: 'white',
                                borderRadius: '12px',
                                padding: '25px',
                                color: '#333'
                            }
                        },
                            el('h3', { style: { textAlign: 'center', marginBottom: '15px', fontSize: '20px', fontWeight: 'bold' }}, 'Enroll Today'),
                            el('div', { style: { textAlign: 'center', fontSize: '36px', fontWeight: 'bold', color: '#00d4aa', marginBottom: '5px' }}, price),
                            priceSubtext && el('p', { style: { textAlign: 'center', fontSize: '12px', color: '#666', marginBottom: '20px' }}, priceSubtext),
                            el('div', { style: { padding: '10px', background: 'linear-gradient(135deg, rgba(0,212,170,0.1) 0%, rgba(102,126,234,0.1) 100%)', border: '2px solid #00d4aa', borderRadius: '8px', marginBottom: '20px' }},
                                el('p', { style: { margin: 0, fontSize: '12px', textAlign: 'center' }}, 'Lorem ipsum dolor sit amet + consectetur adipiscing')
                            ),
                            el('ul', { style: { listStyle: 'none', padding: 0, margin: '0 0 20px', fontSize: '14px' }},
                                enrollmentFeatures.slice(0, 7).map((feature, i) => 
                                    el('li', { key: i, style: { padding: '5px 0 5px 20px', position: 'relative' }},
                                        el('span', { style: { position: 'absolute', left: 0, color: '#00d4aa' }}, '✓'),
                                        feature
                                    )
                                )
                            ),
                            el('div', { style: { display: 'flex', flexDirection: 'column', gap: '10px' }},
                                el('button', { style: { padding: '10px', background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)', color: 'white', borderRadius: '6px', fontWeight: 'bold', border: 'none' }}, enrollmentText),
                                secondaryButtonText && el('button', { style: { padding: '10px', background: 'transparent', color: '#667eea', border: '2px solid #667eea', borderRadius: '6px', fontWeight: 'bold' }}, secondaryButtonText)
                            ),
                            guarantee && el('p', { style: { textAlign: 'center', fontSize: '11px', color: '#999', marginTop: '15px', margin: '15px 0 0' }}, guarantee)
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });
    
    // BW Course Curriculum Block
    registerBlockType('bw-blocks/course-curriculum', {
        title: __('BW Course Curriculum', 'bw-blocks'),
        description: __('Course curriculum with expandable modules', 'bw-blocks'),
        category: 'design',
        icon: 'index-card',
        keywords: ['course', 'curriculum', 'modules', 'bw'],
        attributes: {
            title: {
                type: 'string',
                default: 'Course Curriculum'
            },
            modules: {
                type: 'array',
                default: [
                    {
                        title: 'Module 1: Introduction to AI',
                        duration: '2 weeks',
                        lessons: [
                            'What is Artificial Intelligence?',
                            'History and Evolution of AI',
                            'Types of AI Systems',
                            'Real-world Applications'
                        ]
                    },
                    {
                        title: 'Module 2: Machine Learning Basics',
                        duration: '3 weeks',
                        lessons: [
                            'Introduction to Machine Learning',
                            'Supervised vs Unsupervised Learning',
                            'Common ML Algorithms',
                            'Hands-on Projects'
                        ]
                    }
                ]
            },
            showDuration: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { title, modules, showDuration } = attributes;
            
            const updateModule = (index, field, value) => {
                const newModules = [...modules];
                newModules[index][field] = value;
                setAttributes({ modules: newModules });
            };
            
            const updateLesson = (moduleIndex, lessonIndex, value) => {
                const newModules = [...modules];
                newModules[moduleIndex].lessons[lessonIndex] = value;
                setAttributes({ modules: newModules });
            };
            
            const addModule = () => {
                const newModules = [...modules, {
                    title: `Module ${modules.length + 1}: New Module`,
                    duration: '1 week',
                    lessons: ['Lesson 1', 'Lesson 2']
                }];
                setAttributes({ modules: newModules });
            };
            
            const removeModule = (index) => {
                const newModules = modules.filter((_, i) => i !== index);
                setAttributes({ modules: newModules });
            };
            
            const addLesson = (moduleIndex) => {
                const newModules = [...modules];
                newModules[moduleIndex].lessons.push(`Lesson ${newModules[moduleIndex].lessons.length + 1}`);
                setAttributes({ modules: newModules });
            };
            
            const removeLesson = (moduleIndex, lessonIndex) => {
                const newModules = [...modules];
                newModules[moduleIndex].lessons = newModules[moduleIndex].lessons.filter((_, i) => i !== lessonIndex);
                setAttributes({ modules: newModules });
            };
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Curriculum Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Section Title', 'bw-blocks'),
                            value: title,
                            onChange: (newValue) => setAttributes({ title: newValue })
                        }),
                        el(ToggleControl, {
                            label: __('Show Duration', 'bw-blocks'),
                            help: __('Display duration text for each module', 'bw-blocks'),
                            checked: showDuration,
                            onChange: (newValue) => setAttributes({ showDuration: newValue })
                        }),
                        el('p', { style: { marginTop: '20px', fontWeight: 'bold' }}, 
                            __('Manage modules in the editor view', 'bw-blocks')
                        )
                    )
                ),
                el('div', {
                    style: {
                        padding: '30px',
                        background: '#f8f9fa',
                        borderRadius: '12px'
                    }
                },
                    el(RichText, {
                        tagName: 'h2',
                        value: title,
                        onChange: (newValue) => setAttributes({ title: newValue }),
                        placeholder: __('Enter curriculum title...', 'bw-blocks'),
                        style: { marginBottom: '20px', fontSize: '24px', fontWeight: 'bold' }
                    }),
                    modules.map((module, moduleIndex) =>
                        el('div', {
                            key: moduleIndex,
                            style: {
                                background: 'white',
                                padding: '20px',
                                borderRadius: '8px',
                                marginBottom: '15px',
                                border: '2px solid #e0e0e0',
                                position: 'relative'
                            }
                        },
                            el('button', {
                                onClick: () => removeModule(moduleIndex),
                                style: {
                                    position: 'absolute',
                                    top: '10px',
                                    right: '10px',
                                    background: '#ff4444',
                                    color: 'white',
                                    border: 'none',
                                    borderRadius: '4px',
                                    padding: '5px 10px',
                                    cursor: 'pointer',
                                    fontSize: '12px'
                                }
                            }, '✕ Remove Module'),
                            
                            el('div', { style: { marginBottom: '15px' }},
                                el('div', { style: { display: 'flex', gap: '15px', marginBottom: '10px' }},
                                    el('div', { style: { flex: 1 }},
                                        el('label', { style: { display: 'block', marginBottom: '5px', fontSize: '12px', color: '#666' }}, 'Module Title'),
                                        el(TextControl, {
                                            value: module.title,
                                            onChange: (value) => updateModule(moduleIndex, 'title', value),
                                            placeholder: 'Module title...',
                                            style: { marginBottom: 0 }
                                        })
                                    ),
                                    showDuration && el('div', { style: { width: '150px' }},
                                        el('label', { style: { display: 'block', marginBottom: '5px', fontSize: '12px', color: '#666' }}, 'Duration'),
                                        el(TextControl, {
                                            value: module.duration,
                                            onChange: (value) => updateModule(moduleIndex, 'duration', value),
                                            placeholder: 'e.g., 2 weeks',
                                            style: { marginBottom: 0 }
                                        })
                                    )
                                )
                            ),
                            
                            el('div', { style: { marginLeft: '20px' }},
                                el('label', { style: { display: 'block', marginBottom: '10px', fontSize: '14px', fontWeight: 'bold', color: '#333' }}, 'Lessons:'),
                                module.lessons.map((lesson, lessonIndex) =>
                                    el('div', {
                                        key: lessonIndex,
                                        style: {
                                            display: 'flex',
                                            alignItems: 'center',
                                            gap: '10px',
                                            marginBottom: '8px'
                                        }
                                    },
                                        el('span', { style: { color: '#667eea' }}, '•'),
                                        el(TextControl, {
                                            value: lesson,
                                            onChange: (value) => updateLesson(moduleIndex, lessonIndex, value),
                                            placeholder: 'Lesson name...',
                                            style: { flex: 1, marginBottom: 0 }
                                        }),
                                        el('button', {
                                            onClick: () => removeLesson(moduleIndex, lessonIndex),
                                            style: {
                                                background: '#ffeeee',
                                                color: '#ff4444',
                                                border: '1px solid #ffcccc',
                                                borderRadius: '4px',
                                                padding: '5px 10px',
                                                cursor: 'pointer',
                                                fontSize: '12px'
                                            }
                                        }, '✕')
                                    )
                                ),
                                el('button', {
                                    onClick: () => addLesson(moduleIndex),
                                    style: {
                                        marginTop: '10px',
                                        background: '#f0f7ff',
                                        color: '#667eea',
                                        border: '1px solid #667eea',
                                        borderRadius: '6px',
                                        padding: '8px 16px',
                                        cursor: 'pointer',
                                        fontSize: '14px'
                                    }
                                }, '+ Add Lesson')
                            )
                        )
                    ),
                    el('button', {
                        onClick: addModule,
                        style: {
                            marginTop: '20px',
                            background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                            color: 'white',
                            border: 'none',
                            borderRadius: '8px',
                            padding: '12px 30px',
                            cursor: 'pointer',
                            fontSize: '16px',
                            fontWeight: 'bold',
                            width: '100%'
                        }
                    }, '+ Add New Module')
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });
    
    // BW Course Instructor Block
    registerBlockType('bw-blocks/course-instructor', {
        title: __('BW Course Instructor', 'bw-blocks'),
        description: __('Course instructor profile with bio and credentials', 'bw-blocks'),
        category: 'design',
        icon: 'admin-users',
        keywords: ['instructor', 'teacher', 'profile', 'bw'],
        attributes: {
            name: {
                type: 'string',
                default: 'Dr. Sarah Johnson'
            },
            title: {
                type: 'string',
                default: 'AI Research Lead'
            },
            bio: {
                type: 'string',
                default: 'With over 15 years of experience in artificial intelligence and machine learning, Dr. Johnson has led groundbreaking research at top tech companies and universities.'
            },
            imageUrl: {
                type: 'string',
                default: 'https://via.placeholder.com/300x300/667eea/ffffff?text=Instructor'
            },
            credentials: {
                type: 'array',
                default: [
                    'PhD in Computer Science, MIT',
                    'Former Google AI Research Lead',
                    'Published 50+ AI Papers',
                    '10+ Years Teaching Experience'
                ]
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { name, title, bio, imageUrl, credentials } = attributes;
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Instructor Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Name', 'bw-blocks'),
                            value: name,
                            onChange: (newValue) => setAttributes({ name: newValue })
                        }),
                        el(TextControl, {
                            label: __('Title', 'bw-blocks'),
                            value: title,
                            onChange: (newValue) => setAttributes({ title: newValue })
                        }),
                        el(TextareaControl, {
                            label: __('Bio', 'bw-blocks'),
                            value: bio,
                            onChange: (newValue) => setAttributes({ bio: newValue })
                        }),
                        el(TextControl, {
                            label: __('Image URL', 'bw-blocks'),
                            value: imageUrl,
                            onChange: (newValue) => setAttributes({ imageUrl: newValue })
                        })
                    )
                ),
                el('div', {
                    style: {
                        display: 'flex',
                        gap: '30px',
                        padding: '30px',
                        background: 'white',
                        borderRadius: '12px',
                        boxShadow: '0 2px 10px rgba(0,0,0,0.1)'
                    }
                },
                    el('img', {
                        src: imageUrl,
                        style: {
                            width: '150px',
                            height: '150px',
                            borderRadius: '50%',
                            objectFit: 'cover'
                        }
                    }),
                    el('div', { style: { flex: 1 }},
                        el('h3', { style: { margin: '0 0 5px' }}, name),
                        el('p', { style: { color: '#667eea', marginBottom: '15px' }}, title),
                        el('p', { style: { marginBottom: '15px' }}, bio),
                        el('ul', { style: { margin: 0, paddingLeft: '20px' }},
                            credentials.map((cred, i) => 
                                el('li', { key: i }, cred)
                            )
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });
    
    // BW Course FAQ Block
    registerBlockType('bw-blocks/course-faq', {
        title: __('BW Course FAQ', 'bw-blocks'),
        description: __('Frequently asked questions with expandable answers', 'bw-blocks'),
        category: 'design',
        icon: 'editor-help',
        keywords: ['faq', 'questions', 'accordion', 'bw'],
        attributes: {
            title: {
                type: 'string',
                default: 'Frequently Asked Questions'
            },
            faqs: {
                type: 'array',
                default: [
                    {
                        question: 'Who is this course for?',
                        answer: 'This course is designed for beginners and intermediate learners who want to understand AI fundamentals and apply them in real-world scenarios.'
                    },
                    {
                        question: 'What are the prerequisites?',
                        answer: 'Basic programming knowledge is helpful but not required. We cover all necessary foundations during the course.'
                    },
                    {
                        question: 'How long do I have access to the course?',
                        answer: 'You get lifetime access to all course materials, including future updates and additions.'
                    }
                ]
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { title, faqs } = attributes;
            
            const updateFAQ = (index, field, value) => {
                const newFAQs = [...faqs];
                newFAQs[index][field] = value;
                setAttributes({ faqs: newFAQs });
            };
            
            const addFAQ = () => {
                const newFAQs = [...faqs, {
                    question: 'New Question?',
                    answer: 'Enter your answer here...'
                }];
                setAttributes({ faqs: newFAQs });
            };
            
            const removeFAQ = (index) => {
                const newFAQs = faqs.filter((_, i) => i !== index);
                setAttributes({ faqs: newFAQs });
            };
            
            const moveFAQ = (index, direction) => {
                const newFAQs = [...faqs];
                const newIndex = direction === 'up' ? index - 1 : index + 1;
                
                if (newIndex < 0 || newIndex >= faqs.length) return;
                
                [newFAQs[index], newFAQs[newIndex]] = [newFAQs[newIndex], newFAQs[index]];
                setAttributes({ faqs: newFAQs });
            };
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('FAQ Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Section Title', 'bw-blocks'),
                            value: title,
                            onChange: (newValue) => setAttributes({ title: newValue })
                        }),
                        el('p', { style: { marginTop: '20px', color: '#666' }}, 
                            __(`Total FAQs: ${faqs.length}`, 'bw-blocks')
                        )
                    )
                ),
                el('div', {
                    style: {
                        padding: '30px',
                        background: '#f8f9fa',
                        borderRadius: '12px'
                    }
                },
                    el(RichText, {
                        tagName: 'h2',
                        value: title,
                        onChange: (newValue) => setAttributes({ title: newValue }),
                        placeholder: __('Enter FAQ title...', 'bw-blocks'),
                        style: { marginBottom: '25px', fontSize: '24px', fontWeight: 'bold' }
                    }),
                    faqs.map((faq, index) =>
                        el('div', {
                            key: index,
                            style: {
                                background: 'white',
                                padding: '20px',
                                borderRadius: '8px',
                                marginBottom: '15px',
                                border: '2px solid #e0e0e0',
                                position: 'relative'
                            }
                        },
                            el('div', {
                                style: {
                                    position: 'absolute',
                                    top: '10px',
                                    right: '10px',
                                    display: 'flex',
                                    gap: '5px'
                                }
                            },
                                index > 0 && el('button', {
                                    onClick: () => moveFAQ(index, 'up'),
                                    style: {
                                        background: '#f0f0f0',
                                        border: 'none',
                                        borderRadius: '4px',
                                        padding: '5px 8px',
                                        cursor: 'pointer',
                                        fontSize: '14px'
                                    },
                                    title: 'Move up'
                                }, '↑'),
                                index < faqs.length - 1 && el('button', {
                                    onClick: () => moveFAQ(index, 'down'),
                                    style: {
                                        background: '#f0f0f0',
                                        border: 'none',
                                        borderRadius: '4px',
                                        padding: '5px 8px',
                                        cursor: 'pointer',
                                        fontSize: '14px'
                                    },
                                    title: 'Move down'
                                }, '↓'),
                                el('button', {
                                    onClick: () => removeFAQ(index),
                                    style: {
                                        background: '#ff4444',
                                        color: 'white',
                                        border: 'none',
                                        borderRadius: '4px',
                                        padding: '5px 10px',
                                        cursor: 'pointer',
                                        fontSize: '12px'
                                    }
                                }, '✕ Remove')
                            ),
                            
                            el('div', { style: { marginBottom: '15px', paddingRight: '100px' }},
                                el('label', { style: { display: 'block', marginBottom: '5px', fontSize: '12px', color: '#666' }}, 
                                    `Question ${index + 1}:`
                                ),
                                el(TextControl, {
                                    value: faq.question,
                                    onChange: (value) => updateFAQ(index, 'question', value),
                                    placeholder: 'Enter question...',
                                    style: { marginBottom: 0 }
                                })
                            ),
                            
                            el('div', {},
                                el('label', { style: { display: 'block', marginBottom: '5px', fontSize: '12px', color: '#666' }}, 
                                    'Answer:'
                                ),
                                el(TextareaControl, {
                                    value: faq.answer,
                                    onChange: (value) => updateFAQ(index, 'answer', value),
                                    placeholder: 'Enter answer...',
                                    rows: 4,
                                    style: { marginBottom: 0 }
                                })
                            )
                        )
                    ),
                    el('button', {
                        onClick: addFAQ,
                        style: {
                            marginTop: '20px',
                            background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                            color: 'white',
                            border: 'none',
                            borderRadius: '8px',
                            padding: '12px 30px',
                            cursor: 'pointer',
                            fontSize: '16px',
                            fontWeight: 'bold',
                            width: '100%'
                        }
                    }, '+ Add New FAQ')
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Icon Card Block
    registerBlockType('bw-blocks/icon-card', {
        title: __('BW Icon Card', 'bw-blocks'),
        description: __('Icon card with SVG icon, title and description', 'bw-blocks'),
        category: 'design',
        icon: 'shield',
        keywords: ['icon', 'card', 'value', 'bw'],
        attributes: {
            iconType: {
                type: 'string',
                default: 'target'
            },
            title: {
                type: 'string',
                default: 'Lorem Ipsum Dolor'
            },
            description: {
                type: 'string',
                default: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.'
            },
            colorScheme: {
                type: 'string',
                default: 'primary'
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { iconType, title, description, colorScheme } = attributes;
            
            // Icon SVGs for preview
            const icons = {
                target: '🎯',
                lightning: '⚡',
                shield: '🛡️',
                rocket: '🚀',
                chart: '📊',
                users: '👥'
            };
            
            const colorMap = {
                primary: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                pink: 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
                teal: 'linear-gradient(135deg, #00d4aa 0%, #4facfe 100%)'
            };
            
            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Icon Card Settings', 'bw-blocks') },
                        el(SelectControl, {
                            label: __('Icon Type', 'bw-blocks'),
                            value: iconType,
                            options: [
                                { label: 'Target', value: 'target' },
                                { label: 'Lightning', value: 'lightning' },
                                { label: 'Shield', value: 'shield' },
                                { label: 'Rocket', value: 'rocket' },
                                { label: 'Chart', value: 'chart' },
                                { label: 'Users', value: 'users' }
                            ],
                            onChange: (newValue) => setAttributes({ iconType: newValue })
                        }),
                        el(TextControl, {
                            label: __('Title', 'bw-blocks'),
                            value: title,
                            onChange: (newValue) => setAttributes({ title: newValue })
                        }),
                        el(TextareaControl, {
                            label: __('Description', 'bw-blocks'),
                            value: description,
                            onChange: (newValue) => setAttributes({ description: newValue })
                        }),
                        el(SelectControl, {
                            label: __('Color Scheme', 'bw-blocks'),
                            value: colorScheme,
                            options: [
                                { label: 'Primary (Purple)', value: 'primary' },
                                { label: 'Pink', value: 'pink' },
                                { label: 'Teal', value: 'teal' }
                            ],
                            onChange: (newValue) => setAttributes({ colorScheme: newValue })
                        })
                    )
                ),
                el('div', {
                    style: {
                        padding: '40px',
                        background: 'white',
                        borderRadius: '24px',
                        textAlign: 'center',
                        boxShadow: '0 10px 30px rgba(0,0,0,0.08)'
                    }
                },
                    el('div', {
                        style: {
                            width: '80px',
                            height: '80px',
                            margin: '0 auto 30px',
                            background: colorMap[colorScheme] || colorMap.primary,
                            borderRadius: '20px',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            fontSize: '36px',
                            color: 'white'
                        }
                    }, icons[iconType] || icons.target),
                    el('h3', { 
                        style: { 
                            fontSize: '24px',
                            fontWeight: '700',
                            marginBottom: '15px'
                        }
                    }, title),
                    el('p', { 
                        style: { 
                            fontSize: '16px',
                            color: '#666',
                            lineHeight: '1.6',
                            maxWidth: '320px',
                            margin: '0 auto'
                        }
                    }, description)
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Value Grid Item Block
    registerBlockType('bw-blocks/value-grid-item', {
        title: __('BW Value Grid Item', 'bw-blocks'),
        description: __('Single value/stat item for grid layout', 'bw-blocks'),
        category: 'design',
        icon: 'chart-bar',
        keywords: ['value', 'stat', 'number', 'grid', 'bw'],
        attributes: {
            number: {
                type: 'string',
                default: '100+'
            },
            label: {
                type: 'string',
                default: 'Projects Completed'
            },
            icon: {
                type: 'string',
                default: '🚀'
            },
            colorScheme: {
                type: 'string',
                default: 'purple'
            },
            animateNumber: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { number, label, icon, colorScheme, animateNumber } = attributes;
            const { SelectControl } = wp.components;
            
            const colorOptions = [
                { label: 'Purple (Primary)', value: 'purple' },
                { label: 'Green (Success)', value: 'green' },
                { label: 'Blue (Info)', value: 'blue' },
                { label: 'Pink (Accent)', value: 'pink' }
            ];

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Value Grid Settings', 'bw-blocks') },
                        el(TextControl, {
                            label: __('Number/Value', 'bw-blocks'),
                            value: number,
                            onChange: (value) => setAttributes({ number: value }),
                            help: __('e.g., 100+, 50K, 99%', 'bw-blocks')
                        }),
                        el(TextControl, {
                            label: __('Label', 'bw-blocks'),
                            value: label,
                            onChange: (value) => setAttributes({ label: value })
                        }),
                        el(TextControl, {
                            label: __('Icon (Emoji)', 'bw-blocks'),
                            value: icon,
                            onChange: (value) => setAttributes({ icon: value }),
                            help: __('Enter an emoji or leave empty', 'bw-blocks')
                        }),
                        el(SelectControl, {
                            label: __('Color Scheme', 'bw-blocks'),
                            value: colorScheme,
                            options: colorOptions,
                            onChange: (value) => setAttributes({ colorScheme: value })
                        }),
                        el(ToggleControl, {
                            label: __('Animate Number on Scroll', 'bw-blocks'),
                            checked: animateNumber,
                            onChange: (value) => setAttributes({ animateNumber: value })
                        })
                    )
                ),
                el('div', { 
                    className: 'bw-value-grid-item-editor',
                    style: {
                        padding: '30px',
                        background: colorScheme === 'purple' ? 'linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%)' :
                                     colorScheme === 'green' ? 'linear-gradient(135deg, rgba(0, 212, 170, 0.1) 0%, rgba(0, 184, 148, 0.1) 100%)' :
                                     colorScheme === 'blue' ? 'linear-gradient(135deg, rgba(77, 171, 255, 0.1) 0%, rgba(43, 130, 214, 0.1) 100%)' :
                                     'linear-gradient(135deg, rgba(240, 147, 251, 0.1) 0%, rgba(245, 87, 108, 0.1) 100%)',
                        borderRadius: '16px',
                        border: '2px solid rgba(102, 126, 234, 0.2)',
                        textAlign: 'center'
                    }
                },
                    icon && el('div', { 
                        style: { 
                            fontSize: '48px',
                            marginBottom: '20px',
                            filter: 'grayscale(0%)'
                        } 
                    }, icon),
                    el('div', {
                        style: {
                            fontSize: '42px',
                            fontWeight: '800',
                            color: colorScheme === 'purple' ? '#667eea' :
                                   colorScheme === 'green' ? '#00d4aa' :
                                   colorScheme === 'blue' ? '#4dabff' :
                                   '#f093fb',
                            marginBottom: '10px',
                            fontFamily: 'Inter, sans-serif'
                        }
                    }, number),
                    el('div', {
                        style: {
                            fontSize: '16px',
                            color: '#666',
                            fontWeight: '500',
                            fontFamily: 'Inter, sans-serif'
                        }
                    }, label)
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

    // BW Comparison Table Block
    registerBlockType('bw-blocks/comparison-table', {
        title: __('BW Comparison Table', 'bw-blocks'),
        description: __('Responsive comparison table with mobile card view', 'bw-blocks'),
        category: 'design',
        icon: 'editor-table',
        keywords: ['table', 'comparison', 'pricing', 'bw'],
        attributes: {
            headers: {
                type: 'array',
                default: ['Path', 'Description', 'Investment', 'Next Step']
            },
            rows: {
                type: 'array',
                default: [
                    { path: 'Individual Training', description: 'Lorem ipsum dolor sit amet consectetur adipiscing', investment: '$197 - $4,999', buttonText: 'Learn More', buttonUrl: '#' },
                    { path: 'Corporate Team Training', description: 'Lorem ipsum dolor sit amet consectetur adipiscing', investment: 'Custom pricing', buttonText: 'Get Quote', buttonUrl: '#' },
                    { path: 'Consulting Engagement', description: 'Lorem ipsum dolor sit amet consectetur adipiscing', investment: 'Project-based', buttonText: 'Contact Us', buttonUrl: '#' },
                    { path: 'Legal Compliance Workshop', description: 'Lorem ipsum dolor sit amet consectetur adipiscing', investment: 'Contact for pricing', buttonText: 'Inquire Now', buttonUrl: '#' }
                ]
            },
            colorScheme: {
                type: 'string',
                default: 'primary'
            },
            mobileCards: {
                type: 'boolean',
                default: true
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { headers, rows, colorScheme, mobileCards } = attributes;
            const { SelectControl, Button, TextareaControl } = wp.components;
            
            const colorOptions = [
                { label: 'Primary (Blue/Purple)', value: 'primary' },
                { label: 'Accent (Teal/Green)', value: 'accent' },
                { label: 'Warm (Pink/Orange)', value: 'warm' },
                { label: 'Dark (Dark Blue)', value: 'dark' }
            ];
            
            const updateRow = (index, field, value) => {
                const newRows = [...rows];
                newRows[index][field] = value;
                setAttributes({ rows: newRows });
            };
            
            const addRow = () => {
                const newRows = [...rows, {
                    path: 'New Path',
                    description: 'Description here',
                    investment: 'Pricing',
                    buttonText: 'Button',
                    buttonUrl: '#'
                }];
                setAttributes({ rows: newRows });
            };
            
            const removeRow = (index) => {
                const newRows = rows.filter((_, i) => i !== index);
                setAttributes({ rows: newRows });
            };

            return el(Fragment, {},
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Table Settings', 'bw-blocks') },
                        el(SelectControl, {
                            label: __('Color Scheme', 'bw-blocks'),
                            value: colorScheme,
                            options: colorOptions,
                            onChange: (value) => setAttributes({ colorScheme: value })
                        }),
                        el(ToggleControl, {
                            label: __('Show Mobile Cards View', 'bw-blocks'),
                            checked: mobileCards,
                            onChange: (value) => setAttributes({ mobileCards: value }),
                            help: __('Display as cards on mobile devices', 'bw-blocks')
                        })
                    ),
                    el(PanelBody, { title: __('Table Rows', 'bw-blocks'), initialOpen: true },
                        rows.map((row, index) => 
                            el('div', {
                                key: index,
                                style: {
                                    padding: '15px',
                                    marginBottom: '15px',
                                    background: '#f5f5f5',
                                    borderRadius: '8px',
                                    border: '1px solid #e0e0e0'
                                }
                            },
                                el('div', {
                                    style: {
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'space-between',
                                        marginBottom: '10px'
                                    }
                                },
                                    el('h4', { style: { margin: 0, fontSize: '14px', fontWeight: '600' } }, 
                                        __('Row ', 'bw-blocks') + (index + 1)
                                    ),
                                    el(Button, {
                                        isDestructive: true,
                                        isSmall: true,
                                        onClick: () => removeRow(index),
                                        disabled: rows.length <= 1
                                    }, '×')
                                ),
                                el(TextControl, {
                                    label: headers[0],
                                    value: row.path,
                                    onChange: (value) => updateRow(index, 'path', value)
                                }),
                                el(TextareaControl, {
                                    label: headers[1],
                                    value: row.description,
                                    onChange: (value) => updateRow(index, 'description', value),
                                    rows: 2
                                }),
                                el(TextControl, {
                                    label: headers[2],
                                    value: row.investment,
                                    onChange: (value) => updateRow(index, 'investment', value)
                                }),
                                el(TextControl, {
                                    label: __('Button Text', 'bw-blocks'),
                                    value: row.buttonText,
                                    onChange: (value) => updateRow(index, 'buttonText', value)
                                }),
                                el(TextControl, {
                                    label: __('Button URL', 'bw-blocks'),
                                    value: row.buttonUrl,
                                    onChange: (value) => updateRow(index, 'buttonUrl', value)
                                })
                            )
                        ),
                        el(Button, {
                            isPrimary: true,
                            onClick: addRow,
                            style: { marginTop: '10px' }
                        }, __('Add Row', 'bw-blocks'))
                    )
                ),
                el('div', { 
                    className: 'bw-table-editor',
                    style: {
                        padding: '20px',
                        background: 'white',
                        borderRadius: '16px',
                        border: '2px solid #e0e0e0',
                        overflowX: 'auto'
                    }
                },
                    el('table', {
                        style: {
                            width: '100%',
                            borderCollapse: 'collapse'
                        }
                    },
                        el('thead', {},
                            el('tr', {},
                                headers.map(header =>
                                    el('th', {
                                        key: header,
                                        style: {
                                            background: colorScheme === 'primary' ? 'linear-gradient(135deg, #667eea, #764ba2)' :
                                                       colorScheme === 'accent' ? 'linear-gradient(135deg, #00d4aa, #00b894)' :
                                                       colorScheme === 'warm' ? 'linear-gradient(135deg, #f093fb, #f5576c)' :
                                                       'linear-gradient(135deg, #1a1a2e, #16213e)',
                                            color: 'white',
                                            padding: '15px',
                                            textAlign: 'left',
                                            fontSize: '14px',
                                            fontWeight: '600',
                                            textTransform: 'uppercase',
                                            letterSpacing: '0.5px'
                                        }
                                    }, header)
                                )
                            )
                        ),
                        el('tbody', {},
                            rows.map((row, index) =>
                                el('tr', {
                                    key: index,
                                    style: {
                                        borderBottom: '1px solid #e0e0e0'
                                    }
                                },
                                    el('td', {
                                        style: {
                                            padding: '15px',
                                            fontWeight: '600'
                                        }
                                    }, row.path),
                                    el('td', {
                                        style: {
                                            padding: '15px',
                                            color: '#666'
                                        }
                                    }, row.description),
                                    el('td', {
                                        style: {
                                            padding: '15px'
                                        }
                                    }, row.investment),
                                    el('td', {
                                        style: {
                                            padding: '15px'
                                        }
                                    },
                                        el('a', {
                                            href: '#',
                                            style: {
                                                color: '#667eea',
                                                textDecoration: 'none'
                                            },
                                            onClick: (e) => e.preventDefault()
                                        }, row.buttonText)
                                    )
                                )
                            )
                        )
                    )
                )
            );
        },
        save: function() {
            return null; // Server-side render
        }
    });

})(window.wp);